//////////////////////////////////////////////////////////////////////
// 
// File:        LSoundPlayer.cpp
// Author:      Brian Postma
// Created:     30 December 1999
//
// Purpose:     The hardware/OS dependent soundplayer layer. Provides
//              all means and methods to play sounds.
//
//////////////////////////////////////////////////////////////////////

#include "LSoundPlayer.h"
#include <stdio.h>
#include "Player.h"

void* SoundCallBack(snd_stream_hnd_t stream, int req, int *ret);

extern Player		*player;

snd_stream_hnd_t	stream;

LSoundPlayer::LSoundPlayer(Module *modptr)
{
  this->modptr=modptr;
  playbuffer=NULL;
  nrofvoices=0;
  stereo=false;
  lastpos=0;
}


LSoundPlayer::~LSoundPlayer()
{
  CleanUp();
}


void LSoundPlayer::CleanUp()
{
  snd_stream_stop(stream);
  snd_stream_destroy(stream);
  delete[] channels;
  delete[] playbuffer;
}


bool LSoundPlayer::Initialize(short nrofvoices, bool stereo)
{
  if (modptr==NULL)
    return false;

  this->nrofvoices=nrofvoices;

  if (stereo)
  {
    this->stereo=1;
  }
  else
  {
    this->stereo=0;
  }
  if ((channels=new LChannel[nrofvoices])==NULL)
    return false;
  if ((playbuffer=new short[2*SAMPLERATE/PLAYRATE])==NULL)
    return false;
  for (int i=0; i<(2*SAMPLERATE/PLAYRATE); i++)
    playbuffer[i]=0;

  snd_stream_init();
  stream = snd_stream_alloc (SoundCallBack, SND_STREAM_BUFFER_MAX);
  snd_stream_volume(stream,255);
  snd_stream_prefill(stream);
  snd_stream_start(stream,SAMPLERATE,1);
  return true;
}


bool LSoundPlayer::Play(short voicenr, BYTE instrument,
                        int frequency, int volume)
{
  WORD size, repeat, replen,dummy;
  BYTE *sampptr;

  if (instrument<1 || instrument>15 || voicenr>=nrofvoices || voicenr<0)
    return false;
  if (volume>64) volume=64;
  if (modptr->IsSynthInstrument(instrument))
  {
    modptr->GetSynthInstrument(instrument,size,sampptr);
    replen=size;
    repeat=0;
  }
  else
  {
    modptr->GetSampledInstrument(instrument,size,repeat,replen,dummy,sampptr);
  }
  channels[voicenr].SetPlayBuffer(sampptr,size,repeat,replen);
  channels[voicenr].SetVolume(volume);
  channels[voicenr].SetFrequency(frequency);
  channels[voicenr].StartPlay();
  return true;
}


bool LSoundPlayer::Stop(short voicenr)
{
  if (voicenr<0 || voicenr>=nrofvoices)
    return false;
  channels[voicenr].StopPlay();
  return true;
}


bool LSoundPlayer::SetPan(short voicenr, int val)
{
  if (voicenr<0 || voicenr>=nrofvoices)
    return false;
  if (val<0 || stereo==0) 
    channels[voicenr].SetLeftRight(true);
  else
    channels[voicenr].SetLeftRight(false);
  return true;
}


bool LSoundPlayer::SetFrequency(short voicenr, int frequency)
{
  if (voicenr<0 || voicenr>=nrofvoices)
    return false;
  channels[voicenr].SetFrequency(frequency);
  return true;
}


bool LSoundPlayer::SetVolume(short voicenr, int volume)
{
  if (voicenr<0 || voicenr>=nrofvoices)
    return false;
  if (volume>64) volume=64;
  channels[voicenr].SetVolume(volume);
  return true;
}


bool LSoundPlayer::Update(short voicenr, BYTE *memptr, 
                          int offset, int numofbytes)
{
  if (voicenr<0 || voicenr>=nrofvoices)
    return false;
  channels[voicenr].Update(memptr,offset,numofbytes);
  return true;
}


bool LSoundPlayer::Mix()
{
  short left,right;
  BYTE lefttmp, righttmp;
  short l,r;
  short *cursor=playbuffer;
  short j;

  for (j=0; j<4; j++)
    vol_data[j]=0;

  for (j=0; j<(SAMPLERATE/PLAYRATE); j++)
  {
    left=0;
    right=0;
    for (short i=0; i<nrofvoices; i++)
    {
      channels[i].GetNextBytes(lefttmp, righttmp);
      l=lefttmp-128;
      r=righttmp-128;
      left+=l;
      right+=r;
      vol_data[i]=vol_data[i]+abs(l)+abs(r);
    }
    if (stereo)
    {
      left=left*(255/nrofvoices);
      right=right*(255/nrofvoices);
      *(cursor++)=left;
      *(cursor++)=right; 
    }
    else
    {
      left=left*(255/nrofvoices);
      *(cursor++)=left;
      *(cursor++)=left;
    }
  } 
  return true;
}

short		*buf=NULL;
int		buf_index=0;

void* SoundCallBack(snd_stream_hnd_t stream, int req, int *ret)
{
	int i;
	short *ptr_src, *ptr_dest;

	if (buf==NULL)
	{
		if ((buf=new short[32768])==NULL)
		{
			printf("Unable to allocate audio buffer\n");
      			return 0;
    		}
  	}

	if (player==NULL)
	{
		if (req>65536)
			req=65536;
		*ret=req;
		memset(buf,req,0);
		return buf;
	}
	ptr_src=player->GetSoundPlayer()->GetBuffer();
	ptr_dest=buf;

	if (req>65536)
		req=65536;
	i=0;
	while (i<(req/2))
	{
		if (buf_index==0)
		{
			player->Run();
			player->Mix();
		}
		ptr_dest[i]=ptr_src[buf_index];
		buf_index++;
		i++;
		if (buf_index>= (2*SAMPLERATE/PLAYRATE) )
			buf_index=0;
	}
	*ret=req;
	return buf;
}
