/**
 * 4-IN-A-ROW
 *
 * Version 0.5beta
 *
 * This is a simple connect4-like game.  You can play with against a friend or
 * or against the computer.
 *
 * Written by Colin Luoma
 * http://www.cluoma.com/
 *
 * This software is released under the GNU GPLv3 licence.  A copy of this licence
 * can be found online.
 *
 */

#include <stdio.h>

#include <stdlib.h>

#include <string.h>

#include <unistd.h>
#include <SDL.h>
#include <SDL_image.h>
#include <SDL_ttf.h>

#ifdef DREAMCAST
	#include <kos.h>
	KOS_INIT_FLAGS(INIT_DEFAULT);
	
	extern uint8 romdisk[];
	KOS_INIT_ROMDISK(romdisk);
#endif


#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )

//Define functions here
void init_game();
int title_screen();
int single_player_game();
int two_player_game();
int add_piece(int piece, int pos);
int check_four_row(int board_array[7][6], int check_final);
int check_stalemate();
void cpu_play();
void switch_turn();
int minmax(int piece, int column, int depth, int array[7][6]);
int move_score(int piece, int column, int array[7][6]);


enum game_pieces {
    EMPTY = 0,
    RED_PIECE = 1,
    BLACK_PIECE = 2
};

enum comp_difficulty {
    EASY_COMPUTER = 1,
    NORMAL_COMPUTER = 2,
    HARD_COMPUTER = 4
};


enum CaanooButtons {
    CAANOO_BUTTON_A = 0,
    CAANOO_BUTTON_X = 1,
    CAANOO_BUTTON_B = 2,
    CAANOO_BUTTON_Y = 3,
    CAANOO_BUTTON_L = 4,
    CAANOO_BUTTON_R = 5,
    CAANOO_BUTTON_HOME = 6,
    CAANOO_BUTTON_1 = 8,
    CAANOO_BUTTON_2 = 9,
    CAANOO_BUTTON_JOY = 10
};



// The screen surfaces
SDL_Surface *screen = NULL;
SDL_Surface *title_bg = NULL;
SDL_Surface *board = NULL;

// These are for text
SDL_Surface *player_turn = NULL;
SDL_Surface *red_turn = NULL;
SDL_Surface *black_turn = NULL;

// Same with these
SDL_Surface *red_label = NULL;
SDL_Surface *black_label = NULL;
SDL_Surface *red_score = NULL;
SDL_Surface *black_score = NULL;

// These are for the chips and the selection arrow
SDL_Surface *select_arrow = NULL;
SDL_Surface *red = NULL;
SDL_Surface *black = NULL;
SDL_Joystick *joy = NULL;

SDL_Rect title_coords;
SDL_Rect board_coords;
SDL_Rect turn_coords;
SDL_Rect p1_label;
SDL_Rect p2_label;
SDL_Rect p1_score_coords;
SDL_Rect p2_score_coords;
SDL_Rect select_coords;

TTF_Font *font = NULL;
TTF_Font *font_small = NULL;
SDL_Color text_color = { 255, 255, 255 };  // White

// Magic numbers for x and y positions of chips
int position[7] = {10, 44, 78, 112, 146, 180, 214};
int y_position[6] = {7, 40, 73, 106, 139, 172};
int board_array[7][6];

char turn_messege[] = "Turn:";
char p1_turn_string[] = "RED";
char p2_turn_string[] = "BLK";
char p1_score_string[2];
char p2_score_string[2];

int p1_score;
int p2_score;

int turn = RED_PIECE;

int winner;
int difficulty;



void Terminate(void)
{
	
	SDL_Quit();

}


int main (int argc, char *argv[])
{

	#ifdef DREAMCAST
		fs_chdir("/rd/");
		printf("Starting...\n");
	#endif
	
	// Initialize SDL
	
	if (SDL_Init (SDL_INIT_VIDEO | SDL_INIT_JOYSTICK) < 0) {
		
		fprintf (stderr, "Couldn't initialize SDL: %s\n", SDL_GetError ());
		
		exit (1);
	
	}

	atexit (Terminate);

	
	
	// Set 320x240 16-bits video mode
	
	screen = SDL_SetVideoMode (320, 240, 16, SDL_SWSURFACE);
	
	
	if (screen == NULL) {
		
		fprintf (stderr, "Couldn't set 320x240x16 video mode: %s\n", SDL_GetError ());
		
		exit (2);
	
	}

	
	
	// Check and open joystick device
	
	if (SDL_NumJoysticks() > 0) {
		
		joy = SDL_JoystickOpen(0);
		
		if(!joy) {
			
			fprintf (stderr, "Couldn't open joystick 0: %s\n", SDL_GetError ());
		
		}
		#ifdef DREAMCAST
			//Activate mouse movements!
	   		SDL_JoystickEventState(SDL_ENABLE);
		#endif

	}

	// Load font
	if (TTF_Init() == -1){
	    fprintf(stderr, "Couldn't load ttf font for: %s\n", TTF_GetError());
	    exit(2);
	}
	font = TTF_OpenFont("data/FreeSans.ttf", 20);
    if (font == NULL){
        printf("Unable to load font: 'data/FreeSans.ttf' %s \n", TTF_GetError());
        exit(2);
    }
    font_small = TTF_OpenFont("data/FreeSans.ttf", 12);
    if (font_small == NULL){
        printf("Unable to load font: 'data/FreeSans.ttf' %s \n", TTF_GetError());
        exit(2);
    }

    	
	#ifdef WIN32
	
		difficulty = HARD_COMPUTER;

	#elif DREAMCAST
		difficulty = HARD_COMPUTER;
//1, 2 or 4!
	#endif

    
	
	// image loading...
    title_bg = IMG_Load("data/title_bg.png");
    if (title_bg == NULL)
    {
        fprintf(stderr, "Couldn't load image file 'title_bg.png'");
        Terminate();
        exit(2);
    }

    board = IMG_Load("data/board.png");
    if (board == NULL)
    {
        fprintf(stderr, "Couldn't load image file 'board.png'");
        Terminate();
        exit(2);
    }
    board_coords.x = 5;
    board_coords.y = 25;

    red = IMG_Load("data/red.png");
    if (red == NULL)
    {
        fprintf(stderr, "Couldn't load image file 'red.png'");
        Terminate();
        exit(2);
    }

    black = IMG_Load("data/black.png");
    if (black == NULL)
    {
        fprintf(stderr, "Couldn't load image file 'black.png'");
        Terminate();
        exit(2);
    }

    select_arrow = IMG_Load("data/select.png");
    if (select_arrow == NULL)
    {
        fprintf(stderr, "Couldn't load image file 'select.png'");
        Terminate();
        exit(2);
    }

    // We leave the score outside the init_game() function
    // because we do not want to restrict init_game to only
    // the first instance of the game
    p1_score = 0;
    p2_score = 0;

	while ( title_screen() ){

	}

	SDL_FreeSurface(player_turn);
	
	SDL_FreeSurface(board);
	
	SDL_FreeSurface(screen);
	TTF_Quit();
	
	
	SDL_Quit();

	
	return 0;

}

// Main loop for the title screen
int title_screen() {

    title_coords.x = 0;
    title_coords.y = 0;
    SDL_BlitSurface(title_bg, NULL, screen, &title_coords);
    SDL_Flip(screen);

    SDL_Event event;

    while (SDL_PollEvent (&event))
	{
			
    	switch (event.type)
 {
				
    		case SDL_KEYDOWN:				
				break;
				
			case SDL_KEYUP:
					
				break;
				
			case SDL_JOYBUTTONDOWN:
					
				// if press Start button, terminate program

				//DREAMCAST:Is the X button (6) :)
				if ( event.jbutton.button == CAANOO_BUTTON_HOME ){
					 return 0;
				}

				break;

			case SDL_JOYBUTTONUP:

				break;
			
			case SDL_MOUSEMOTION:

				break;
     		
            case SDL_MOUSEBUTTONDOWN:
                    if ( event.button.button == SDL_BUTTON_LEFT ){
                        int x = event.button.x;
                        int y = event.button.y; // x > 53 y > 116

                        // Start single player game
                        if ( x >= 53 && y >= 116 && x <= 246 && y <= 147 ){
                            init_game();
                            while ( single_player_game() ){
                                init_game();
                            }
                        }
                        else if ( x >= 60 && y >= 161 && x <= 252 && y <= 196 ){
                            init_game();
                            while ( two_player_game() ){
                                init_game();
                            }
                        }
                    }
                    p1_score = 0;
                    p2_score = 0;
                    break;
				
            case SDL_QUIT:
                    return 0;
					
                    break;
				
            default:

            	break;
			
        }
		
    }

    return 1;
}

// Main loop for a two player game
int two_player_game() {

    int done = 0;
	
    while (!done)
	{
		
    	SDL_Event event;

		
    	/* Check for events */
		
    	
    	while (SDL_PollEvent (&event))
	{
			
    		switch (event.type)
{
				
    			case SDL_KEYDOWN:				
					break;
				
				case SDL_KEYUP:
					
					break;
				
				case SDL_JOYBUTTONDOWN:

					// if press Start button, terminate program

					//Dreamcast: This is the X button :)
					if ( event.jbutton.button == CAANOO_BUTTON_HOME ){
					    return 0;

					    done = 1;
					}
					
					break;
				
				case SDL_JOYBUTTONUP:
					
					break;
				case SDL_MOUSEMOTION:
					SDL_Flip(screen); //Clean screen! (not the best, but works for DC)
					break;
                // Mousebutton events (touchscreen on Caanoo)
                case SDL_MOUSEBUTTONDOWN:
                    if ( event.button.button == SDL_BUTTON_LEFT ){
                        int x = event.button.x;
                        //int y = event.button.y;

                        if ( x >= board_coords.x && x <= board_coords.x + 254){
                            if ( x < board_coords.x + position[1] - 3 ){
                                if ( select_coords.x == board_coords.x + position[0] ){
                                    add_piece(turn, 0);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[0];
                                }
                            }
                            else if ( x < board_coords.x + position[2] - 3 ){
                                if ( select_coords.x == board_coords.x + position[1] ){
                                    add_piece(turn, 1);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[1];
                                }
                            }
                            else if ( x < board_coords.x + position[3] - 3 ){
                                if ( select_coords.x == board_coords.x + position[2] ){
                                    add_piece(turn, 2);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[2];
                                }
                            }
                            else if ( x < board_coords.x + position[4] - 3 ){
                                if ( select_coords.x == board_coords.x + position[3] ){
                                    add_piece(turn, 3);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[3];
                                }
                            }
                            else if ( x < board_coords.x + position[5] - 3 ){
                                if ( select_coords.x == board_coords.x + position[4] ){
                                    add_piece(turn, 4);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[4];
                                }
                            }
                            else if ( x < board_coords.x + position[6] - 3 ){
                                if ( select_coords.x == board_coords.x + position[5] ){
                                    add_piece(turn, 5);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[5];
                                }
                            }
                            else{
                                if ( select_coords.x == board_coords.x + position[6] ){
                                    add_piece(turn, 6);
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[6];
                                }
                            }
                            if ( check_four_row(board_array, 1) || check_stalemate() ){
                                SDL_Delay(2000);
                                return 1;
                            }
                        }
                        SDL_BlitSurface(select_arrow, NULL, screen, &select_coords);
                        SDL_Flip(screen);
                    }
                    break;
				
                case SDL_QUIT:
               	 	return 0;

               		done = 1;

               		break;

             	default:

              		break;
			
             }
	
        }
	
                
    }
	return 1;
}

int single_player_game() {

    int done = 0;
	
    while (!done)
{
    	SDL_Event event;

		
    	// Check for events
		
    	while (SDL_PollEvent (&event))
{
			
    		switch (event.type)
{
				
    			case SDL_KEYDOWN:				
					break;
				
				case SDL_KEYUP:
					
					break;
				
				case SDL_JOYBUTTONDOWN:
					
					// if press Start button, terminate program
					//Dreamcast: This is the X button :)					
					if ( event.jbutton.button == CAANOO_BUTTON_HOME ){
					    return 0;

					    done = 1;
					}
					
					break;
				
				case SDL_JOYBUTTONUP:
					
					break;
				case SDL_MOUSEMOTION:

				    SDL_Flip(screen); //Clean screen! (not the best, but works for DC)
					break;
                // Mousebutton events (touchscreen on Caanoo)
                case SDL_MOUSEBUTTONDOWN:
                    if ( event.button.button == SDL_BUTTON_LEFT ){
                        int x = event.button.x;
                        //int y = event.button.y;

                        if ( x >= board_coords.x && x <= board_coords.x + 254){
                            if ( x < board_coords.x + position[1] - 3 ){
                                if ( select_coords.x == board_coords.x + position[0] ){
                                    add_piece(turn, 0);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[0];
                                }
                            }
                            else if ( x < board_coords.x + position[2] - 3 ){
                                if ( select_coords.x == board_coords.x + position[1] ){
                                    add_piece(turn, 1);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[1];
                                }
                            }
                            else if ( x < board_coords.x + position[3] - 3 ){
                                if ( select_coords.x == board_coords.x + position[2] ){
                                    add_piece(turn, 2);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[2];
                                }
                            }
                            else if ( x < board_coords.x + position[4] - 3 ){
                                if ( select_coords.x == board_coords.x + position[3] ){
                                    add_piece(turn, 3);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[3];
                                }
                            }
                            else if ( x < board_coords.x + position[5] - 3 ){
                                if ( select_coords.x == board_coords.x + position[4] ){
                                    add_piece(turn, 4);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[4];
                                }
                            }
                            else if ( x < board_coords.x + position[6] - 3 ){
                                if ( select_coords.x == board_coords.x + position[5] ){
                                    add_piece(turn, 5);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[5];
                                }
                            }
                            else{
                                if ( select_coords.x == board_coords.x + position[6] ){
                                    add_piece(turn, 6);
                                    cpu_play();
                                }
                                else{
                                    SDL_FillRect(screen, &select_coords, 250);
                                    select_coords.x = board_coords.x + position[6];
                                }
                            }
                            if ( check_four_row(board_array, 1) || check_stalemate() ){
                                SDL_Delay(2000);
                                return 1;
                            }
                        }
                        SDL_BlitSurface(select_arrow, NULL, screen, &select_coords);
                        SDL_Flip(screen);
                    }
                    break;

                    case SDL_QUIT:
                   		return 0;
                   		done = 1;
                   		break;

                   	default:

                   		break;
                }
            }
		
        }
	return 1;
}

// Initialize game variables
void init_game() {

    int i = 0;
    int j = 0;
    while ( i < 7 ){
        while ( j < 6 ){
            board_array[i][j] = 0;
            j++;
        }
        i++;
        j = 0;
    }

    // Red usually starts first
    turn = RED_PIECE;

    select_coords.x = board_coords.x + 11;
    select_coords.y = board_coords.y - 17;
    select_coords.h = 14;
    select_coords.w = 27;
    turn_coords.x = 267;
    turn_coords.y = 28;
    turn_coords.w = 53;
    turn_coords.h = 10;

    p1_label.x = 267;
    p1_label.y = 150;
    p2_label.x = 267;
    p2_label.y = 180;
    p1_score_coords.x = 267;
    p1_score_coords.y = 160;
    p2_score_coords.x = 267;
    p2_score_coords.y = 190;

    // messege
    snprintf(p1_score_string, 2, "%d", p1_score);
    snprintf(p2_score_string, 2, "%d", p2_score);
    player_turn = TTF_RenderText_Blended( font, turn_messege, text_color );
    red_turn = TTF_RenderText_Blended( font, p1_turn_string, text_color );
    black_turn = TTF_RenderText_Blended( font, p2_turn_string, text_color);
    red_label = TTF_RenderText_Blended( font_small, p1_turn_string, text_color );
    black_label = TTF_RenderText_Blended (font_small, p2_turn_string, text_color );
    red_score = TTF_RenderText_Blended (font_small, p1_score_string, text_color );
    black_score = TTF_RenderText_Blended (font_small, p2_score_string, text_color );

    // Initial draw
    SDL_FillRect(screen, NULL, 250);
    SDL_BlitSurface(board, NULL, screen, &board_coords);
    SDL_BlitSurface(select_arrow, NULL, screen, &select_coords);
    SDL_BlitSurface(player_turn, NULL, screen, &turn_coords);

    turn_coords.y += 25;
    SDL_BlitSurface(red_turn, NULL, screen, &turn_coords);

    SDL_BlitSurface(red_label, NULL, screen, &p1_label);
    SDL_BlitSurface(black_label, NULL, screen, &p2_label);
    SDL_BlitSurface(red_score, NULL, screen, &p1_score_coords);
    SDL_BlitSurface(black_score, NULL, screen, &p2_score_coords);

    SDL_Flip(screen);
}

// Check to see if we can place a piece at a given column
// return 0:  cannot be placed
// If a piece can be placed, returns the row (NOT the index) it will be placed in
int check_position(int pos, int array[7][6]) {
    int i = 5;
    while ( i >= 0 ){
        if ( array[pos][i] == 0 ){
            return i+1;
        }
        i--;
    }
    return 0;
}

// Check for a stalemate
int check_stalemate(){
    int stalemate = 1;

    int i = 0;
    int j = 0;
    while ( i < 6 ){
        while ( j < 5 ){
            if ( board_array[i][j] == EMPTY ){
                stalemate = 0;
                break;
            }
            j++;
        }
        j = 0;
        i++;
    }
    return stalemate;
}

// Adds a given game piece to a given column
// If the addition is successful, change turns.
int add_piece(int piece, int pos) {
    int row = check_position(pos, board_array);
    SDL_Rect piece_coords;

    if ( row ){
        piece_coords.x = board_coords.x + position[pos];
        piece_coords.y = board_coords.y + y_position[row-1];

        if ( piece == RED_PIECE ){
            SDL_BlitSurface(red, NULL, screen, &piece_coords);
            switch_turn();
            board_array[pos][row-1] = RED_PIECE;
        }
        else if ( piece == BLACK_PIECE ){
            SDL_BlitSurface(black, NULL, screen, &piece_coords);
            switch_turn();
            board_array[pos][row-1] = BLACK_PIECE;
        }
        SDL_Flip(screen);
        return 1;
    }
    else{
        return 0;
    }
}

// This is similar to 'int add_piece()' but this is just for
// the AI to use when making decisions
int add_piece_nhp(int piece, int pos, int game_board[7][6]) {
    int row = check_position(pos, game_board);

    if ( row ){
        if ( piece == RED_PIECE ){
            game_board[pos][row-1] = RED_PIECE;
        }
        else if ( piece == BLACK_PIECE ){
            game_board[pos][row-1] = BLACK_PIECE;
        }
        return 1;
    }
    return 0;
}

// Checks to see if there is a 4-in-a-row in a game board
// If no 4-in-a-row:  return 0
// If yes 4-in-a-row:  return the winner
int check_four_row(int board_array[7][6], int check_final){
    int i = 0;
    int j = 0;

    // Check horizontal rows of 4
    while ( i < 4 ){
        while ( j < 6 ){
            if ( board_array[i][j] == RED_PIECE && board_array[i+1][j] == RED_PIECE && board_array[i+2][j] == RED_PIECE && board_array[i+3][j] == RED_PIECE ){
                if (check_final){p1_score++;}
                return RED_PIECE;
            }
            if ( board_array[i][j] == BLACK_PIECE && board_array[i+1][j] == BLACK_PIECE && board_array[i+2][j] == BLACK_PIECE && board_array[i+3][j] == BLACK_PIECE ){
                if (check_final){p2_score++;}
                return BLACK_PIECE;
            }
            j++;
        }
        i++;
        j = 0;
    }

    // Check vertical rows of 4
    i = 0;
    j = 0;
    while ( i < 7 ){
        while ( j < 3 ){
            if ( board_array[i][j] == RED_PIECE && board_array[i][j+1] == RED_PIECE && board_array[i][j+2] == RED_PIECE && board_array[i][j+3] == RED_PIECE ){
                if (check_final){p1_score++;}
                return RED_PIECE;
            }
            if ( board_array[i][j] == BLACK_PIECE && board_array[i][j+1] == BLACK_PIECE && board_array[i][j+2] == BLACK_PIECE && board_array[i][j+3] == BLACK_PIECE ){
                if (check_final){p2_score++;}
                return BLACK_PIECE;
            }
            j++;
        }
        i++;
        j = 0;
    }

    // Check Diagonal upward
    i = 0;
    j = 3;
    while ( i < 4 ){
        while ( j < 6 ){
            if ( board_array[i][j] == RED_PIECE && board_array[i+1][j-1] == RED_PIECE && board_array[i+2][j-2] == RED_PIECE && board_array[i+3][j-3] == RED_PIECE){
                if (check_final){p1_score++;}
                return RED_PIECE;
            }
            if ( board_array[i][j] == BLACK_PIECE && board_array[i+1][j-1] == BLACK_PIECE && board_array[i+2][j-2] == BLACK_PIECE && board_array[i+3][j-3] == BLACK_PIECE){
                if (check_final){p2_score++;}
                return BLACK_PIECE;
            }
            j++;
        }
        i++;
        j = 3;
    }

    // Check Diagonal downward
    i = 0;
    j = 0;
    while ( i < 4 ){
        while ( j < 3 ){
            if ( board_array[i][j] == RED_PIECE && board_array[i+1][j+1] == RED_PIECE && board_array[i+2][j+2] == RED_PIECE && board_array[i+3][j+3] == RED_PIECE){
                if (check_final){p1_score++;}
                return RED_PIECE;
            }
            if ( board_array[i][j] == BLACK_PIECE && board_array[i+1][j+1] == BLACK_PIECE && board_array[i+2][j+2] == BLACK_PIECE && board_array[i+3][j+3] == BLACK_PIECE){
                if (check_final){p2_score++;}
                return BLACK_PIECE;
            }
            j++;
        }
        i++;
        j = 0;
    }
    return 0;
}

// Changes player turn from red to black or vice versa
void switch_turn() {
    if ( turn == RED_PIECE ){
        SDL_FillRect(screen, &turn_coords, 250);
        SDL_BlitSurface(black_turn, NULL, screen, &turn_coords);
        turn = BLACK_PIECE;
    }
    else if ( turn == BLACK_PIECE ){
        SDL_FillRect(screen, &turn_coords, 250);
        SDL_BlitSurface(red_turn, NULL, screen, &turn_coords);
        turn = RED_PIECE;
    }
}

// Makes a play for a the non-human-player
void cpu_play(){

    if ( check_four_row(board_array, 0) || check_stalemate() ){
        return;
    }

    int i = 0;
    int possible_moves[7][2];
    int old_maximum = -1001;
    int new_maximum;
    int our_move = 0;

    // Gets the heuristic values of different moves, and if they are possible
    if ( turn == BLACK_PIECE ){
        while ( i < 7 ){
            possible_moves[i][0] = minmax(BLACK_PIECE, i, 3, board_array);
            possible_moves[i][1] = check_position(i, board_array);
            i++;
        }
    }

    i = 0;
    while ( i < 7 ){
        new_maximum = max ( old_maximum, possible_moves[i][0] );
        if ( new_maximum != old_maximum && possible_moves[i][1] != 0 ){
            our_move = i;
            old_maximum = new_maximum;
        }
        i++;
    }
    add_piece(turn, our_move);
}

// This is the heart of the AI.  It's an implementation of the
// minmax algorithm.  It maximises our moves while minimizing
// our opponents
int minmax(int piece, int column, int depth, int array[7][6]){

    // Clone array
    int new_array[7][6];
    int j = 0;
    int k = 0;
    while ( j < 7 ){
        while ( k < 6 ){
            new_array[j][k] = array[j][k];
            k++;
        }
        k = 0;
        j++;
    }

    if ( add_piece_nhp(piece, column, new_array) ){
        if ( check_four_row(new_array, 0) ){
            return 1000;
        }
        if ( depth == 0 ){
            return move_score(piece, column, new_array);
        }

        int alpha = 1000;
        int temp;

        int i = 0;
        while ( i < 7 ){
            if ( piece == BLACK_PIECE ){
                temp = (-1) * minmax(RED_PIECE, i, depth-1, new_array);
            }
            else{
                temp = (-1) * minmax(BLACK_PIECE, i, depth-1, new_array);
            }
            alpha = min (alpha, temp);
            i++;
        }
        return alpha;
    }
    return 0;
}

// This determines the heuristic 'score' of a particular move
// 1 point for a group of 4 with 1 piece
// 4 points for a group of 4 with 2 pieces
// 32 points for a group of 4 with 3 pieces
int move_score(int piece, int column, int array[7][6]){

    int score = 0;

    int i = 0;
    int j = 0;
    int row = check_position(column, array);
    if (row != 0){ row -= 1; }

    // Count score for a red move
    if ( piece == RED_PIECE ){
    // Count horizontal score
    while ( i < 4 ){
        if ( (array[i][row] == RED_PIECE && array[i+1][row] == EMPTY     && array[i+2][row] == EMPTY     && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == RED_PIECE && array[i+2][row] == EMPTY     && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == EMPTY     && array[i+2][row] == RED_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == EMPTY     && array[i+2][row] == EMPTY     && array[i+3][row] == RED_PIECE) ){
            score += 1;
        }
        if ( (array[i][row] == RED_PIECE && array[i+1][row] == RED_PIECE && array[i+2][row] == EMPTY     && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == RED_PIECE && array[i+2][row] == RED_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == EMPTY     && array[i+2][row] == RED_PIECE && array[i+3][row] == RED_PIECE) ){
            score += 4;
        }
        if ( (array[i][row] == RED_PIECE && array[i+1][row] == RED_PIECE && array[i+2][row] == RED_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY     && array[i+1][row] == RED_PIECE && array[i+2][row] == RED_PIECE && array[i+3][row] == RED_PIECE) ){
            score += 32;
        }
        i++;
    }

    // Count vertical score
    j = 0;
    while ( j < 3 ){
        if ( (array[column][j] == RED_PIECE && array[column][j+1] == EMPTY     && board_array[column][j+2] == EMPTY     && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == RED_PIECE && board_array[column][j+2] == EMPTY     && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == EMPTY     && board_array[column][j+2] == RED_PIECE && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == EMPTY     && board_array[column][j+2] == EMPTY     && board_array[column][j+3] == RED_PIECE) ){
            score += 1;
        }
        if ( (array[column][j] == RED_PIECE && array[column][j+1] == RED_PIECE && array[column][j+2] == EMPTY     && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == RED_PIECE && array[column][j+2] == RED_PIECE && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == EMPTY     && array[column][j+2] == RED_PIECE && array[column][j+3] == RED_PIECE) ){
            score += 4;
        }
        if ( (array[column][j] == RED_PIECE && array[column][j+1] == RED_PIECE && array[column][j+2] == RED_PIECE && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY     && array[column][j+1] == RED_PIECE && array[column][j+2] == RED_PIECE && array[column][j+3] == RED_PIECE) ){
            score += 32;
        }
        j++;
    }

    // Count diagonal downward score
    i = 0;
    j = 0;
    while ( i < 4 ){
        while ( j < 3 ){
            if ( (i == column || i+1 == column || i+2 == column || i+3 == column) &&
                 (j == row || j+1 == row || j+2 == row || j+3 == row) ) {
                if ( (array[i][j] == RED_PIECE && array[i+1][j+1] == EMPTY     && array[i+2][j+2] == EMPTY     && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == RED_PIECE && array[i+2][j+2] == EMPTY     && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == EMPTY     && array[i+2][j+2] == RED_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == EMPTY     && array[i+2][j+2] == EMPTY     && array[i+3][j+3] == RED_PIECE) ){
                    score += 1;
                }
                if ( (array[i][j] == RED_PIECE && array[i+1][j+1] == RED_PIECE && array[i+2][j+2] == EMPTY     && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == RED_PIECE && array[i+2][j+2] == RED_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == EMPTY     && array[i+2][j+2] == RED_PIECE && array[i+3][j+3] == RED_PIECE) ){
                    score += 4;
                }
                if ( (array[i][j] == RED_PIECE && array[i+1][j+1] == RED_PIECE && array[i+2][j+2] == RED_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j+1] == RED_PIECE && array[i+2][j+2] == RED_PIECE && array[i+3][j+3] == RED_PIECE) ){
                    score += 32;
                }
            }
            j++;
        }
        i++;
        j = 0;
    }

    // Count diagonal upward score
    i = 0;
    j = 3;
    while ( i < 4 ){
        while ( j < 6 ){
            if ( (i == column || i+1 == column || i+2 == column || i+3 == column) &&
                 (j == row || j-1 == row || j-2 == row || j-3 == row) ) {
                if ( (array[i][j] == RED_PIECE && array[i+1][j-1] == EMPTY     && array[i+2][j-2] == EMPTY     && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == RED_PIECE && array[i+2][j-2] == EMPTY     && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == EMPTY     && array[i+2][j-2] == RED_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == EMPTY     && array[i+2][j-2] == EMPTY     && array[i+3][j-3] == RED_PIECE) ){
                    score += 1;
                }
                if ( (array[i][j] == RED_PIECE && array[i+1][j-1] == RED_PIECE && array[i+2][j-2] == EMPTY     && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == RED_PIECE && array[i+2][j-2] == RED_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == EMPTY     && array[i+2][j-2] == RED_PIECE && array[i+3][j-3] == RED_PIECE) ){
                    score += 4;
                }
                if ( (array[i][j] == RED_PIECE && array[i+1][j-1] == RED_PIECE && array[i+2][j-2] == RED_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY     && array[i+1][j-1] == RED_PIECE && array[i+2][j-2] == RED_PIECE && array[i+3][j-3] == RED_PIECE) ){
                    score += 32;
                }
            }
            j++;
        }
        i++;
        j = 3;
    }
    }

    // Count score for a black move
    if ( piece == BLACK_PIECE ){
    // Count horizontal score
    while ( i < 4 ){
        if ( (array[i][row] == BLACK_PIECE && array[i+1][row] == EMPTY       && array[i+2][row] == EMPTY       && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == BLACK_PIECE && array[i+2][row] == EMPTY       && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == EMPTY       && array[i+2][row] == BLACK_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == EMPTY       && array[i+2][row] == EMPTY       && array[i+3][row] == BLACK_PIECE) ){
            score += 1;
        }
        if ( (array[i][row] == BLACK_PIECE && array[i+1][row] == BLACK_PIECE && array[i+2][row] == EMPTY       && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == BLACK_PIECE && array[i+2][row] == BLACK_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == EMPTY       && array[i+2][row] == BLACK_PIECE && array[i+3][row] == BLACK_PIECE) ){
            score += 4;
        }
        if ( (array[i][row] == BLACK_PIECE && array[i+1][row] == BLACK_PIECE && array[i+2][row] == BLACK_PIECE && array[i+3][row] == EMPTY) ||
             (array[i][row] == EMPTY       && array[i+1][row] == BLACK_PIECE && array[i+2][row] == BLACK_PIECE && array[i+3][row] == BLACK_PIECE) ){
            score += 32;
        }
        i++;
    }

    // Count vertical score
    j = 0;
    while ( j < 3 ){
        if ( (array[column][j] == BLACK_PIECE && array[column][j+1] == EMPTY       && board_array[column][j+2] == EMPTY       && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == BLACK_PIECE && board_array[column][j+2] == EMPTY       && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == EMPTY       && board_array[column][j+2] == BLACK_PIECE && board_array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == EMPTY       && board_array[column][j+2] == EMPTY       && board_array[column][j+3] == BLACK_PIECE) ){
            score += 1;
        }
        if ( (array[column][j] == BLACK_PIECE && array[column][j+1] == BLACK_PIECE && array[column][j+2] == EMPTY       && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == BLACK_PIECE && array[column][j+2] == BLACK_PIECE && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == EMPTY       && array[column][j+2] == BLACK_PIECE && array[column][j+3] == BLACK_PIECE) ){
            score += 4;
        }
        if ( (array[column][j] == BLACK_PIECE && array[column][j+1] == BLACK_PIECE && array[column][j+2] == BLACK_PIECE && array[column][j+3] == EMPTY) ||
             (array[column][j] == EMPTY       && array[column][j+1] == BLACK_PIECE && array[column][j+2] == BLACK_PIECE && array[column][j+3] == BLACK_PIECE) ){
            score += 32;
        }
        j++;
    }

    // Count diagonal downward score
    i = 0;
    j = 0;
    while ( i < 4 ){
        while ( j < 3 ){
            if ( (i == row || i+1 == row || i+2 == row || i+3 == row) &&
                 (j == column || j+1 == column || j+2 == column || j+3 == column) ) {
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j+1] == EMPTY       && array[i+2][j+2] == EMPTY       && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == BLACK_PIECE && array[i+2][j+2] == EMPTY       && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == EMPTY       && array[i+2][j+2] == BLACK_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == EMPTY       && array[i+2][j+2] == EMPTY       && array[i+3][j+3] == BLACK_PIECE) ){
                    score += 1;
                }
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j+1] == BLACK_PIECE && array[i+2][j+2] == EMPTY       && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == BLACK_PIECE && array[i+2][j+2] == BLACK_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == EMPTY       && array[i+2][j+2] == BLACK_PIECE && array[i+3][j+3] == BLACK_PIECE) ){
                    score += 4;
                }
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j+1] == BLACK_PIECE && array[i+2][j+2] == BLACK_PIECE && array[i+3][j+3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j+1] == BLACK_PIECE && array[i+2][j+2] == BLACK_PIECE && array[i+3][j+3] == BLACK_PIECE) ){
                    score += 32;
                }
            }
            j++;
        }
        i++;
        j = 0;
    }

    // Count diagonal upward score
    i = 0;
    j = 3;
    while ( i < 4 ){
        while ( j < 6 ){
            if ( (i == row || i+1 == row || i+2 == row || i+3 == row) &&
                 (j == column || j-1 == column || j-2 == column || j-3 == column) ) {
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j-1] == EMPTY       && array[i+2][j-2] == EMPTY       && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == BLACK_PIECE && array[i+2][j-2] == EMPTY       && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == EMPTY       && array[i+2][j-2] == BLACK_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == EMPTY       && array[i+2][j-2] == EMPTY       && array[i+3][j-3] == BLACK_PIECE) ){
                    score += 1;
                }
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j-1] == BLACK_PIECE && array[i+2][j-2] == EMPTY       && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == BLACK_PIECE && array[i+2][j-2] == BLACK_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == EMPTY       && array[i+2][j-2] == BLACK_PIECE && array[i+3][j-3] == BLACK_PIECE) ){
                    score += 4;
                }
                if ( (array[i][j] == BLACK_PIECE && array[i+1][j-1] == BLACK_PIECE && array[i+2][j-2] == BLACK_PIECE && array[i+3][j-3] == EMPTY) ||
                     (array[i][j] == EMPTY       && array[i+1][j-1] == BLACK_PIECE && array[i+2][j-2] == BLACK_PIECE && array[i+3][j-3] == BLACK_PIECE) ){
                    score += 32;
                }
            }
            j++;
        }
        i++;
        j = 3;
    }
    }
    return score;
}
