//
// License: GPL
// Date: 25.02.2006
// Author: Andr�Schnabel (thefrogs@web.de)
// http://www.masterspace.biz/andreschnabel/
//

// ResourceManager.cpp

#include "ResourceManager.h"

//
// ResourceManager
// Constructor
//
ResourceManager::ResourceManager(string filename)
{
	char curLine[MAX_STR_LENGTH];
	ifstream inputStream;
	ResType type;
	string rawName;
	int ptSize;
	char temp[MAX_STR_LENGTH];
	int numResources, curResource = 0;
	int curPercent = 0;

	framework->Output("Initializing resource manager...");

	numResources = framework->CountLinesInFile(filename);

	inputStream.open(filename.c_str());
	if(!inputStream.good())
	{
		string errStr = "Can't open resource input file: ";
		errStr += filename;
		errStr += "!";
		framework->Error(errStr);
	}

	while(!inputStream.eof())
	{
		inputStream.getline(curLine, MAX_STR_LENGTH);

		int strLength = strlen(curLine);
		curLine[strLength-1] = '\0';

		if(!isalpha(curLine[0]))
			continue;

		curResource++;

		type = IsResType(curLine);
		rawName = StripToRawName(curLine);

		if(images[rawName] || sounds[rawName] || songs[rawName] || fonts[rawName])
		{
			string errMsg = "Two resources with same name: ";
			errMsg += rawName;
			errMsg += "!";
			framework->Error(errMsg);
		}

		switch(type)
		{
		case RES_IMAGE:
			images[rawName] = framework->LoadImage(curLine, true);
			break;
		case RES_SOUND:
			sounds[rawName] = framework->LoadSound(curLine);
			break;
		case RES_MUSIC:
			songs[rawName] = framework->LoadMusic(curLine);
			break;
		case RES_FONT:
			strcpy_s(temp, curLine);
			inputStream.getline(curLine, MAX_STR_LENGTH);

			if(!strcmp(curLine, "Tiny"))
				ptSize = 5;
			else if(!strcmp(curLine, "Small"))
				ptSize = 9;
			else if(!strcmp(curLine, "Medium"))
				ptSize = 14;
			else if(!strcmp(curLine, "Big"))
				ptSize = 20;
			else if(!strcmp(curLine, "Huge"))
				ptSize = 40;

			fonts[rawName] = framework->LoadFont(temp, ptSize);
			break;
		}

		curPercent = (100 / numResources) * curResource;
		DrawLoadingBar(curPercent);
	}	

	inputStream.close();

	framework->Output("Resource manager initialized!");
}

//
// ~ResourceManager
// Destructor
//
ResourceManager::~ResourceManager()
{
	framework->Output("Shutting down resource manager...");

	images.clear();
	sounds.clear();
	songs.clear();

	framework->Output("Resource manager shut down!");
}

//
// GetImage
//
SDL_Surface *ResourceManager::GetImage(string name)
{
	SDL_Surface *image = images[name];
	if(!image)
	{
		string errMsg = "Failed to get image: ";
		errMsg += name;
		errMsg += "!";
		framework->Error(errMsg);
	}

	return image;
}

//
// GetSound
//
Mix_Chunk *ResourceManager::GetSound(string name)
{
	Mix_Chunk *sound = sounds[name];
	if(!sound && !framework->GetAudioDisabled())
	{
		string errMsg = "Failed to get sound: ";
		errMsg += name;
		errMsg += "!";
		framework->Error(errMsg);
	}

	return sound;
}

//
// GetMusic
//
Mix_Music *ResourceManager::GetMusic(string name)
{
	Mix_Music *song = songs[name];
	if(!song && !framework->GetAudioDisabled())
	{
		string errMsg = "Failed to get song: ";
		errMsg += name;
		errMsg += "!";
		framework->Error(errMsg);
	}

	return song;
}

//
// GetFont
//
TTF_Font *ResourceManager::GetFont(string name)
{
	TTF_Font *font = fonts[name];
	if(!font)
	{
		string errMsg = "Failed to get font: ";
		errMsg += name;
		errMsg += "!";
		framework->Error(errMsg);
	}

	return font;
}

//
// IsResType
//
ResourceManager::ResType ResourceManager::IsResType(string filename) const
{
	char string[MAX_STR_LENGTH], extension[MAX_STR_LENGTH];
	char *c;
	int i;

	strcpy_s(string, filename.c_str());

	for(c=string; *c; c++)
	{
		if(*c == '.')
		{
			c++;
			for(i=0; *c; i++, c++)
				extension[i] = *c;
				
			extension[i] = '\0';

			break;
		}
	}


	if((strcmp(extension, "bmp") == 0) || (strcmp(extension, "jpg") == 0) || (strcmp(extension, "png") == 0))
		return RES_IMAGE;
	else if((strcmp(string, "Data/gameover.wav") == 0) || (strcmp(string, "Data/mov.wav") == 0))
		return RES_SOUND;
	else if(strcmp(string, "Data/music.wav") == 0)
		return RES_MUSIC;
	else if(strcmp(extension, "ttf") == 0)
		return RES_FONT;
	else
	{
		cout << "String:" << string << "Extension:" << extension << endl;
		framework->Error("Unknown resource!");
		return RES_UNKNOWN;
	}
}

//
// StripToRawName
//
string ResourceManager::StripToRawName(string filename) const
{
	char rawName[MAX_STR_LENGTH];
	int length, i;
	char fullName[MAX_STR_LENGTH];
	int lastSlash = 0;
	char *c;

	strcpy_s(fullName, filename.c_str());
	length = (int)strlen(fullName);

	// Strip extension
	for(i=length; i>0; i--)
	{		
		if(fullName[i] == '.')
		{
			fullName[i] = '\0';
			break;
		}
	}

	// Find last slash
	for(i=0, c=fullName; *c; c++, i++)
	{
		if(*c == '/')
			lastSlash = i;
	}

	// Copy the name behind that
	for(i=0, c=&fullName[lastSlash+1]; *c; c++, i++)
		rawName[i] = *c;
	rawName[i] = '\0';

	return rawName;
}

//
// DrawLoadingBar
//
void ResourceManager::DrawLoadingBar(int percent) const
{
	static SDL_Rect rect = { 0, 0, 0, 10 };

	framework->FillRect();

	rect.y = framework->GetScreenH() - 10;
	rect.w = (framework->GetScreenW() / 100) * percent;

	framework->FillRect(&rect, Color::white);

	framework->Flip();
}
