// game.c

/*
	Sokoban
	2006 by André Schnabel (thefrogs@web.de)
	http://www.semler-service.de/andre/
	License: GPL
*/

#include "common.h"
#include "game.h"

static char tileFilenames[NUM_TILES-1][MAX_STR_LENGTH] = {
	"floor.bmp",
	"target.bmp",
	"wall.bmp",
	"box.bmp",
	"ps.bmp"
};

Uint32 bgColor;
SDL_Surface *playerImg, *tileImgs[NUM_TILES-1], *cursorImg;

//Need to change to chars, to allow Sokoban format!
char map[MAP_H][MAP_W];
//int map[MAP_H][MAP_W];

int curMap;
char curMapName[MAX_STR_LENGTH];
player_t player;
int numBoces;
boolean levelDone;
Uint32 levelDoneStart;

boolean editMode;
int curSelTile, selPos[2];

target_t hiddenTargets[MAX_HIDDEN_TARGETS];

Mix_Music *music;
Mix_Chunk *stepSnd, *pushSnd, *levelDoneSnd;

/*
#define CLEAR_MAP() \
	memset(map, 0, sizeof(int) * MAP_W * MAP_H)
*/

#define CLEAR_MAP() \
	memset(map, 0, sizeof(char) * MAP_W * MAP_H)

void GM_LoadMap(const char *filename, int *psX, int *psY);

void GM_NewRound(void)
{
	int i;
	
	editMode = false;
	curSelTile = 0;
	selPos[0] = selPos[1] = 0;
	
	for(i=0; i<MAX_HIDDEN_TARGETS; i++)
	{
		hiddenTargets[i].x = hiddenTargets[i].y = 0;
		hiddenTargets[i].active = false;
	}
	player.bocesDone = 0;

	sprintf(curMapName, DATA_PREFIX MAP_PREFIX "%d", curMap + 1);
	
	if(COM_FileExists(curMapName))
	
		//Load Map can add hiddenTargets and increase player.bocesDone!
		GM_LoadMap(curMapName, &player.x, &player.y);
	else	
	{
		CLEAR_MAP();
		player.x = player.y = 0;
		numBoces = 0;
	}	
	
	player.steps = 0;
	player.pushes = 0;
	levelDone = false;
}

void GM_AddHiddenTarget(int x, int y)
{
	int i;
	
	for(i=0; i<MAX_HIDDEN_TARGETS; i++)
	{
		if(!hiddenTargets[i].active)
		{
			hiddenTargets[i].x = x;
			hiddenTargets[i].y = y;
			hiddenTargets[i].active = true;
			break;
		}
	}
}

void GM_Init(void)
{
	int i;
	
	playerImg = COM_LoadImage("player.bmp", true);
	
	for(i=0; i<NUM_TILES-1; i++)
		tileImgs[i] = COM_LoadImage(tileFilenames[i], false);
	
	cursorImg = COM_LoadImage("cursor.bmp", true);

	bgColor = SDL_MapRGB(screen->format, 10, 10, 10);
	
	curMap = 0;
	GM_NewRound();
	
	music = Mix_LoadMUS(DATA_PREFIX AUDIO_PREFIX "music.wav");
	if(music){
		Mix_PlayMusic(music, -1);
	}

	stepSnd = COM_LoadSound("step.wav");
	pushSnd = COM_LoadSound("push.wav");
	levelDoneSnd = COM_LoadSound("levelDone.wav");
	
}

void GM_LoadMap(const char *filename, int *psX, int *psY)
{
	FILE *fp;
	int i, j, ret;
	boolean playerStartFound = false;
	char caracter;
	
	fp = fopen(filename, "rb");

	//Read and discard EOL!
	i=0;j=0;
	
	while ((ret = fread(&caracter, 1, 1, fp)) > 0)
	{
		if ( ((int)(caracter)) != 10 )
		{
			map[i][j] = caracter;
			j++;
			if (j == MAP_W){ j=0; i++; }
		}
		else
		{
			//Ok, check if there are still spaces to be filled!
			for (j = j; j<MAP_W; j++){
				map[i][j] = '0';
			}
			//Jump!
			j=0; i++;
		}
	}
	fclose(fp);
	
	//Check for missing spaces, too!
	for (j = j; j<MAP_W; j++){
		map[i][j] = '0';
	}

	i++;
	for (i = i; i<MAP_H; i++){
		for (j = 0; j<MAP_W; j++){
			map[i][j] = '0';
		}
	}
	
	
	numBoces = 0;
	
	for(i=0; i<MAP_H; i++)
	{
		for(j=0; j<MAP_W; j++)
		{
			if(map[i][j] == TILE_PLAYER_START)
			{
				map[i][j] = TILE_FLOOR;
				*psX = j;
				*psY = i;
				playerStartFound = true;
			}
			else if(map[i][j] == TILE_PLAYER_TARGET)
			{
				map[i][j] = TILE_TARGET;
				*psX = j;
				*psY = i;
				playerStartFound = true;
			}
			else if(map[i][j] == TILE_BOX_TARGET)
			{
				map[i][j] = TILE_BOX;
				GM_AddHiddenTarget(j,i); //box render, but behind a target!
				numBoces++;
				player.bocesDone++;
			}
			else if(map[i][j] == TILE_BOX)
				numBoces++;
		}
	}
	
	if(!playerStartFound)
		*psX = *psY = 0;
}

void GM_SaveMap(const char *filename)
{
	FILE *fp;
	
	fp = fopen(filename, "wb");
	fwrite(map, sizeof(map), 1, fp);
	fclose(fp);
}

void GM_Quit(void)
{
	int i;
	
	Mix_FreeChunk(levelDoneSnd);
	Mix_FreeChunk(pushSnd);
	Mix_FreeChunk(stepSnd);
	
	Mix_HaltMusic();
	Mix_FreeMusic(music);	
	SDL_FreeSurface(cursorImg);
	
	for(i=0; i<NUM_TILES-1; i++)
		SDL_FreeSurface(tileImgs[i]);
	
	SDL_FreeSurface(playerImg);
}

#define DO_STEP(dX, dY) \
			player.x += dX; \
			player.y += dY; \
			player.steps++; \
			Mix_PlayChannel(-1, stepSnd, 0);

#define DO_PUSH(dX, dY) \
			player.x += dX; \
			player.y += dY; \
			player.pushes++; \
			Mix_PlayChannel(-1, pushSnd, 0);

#define BOX_DONE() \
			player.bocesDone++; \
			if(player.bocesDone == numBoces) \
			{ \
				levelDone = true; \
				levelDoneStart = SDL_GetTicks(); \
				Mix_PlayChannel(-1, levelDoneSnd, 0); \
			}

boolean GM_HiddenTargetAtPos(int x, int y)
{
	int i;
	
	for(i=0; i<MAX_HIDDEN_TARGETS; i++)
	{
		if(hiddenTargets[i].active)
		{
			if(hiddenTargets[i].x == x && hiddenTargets[i].y == y)
			{
				// Deactivate this
				hiddenTargets[i].x = hiddenTargets[i].y = 9999;
				player.bocesDone--;
				return true;
			}
		}
		else
			break;
	}
	
	return false;
}

void GM_Input(void)
{
	static boolean left, right, up, down, any;
	static int dX, dY;
	
	if(!editMode)
	{
		editMode = COM_KeyPressed(SDLK_e, true);
		
		//Restart level?
		if(COM_KeyPressed(SDLK_r, true))
			GM_NewRound();
		
		//Change level?
		if(COM_KeyPressed(SDLK_a, true) && (curMap > 0)){
			curMap--;
			GM_NewRound();
		}
		
		if(COM_KeyPressed(SDLK_q, true)){
			curMap++;
			GM_NewRound();
		}
		
		left = COM_KeyPressed(SDLK_LEFT, true);
		right = COM_KeyPressed(SDLK_RIGHT, true);
		up = COM_KeyPressed(SDLK_UP, true);
		down = COM_KeyPressed(SDLK_DOWN, true);
		any = left | right | up | down;
		
		if(any)
		{
			if(left)
			{
				dX = -1; dY = 0;
				if(player.x - 1 < 0)
					return;
			}
			else if(right)
			{
				dX = 1; dY = 0;
				if(player.x + 1 >= MAP_W)
					return;
			}
			else if(up)
			{
				dX = 0; dY = -1;
				if(player.y - 1 < 0)
					return;
			}
			else if(down)
			{
				dX = 0; dY = 1;
				if(player.y + 1 >= MAP_H)
					return;
			}
			
			if(map[player.y+dY][player.x+dX] == TILE_FLOOR
				|| map[player.y+dY][player.x+dX] == TILE_TARGET)
			{
				DO_STEP(dX, dY);
			}
			else
			{
				if((left && player.x - 2 < 0)
					|| (right && player.x + 2 >= MAP_W)			
					|| (up && player.y - 2 < 0)
					|| (down && player.y + 2 >= MAP_H))
					return;
				
				if(map[player.y+dY][player.x+dX] == TILE_BOX &&
					(map[player.y+2*dY][player.x+2*dX] == TILE_FLOOR ||
					 map[player.y+2*dY][player.x+2*dX] == TILE_TARGET))
				{
					if(GM_HiddenTargetAtPos(player.x+dX, player.y+dY))
						map[player.y+dY][player.x+dX] = TILE_TARGET;
					else
						map[player.y+dY][player.x+dX] = TILE_FLOOR;
					
					if(map[player.y+2*dY][player.x+2*dX] == TILE_TARGET)
					{
						GM_AddHiddenTarget(player.x+2*dX, player.y+2*dY);
						BOX_DONE();
					}				
					
					map[player.y+2*dY][player.x+2*dX] = TILE_BOX;
					
					DO_PUSH(dX, dY);
				}
			}
		}
	}
	else
	{
		editMode = !COM_KeyPressed(SDLK_e, true);
		
		if(COM_KeyPressed(SDLK_PLUS, true))
			curSelTile++;
		else if(COM_KeyPressed(SDLK_MINUS, true))
			curSelTile--;
		
		if(curSelTile < 0)
			curSelTile = NUM_TILES - 1;
		else if(curSelTile > NUM_TILES - 1)
			curSelTile = 0;
		
		if(COM_KeyPressed(SDLK_UP, true))
		{
			selPos[1]--;
			if(selPos[1] < 0)
				selPos[1] = 0;	
		}
		
		if(COM_KeyPressed(SDLK_DOWN, true))
		{
			selPos[1]++;
			if(selPos[1] >= MAP_H)
				selPos[1] = MAP_H - 1;
		}
		
		if(COM_KeyPressed(SDLK_LEFT, true))
		{
			selPos[0]--;
			if(selPos[0] < 0)
				selPos[0] = 0;
		}
		
		if(COM_KeyPressed(SDLK_RIGHT, true))
		{
			selPos[0]++;
			if(selPos[0] >= MAP_W)
				selPos[0] = MAP_W - 1;
		}
		
		if(COM_KeyPressed(SDLK_RETURN, true))
			map[selPos[1]][selPos[0]] = curSelTile;
		
		if(COM_KeyPressed(SDLK_s, true))
			GM_SaveMap(curMapName);
	}
}

void GM_Update(void)
{
}

void GM_Draw(void)
{
	int i, j;
	
	SDL_FillRect(screen, NULL, bgColor);
	
	for(i=0; i<MAP_H; i++)
		for(j=0; j<MAP_W; j++)
			if(map[i][j] != TILE_EMPTY && (map[i][j] != TILE_PLAYER_START || editMode)){

				//Do conversion between map chars, and the structure of resources!!
				if (map[i][j] == TILE_FLOOR) COM_DrawImage(tileImgs[0], j * TILE_W, i * TILE_H);
				else if (map[i][j] == TILE_TARGET) COM_DrawImage(tileImgs[1], j * TILE_W, i * TILE_H);
				else if (map[i][j] == TILE_WALL) COM_DrawImage(tileImgs[2], j * TILE_W, i * TILE_H);
				else if (map[i][j] == TILE_BOX) COM_DrawImage(tileImgs[3], j * TILE_W, i * TILE_H);

				//COM_DrawImage(tileImgs[map[i][j]-1], j * TILE_W, i * TILE_H);
			}
	
	if(editMode)
	{
		static char curSelTileStr[MAX_STR_LENGTH];
		COM_DrawImage(cursorImg, selPos[0] * TILE_W, selPos[1] * TILE_H);
		sprintf(curSelTileStr, "Selected Tile: %d", curSelTile);
		COM_DrawText(curSelTileStr, 0, 0);
	}
	else
	{
		static char stepsAndPushesStr[MAX_STR_LENGTH];
		
		COM_DrawImage(playerImg, player.x * TILE_W, player.y * TILE_H);
		
		sprintf(stepsAndPushesStr, "Steps: %d Pushes: %d Lvl %d", player.steps, player.pushes, (curMap+1));
		COM_DrawText(stepsAndPushesStr, 3, 445);
		
		if(levelDone)
		{
			if(SDL_GetTicks() - levelDoneStart < SHOW_LEVEL_DONE_TIME)
				COM_DrawText("Level done!", 100, 100);
			else
			{
				curMap++;
				GM_NewRound();
			}
		}
	}
}
