// common.c

/*
	Sokoban
	2006 by André Schnabel (thefrogs@web.de)
	http://www.semler-service.de/andre/
	License: GPL
*/

#include "common.h"

#ifdef DREAMCAST

	#include <SDL/SDL_dreamcast.h>

#endif

SDL_Surface *screen;
Uint8 *keys, *oldKeys;
int numKeys;
boolean curStateDone;
Uint32 magenta;
SDL_Surface *fontMap;
int charW, charH;

void COM_Init(const char *caption, int scrW, int scrH, int scrBpp, boolean fullscreen)
{
	int result;
	//Uint32 scrFlags = SDL_HWSURFACE | SDL_DOUBLEBUF;
	Uint32 scrFlags = SDL_SWSURFACE;

	result = SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_JOYSTICK);
	if(result == -1)
		COM_Error("Can't initialize the SDL!");
	
	if(fullscreen)
		scrFlags |= SDL_FULLSCREEN;
	
	screen = SDL_SetVideoMode(scrW, scrH, scrBpp, scrFlags);
	if(!screen)
		COM_Error("Can't set video mode!");
		
	SDL_WM_SetCaption(caption, NULL);
	SDL_ShowCursor(SDL_FALSE);
	
	result = Mix_OpenAudio(MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, MIX_DEFAULT_CHANNELS, 1024);
	if(result == -1)
		COM_Error("Can't open audio!");
	
	keys = SDL_GetKeyState(&numKeys);
	oldKeys = (Uint8 *)malloc(numKeys * sizeof(Uint8));
	
	#ifdef DREAMCAST
		//NOTE: Edit mode will not be binded: I prefer to create maps
		//on PC and then transfer into DC build.
 		SDL_DC_MapKey(0, SDL_DC_UP, SDLK_UP);
   		SDL_DC_MapKey(0, SDL_DC_DOWN, SDLK_DOWN);
   		SDL_DC_MapKey(0, SDL_DC_LEFT, SDLK_LEFT);
   		SDL_DC_MapKey(0, SDL_DC_RIGHT, SDLK_RIGHT);
   		SDL_DC_MapKey(0, SDL_DC_START, SDLK_RETURN);
   		SDL_DC_MapKey(0, SDL_DC_B, SDLK_r);
		//SDL_DC_MapKey(0, SDL_DC_A, SDLK_e);
		SDL_DC_MapKey(0, SDL_DC_X, SDLK_a);
		SDL_DC_MapKey(0, SDL_DC_Y, SDLK_q);
   		SDL_DC_MapKey(0, SDL_DC_L, SDLK_ESCAPE);

   		SDL_JoystickOpen(0);
		SDL_Event event;
	    while(SDL_PollEvent(&event))
        	SDL_Delay(20);
	#endif

	magenta = SDL_MapRGB(screen->format, 255, 0, 255);
	
	fontMap = COM_LoadImage("fontmap.bmp", true);
	charW = fontMap->w / ('@' - '!');
	charH = fontMap->h / 3;
}

void COM_Quit(void)
{
	SDL_FreeSurface(fontMap);
	free(oldKeys);
	Mix_CloseAudio();
	SDL_Quit();
}

void COM_Error(const char *message)
{
	printf("Fatal error: %s\n", message);
	fflush(stdout);
	exit(1);
}

void COM_DrawImage(SDL_Surface *image, int x, int y)
{
	static SDL_Rect rect;
	
	rect.x = x;
	rect.y = y;
	
	rect.w = image->w;
	rect.h = image->h;
	
	SDL_BlitSurface(image, NULL, screen, &rect);
}

void COM_DrawText(const char *text, int x, int y)
{
	const char *c;
	int row, column;
	static SDL_Rect srcRect, destRect;
	
	srcRect.w = destRect.w = charW;
	srcRect.h = destRect.h = charH;
	
	destRect.y = y;
	
	for(c = text; *c; c++, x += charW)
	{
		if(*c >= '!' && *c <= '@')
		{
			row = 0;
			column = *c - '!';
		}
		else if(*c >= 'A' && *c <= '_')
		{
			row = 1;
			column = *c - 'A';
		}
		else if(*c >= 'a' && *c <= '~')
		{
			row = 2;
			column = *c - 'a';
		}
		else
			continue;
		
		srcRect.x = column * charW;
		srcRect.y = row * charH;
		destRect.x = x;
		
		SDL_BlitSurface(fontMap, &srcRect, screen, &destRect);
	}
}

boolean COM_UpdateKeys(void)
{
	static SDL_Event event;
	
	memcpy(oldKeys, keys, sizeof(Uint8) * numKeys);
	
	while(SDL_PollEvent(&event))
	{
		if(event.type == SDL_QUIT)
			return true;
	}

	SDL_PumpEvents();
	
	return false;
}

boolean COM_KeyPressed(int keyNum, boolean wasReleased)
{
	return wasReleased ? keys[keyNum] && ! oldKeys[keyNum] : keys[keyNum];
}

void COM_MainLoop(state_t *state)
{
	Uint32 sTime, diff;

	curStateDone = false;

	state->init();
	
	while(!curStateDone)
	{
		sTime = SDL_GetTicks();
		
		if(COM_UpdateKeys() || COM_KeyPressed(SDLK_ESCAPE, true))
			break;
		
		state->input();		
		state->update();
		state->draw();
		
		SDL_Flip(screen);
		
		diff = SDL_GetTicks() - sTime;
		if(diff < 1000 / MAX_FPS)
			SDL_Delay(1000 / MAX_FPS - diff);
	}
	
	state->quit();
}

SDL_Surface *COM_LoadImage(const char *filename, boolean useAlpha)
{
	SDL_Surface *temp, *image;
	char completeFilename[MAX_STR_LENGTH], errMsg[MAX_STR_LENGTH];
	
	strcpy(completeFilename, DATA_PREFIX);
	strcat(completeFilename, IMAGE_PREFIX);
	strcat(completeFilename, filename);
	
	temp = SDL_LoadBMP(completeFilename);
	if(!temp)
	{
		sprintf(errMsg, "Can't load image: %s", completeFilename);
		COM_Error(errMsg);
	}
	
	image = SDL_DisplayFormat(temp);
	SDL_FreeSurface(temp);
	
	if(useAlpha)
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, magenta);
	
	return image;
}

Mix_Chunk *COM_LoadSound(const char *filename)
{
	Mix_Chunk *sound;
	char completeFilename[MAX_STR_LENGTH], errMsg[MAX_STR_LENGTH];
	
	strcpy(completeFilename, DATA_PREFIX);
	strcat(completeFilename, AUDIO_PREFIX);
	strcat(completeFilename, filename);
	
	sound = Mix_LoadWAV(completeFilename);
	if(!sound)
	{
		sprintf(errMsg, "Can't load sound: %s", completeFilename);
		COM_Error(errMsg);
	}
	
	return sound;
}

boolean COM_FileExists(const char *filename)
{
	FILE *fp;
	
	fp = fopen(filename, "r");
	if(!fp)
		return false;
	
	fclose(fp);
	
	return true;	
}
