#include "lcavity.h"

#ifdef DREAMCAST
	#include <SDL/SDL_dreamcast.h>
#endif

using namespace Lcav;

// creates a new Lcavity game object
Lcavity::Lcavity(int argc,char* argv[])
{
  setConfig();
  accel=Acceleration(.20,0,120);
  initSDL();
  playGame();
} // Lcavity::Lcavity()

// destroys the object
Lcavity::~Lcavity()
{
} // Lcavity::~Lcavity()

/* sets the configuration.
 * This function will in a later version of this game be able
 * to take the arguments passed to the command-line and use
 * them as parameters for the game.
 */
void Lcavity::setConfig()
{
  width=320;
  height=240;
  hposition=width/4;
  colPosition=0;
  caveChangedistance=width/4;
  prevCavetop=height/4;
  nextCavetop=height/4;
  prevCaveheight=3*height/4-prevCavetop;
  nextCaveheight=3*height/4-nextCavetop;
  for(int i=0;i<width;i++)
  {
    cols[i]=new Gamecol(-1,prevCavetop,prevCavetop+prevCaveheight);
  }
} // Lcavity::setConfig()

/* initializes an SDL video screen (and perhaps later it will
 * also initialize SDL sound output.
 * This function will later on be able to set fullscreen right
 * at the beginning if passed by a command-line parameter
 */
int Lcavity::initSDL()
{
  // initialize the video screen
  if(SDL_Init(SDL_INIT_VIDEO|SDL_INIT_JOYSTICK)>0)
  {
    fprintf(stderr,"Couldn't initalize SDL: %s\n",SDL_GetError());
    exit(1);
  }
  screen=SDL_SetVideoMode(width,height,8,SDL_SWSURFACE);
  if(screen==NULL)
  {
    fprintf(stderr,"Couldn't set %sx%sx8video mode: %s\n",width,height,SDL_GetError());
    exit(1);
  }
  
  #ifdef DREAMCAST
	 if(SDL_InitSubSystem(SDL_INIT_JOYSTICK) == 0){
	  SDL_DC_MapKey(0,SDL_DC_START,SDLK_RETURN); //START button
	  SDL_DC_MapKey(0,SDL_DC_UP,SDLK_UP);
	  SDL_DC_MapKey(0,SDL_DC_DOWN,SDLK_DOWN);
	  SDL_DC_MapKey(0,SDL_DC_LEFT,SDLK_LEFT);
	  SDL_DC_MapKey(0,SDL_DC_RIGHT,SDLK_RIGHT);
	  SDL_DC_MapKey(0,SDL_DC_A,SDLK_SPACE); //a button
	  SDL_DC_MapKey(0,SDL_DC_R,SDLK_ESCAPE); //L button
	  SDL_JoystickOpen(0);
	  SDL_ShowCursor(false);
	  
	  //Dreamcast delay...
	  SDL_Delay(20);
	  SDL_Event event;
	  while(SDL_PollEvent(&event))
	   SDL_Delay(20);
	 }
  #endif
} // int Lcavity::initSDL()

/* draws a column with a single-pixel width to the sdl screen.
 * is called for each pixel of the screen width in every step
 * of the game.
 */
void Lcavity::drawColumn(int x,int pointheight,int top,int bottom)
{
  //create the background (make it all black again)
  SDL_PixelFormat* pixelFormat=screen->format;
  SDL_Rect *background=new SDL_Rect;
  background->x=x;
  background->w=1;
  background->y=0;
  background->h=height;
  Uint32 backgroundColor;
  backgroundColor=SDL_MapRGB(pixelFormat,0,0,0);
  int backgroundRect=SDL_FillRect(screen,background,backgroundColor);
  SDL_free(background); //free resources!

  //create the point in the between
  SDL_Rect *point=new SDL_Rect;
  point->y=pointheight;
  point->w=1;
  point->h=1;
  point->x=x;
  Uint32 pixelColor;
  pixelColor=SDL_MapRGB(pixelFormat,255,255,255);
  int frect=SDL_FillRect(screen,point,pixelColor);
  SDL_free(point);

  //create the top rectangle
  if(top>=0)
  {
    SDL_Rect *topRect=new SDL_Rect;
    topRect->y=0;
    topRect->h=top;
    topRect->w=1;
    topRect->x=x;
    int trect=SDL_FillRect(screen,topRect,pixelColor);
    SDL_free(topRect);
  }

  //create the bottom rectangle
  if(bottom>=0)
  {
    SDL_Rect *bottomRect=new SDL_Rect;
    bottomRect->y=bottom;
    bottomRect->h=height-bottom;
    bottomRect->w=1;
    bottomRect->x=x;
    int brect=SDL_FillRect(screen,bottomRect,pixelColor);
    SDL_free(bottomRect);
  }

} // void Lcavity::drawColumn()

/* starts the gameplay.
 * this function will later on be called by another function controling the
 * menus and so on. It checks in each round the pressed keys (and sets the
 * acceleration after it) by calling checkKeyInput(), calculates the next
 * step of the game by calling the stepOn() function and then updates the
 * screen by calling updateScreen(). This is being done, until a collision
 * in the game occurs. Later on this will be extended, so that you can use
 * the Escape key or the 'q' key to leave the game.
 */
void Lcavity::playGame()
{
  while(1)
  {
    checkKeyInput();
    stepOn();
    updateScreen();
    
    if (cols[colPosition]->collides()){ //Reset game!
    	setConfig();
    	accel=Acceleration(.20,0,120);
    }
  }
} // void Lcavity::playGame()

/* updates the whole screen to the currently saved step
 * by drawing the whole saved cols to the screen and updating it afterwards
 */
void Lcavity::updateScreen()
{
  for(int i=0;i<width;i++)
  {
    int curCol=(i+colPosition+width-hposition)%width;
    drawColumn(i,cols[curCol]->getPointheight(),cols[curCol]->getTop(),cols[curCol]->getBottom());
  }
  SDL_UpdateRect(screen,0,0,width,height);
} // void Lcavity::updateScreen()

/* calculates the next step that is done in the game
 * this function sets the needed values in the column that is being wrapped around
 * the screen beginning and also in the column where the point is flying at the
 * moment.
 */
void Lcavity::stepOn()
{
  int oldPosition=(colPosition+width-hposition)%width;
  cols[oldPosition]->setPointheight(-1);
  if(colPosition%caveChangedistance==0)
  {
    changeCave();
  }
  cols[oldPosition]->setTop(getTop(oldPosition));
  cols[oldPosition]->setBottom(getBottom(oldPosition));
  colPosition++;
  colPosition%=width; // set colPosition to zero if it is larger than the width
  checkKeyInput();
  float pos=accel.getPosition(1);
  accel.setToCurrent(1);
  cols[colPosition]->setPointheight((int)pos);
} // void Lcavity::stepOn()

/* returns the top point at a certain position
 * this function gives you the position where the top of the cave is at a certain
 * position.
 */
int Lcavity::getTop(int position)
{
  int rawPosition=position%caveChangedistance;
  float returnPosition=(rawPosition*(nextCavetop-prevCavetop))/caveChangedistance+prevCavetop;
  return(int)returnPosition;
} // int Lcavity::getTop()

/* returns the bottom point at a certain position
 * this function gives you the position where the bottom of the cave is at a certain
 * position.
 */
int Lcavity::getBottom(int position)
{
  int rawPosition=position%caveChangedistance;
  int prevCavebottom=prevCavetop+prevCaveheight;
  int nextCavebottom=nextCavetop+nextCaveheight;
  float returnPosition=(rawPosition*(nextCavebottom-prevCavebottom))/
    caveChangedistance+prevCavebottom;
  return (int)returnPosition;
} // int Lcavity::getBottom()

/* calculates the top position and height of the cave at the next change point.
 * in general, the height of the cave is reduced by a certain factor until it
 * has reached a minimum height required. in addition, the next top point is
 * created by a simple randomizer.
 * somewhere in the future, I will perhaps find a way to calculate only change
 * points where it is possible to get around when playing the game (which is
 * currently not the case. if you reach a change point which changes too hard
 * to get around it is bad luck for you).
 */
void Lcavity::changeCave()
{
  prevCavetop=nextCavetop;
  prevCaveheight=nextCaveheight;
  nextCaveheight=(nextCaveheight>10?(int)(prevCaveheight*.95):nextCaveheight);
  // a simple randomizer for getting the next position
  srand((unsigned)time(NULL));
  nextCavetop=rand()%(height-nextCaveheight);
} // void Lcavity::changeCave()

/* checks the input keys and sets the acceleration after it.
 * this function checks if the space bar is pressed, and sets the acceleration
 * up, if space is pressed, and sets it down if space is not pressed.
 */
void Lcavity::checkKeyInput()
{
  if(grabKeyInput())
  {
    accel.setAcceleration(-.20);
  }
  else
  {
    accel.setAcceleration(.20);
  }
} // void Lcavity::checkKeyInput()

/* takes all the input from the keyboard
 * and returns a bool indicating wether the space bar is pressed or not.
 * in the future, this function will have to change because it will be
 * possible to use the 'q' or ESC key to leave the game area.
 */
bool Lcavity::grabKeyInput()
{
  static bool keyIsDown;
  SDL_Event event;
  while(SDL_PollEvent(&event))
  {
    if(event.type==SDL_KEYDOWN&&event.key.keysym.sym==SDLK_SPACE)
    {
      keyIsDown=true;
    }
    if(event.type==SDL_KEYUP&&event.key.keysym.sym==SDLK_SPACE)
    {
      keyIsDown=false;
    }
  }
  return keyIsDown;
} // bool Lcavity::grabInputKey()
