#define TIL_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <png/png.h>

#include "utils.h"
#include "tiles.h"
#include "video.h"

#if !defined(TIL_CALCULATE_UV)

/* ---------------------------------------------------------------- */
/* lookup tables used by til_render to determine u and v values     */
/* for a given tile - if there are more tiles per line then this    */
/* will have to be updated - or TILE_CALCULATE_UV can be used.      */
/* ---------------------------------------------------------------- */

float u1_lookup[64] = { 0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875,
                        0, 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875 };

float u2_lookup[64] = { 0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1,
                        0.125, 0.25, 0.375, 0.5, 0.625, 0.75, 0.875, 1};

float v1_lookup[64] = { 0,     0,     0,     0,     0,     0,     0,     0,
                        0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125,
                        0.25,  0.25,  0.25,  0.25,  0.25,  0.25,  0.25,  0.25,
                        0.375, 0.375, 0.375, 0.375, 0.375, 0.375, 0.375, 0.375,
                        0.5,   0.5,   0.5,   0.5,   0.5,   0.5,   0.5,   0.5,
                        0.625, 0.625, 0.625, 0.625, 0.625, 0.625, 0.625, 0.625,
                        0.75,  0.75,  0.75,  0.75,  0.75,  0.75,  0.75,  0.75,
                        0.875, 0.875, 0.875, 0.875, 0.875, 0.875, 0.875, 0.875};

float v2_lookup[64] = { 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125, 0.125,
                        0.25,  0.25,  0.25,  0.25,  0.25,  0.25,  0.25,  0.25,
                        0.375, 0.375, 0.375, 0.375, 0.375, 0.375, 0.375, 0.375,
                        0.5,   0.5,   0.5,   0.5,   0.5,   0.5,   0.5,   0.5,
                        0.625, 0.625, 0.625, 0.625, 0.625, 0.625, 0.625, 0.625,
                        0.75,  0.75,  0.75,  0.75,  0.75,  0.75,  0.75,  0.75,
                        0.875, 0.875, 0.875, 0.875, 0.875, 0.875, 0.875, 0.875,
                        1,     1,     1,     1,     1,     1,     1,     1};

#endif  /* !defined(TILE_CALCULATE_UV) */

til_set_t sets[TIL_MAX_SETS];                     /* static array of all sets */

/* til_init _________________________________________________________________ */
/*                                                                            */
/* Initialise tile support.                                                   */
/* __________________________________________________________________________ */

void til_init(void)
{
  DBG_ENTRY(99,("til_init()"));
  
  memset(sets,0x00,sizeof(sets));

  DBG_EXIT(99,("til_init"));
  
  return;
}

/* til_term _________________________________________________________________ */
/*                                                                            */
/* Terminate tile support.                                                    */
/* __________________________________________________________________________ */

void til_term(void)
{
  DBG_ENTRY(99,("til_term()"));
  
  DBG_EXIT(99,("til_term"));
  
  return;
}

/* til_set_load _____________________________________________________________ */
/*                                                                            */
/* Ensure that a set is loaded appropriately.                                 */
/* __________________________________________________________________________ */

uint8 til_set_load(uint8 set, uint8 mode)
{
  uint8 rc = 1;
  
  char   file[256];
  FILE   *fp = 0;
  
  uint32 w = 0;
  uint32 h = 0;
  
  int32  tilewidth  = -1;
  int32  tileheight = -1;
  
  DBG_ENTRY(99,("til_set_load(%d,%d)",set,mode));

  /* --------------- */
  /* Texture loading */
  /* --------------- */

  /* Load a TIL file if there is one */
  if (((mode & TIL_LOAD_TEXTURE) && !sets[set].texture) ||
      ((mode & TIL_LOAD_MASK   ) && !sets[set].mask))
  {
    sprintf(file,"%s/gfx/%s%03d.%s",data_dir,TIL_DEF_PREFIX,set,TIL_DEF_FILETYPE);

    DBG_PRINT(99,("Attempting load of %s file (%s)",TIL_DEF_FILETYPE, file));

    /* Load information from a TIL file if there is one */
    
    if ((fp = fopen(file,"rb")))
    {
      char line[1024];
      
      DBG_PRINT(99,("%s file found - reading", TIL_DEF_FILETYPE));
      
       while (fgets(line,sizeof(line),fp))
      {   
        char *t = strtok(line, UTL_WHITESPACE);
        char *e;
      
        if (!t || *t == '#')
          continue;
   
        tilewidth = tileheight = strtol(t,&e,10);

        /* Uneven tiles specify a second value */ 
        if ((t=strtok(0, UTL_WHITESPACE)))
          tileheight = strtol(t,0,0);
                  
        DBG_PRINT(99,("tile size: %dx%d", (int)tilewidth, (int)tileheight));
      }
      
      fclose(fp);
    }
  }  
    
  if (rc && (mode & TIL_LOAD_TEXTURE) && !sets[set].texture)
  {
    sprintf(file,"%s/gfx/%s%03d.%s",data_dir,TIL_DEF_PREFIX,set,TIL_DEF_GFX_FILETYPE);
       
    DBG_PRINT(99,("png_load_texture for %s",file));
  
    if (0 == png_load_texture(file,
                              &sets[set].texture,
                              PNG_FULL_ALPHA,
                              &w,
                              &h))
    {
      pvr_poly_cxt_t cxt;

      pvr_poly_cxt_txr(&cxt,
                       PVR_LIST_TR_POLY,
                       PVR_TXRFMT_ARGB4444,
                       w, h,
                       sets[set].texture,
                       PVR_FILTER_BILINEAR);

      pvr_poly_compile(&sets[set].header,
                       &cxt);
    }
    else
    {
      DBG_PRINT(0,("failed png_load_texture %d",set));
      rc = 0;
    }
  }
  DBG_ELSE(99,("not loading texture (mode=%d tex=%p)",mode,sets[set].texture));
  
  /* ------------- */
  /* Mask creation */
  /* ------------- */

  if (rc && (mode & TIL_LOAD_MASK) && !sets[set].mask)
  {
    kos_img_t rv;

    sprintf(file,"%s/gfx/%s%03d.%s",data_dir,TIL_DEF_PREFIX,set,TIL_DEF_GFX_FILETYPE);
    
    DBG_PRINT(99,("png_to_img for %s",file));

    /* load the png into an img */
    if (0 == png_to_img(file,PNG_FULL_ALPHA,&rv))
    {
      int masksize = rv.w * rv.h;

      DBG_PRINT(99,("MASKSIZE: %d",masksize));
      
      /* allocate mask storage */
      if ((sets[set].mask = utl_malloc(masksize)))
      {
        int m=0;
        uint8  *mask_pos = sets[set].mask;
        uint16 *data_pos = rv.data;

        if (w==0)
          w = rv.w; /* for halftilewidth calculation */
        if (h==0)
          h = rv.h; /* for halftileheight calculation */
        
        if (tilewidth == -1)
          tilewidth = tileheight = w / TIL_TILES_PER_LINE;
        
        /* ------------------------- */
        /* create a BYTE (!) mask... */
        /* ------------------------- */

        /* For X-extent mask, need to do womething here */

        DBG_PRINT(99,("mask w:%lu (%lu) fmt:0x%02X",rv.w,w,(int)rv.fmt));

        /* masksize unusable after this point */
        if (mode & TIL_MASK_FLAT)
        {
          int x = 0;
          int v = 0x00;

          while(masksize-- > 0)
          {
            if (*data_pos++ & 0x8000)
            {
              m++;
              v = 0x01;
            }
            
            if (++x == tilewidth)
            {
              memset(mask_pos, v, tilewidth);              
              mask_pos += tilewidth;
             
              v = 0x00;
              x = 0;
            } 
          }  
        }
        else if (mode & TIL_MASK_SPECIAL)
        {
          int x = 0;
          int y = 0;
          
          while(masksize-- > 0)
          {
            if ((++x % tilewidth) == 0)
            {
              if (y > 1 && y < tileheight-3)
              {
                memset(mask_pos, 0x01, tilewidth);
              }
              else
                memset(mask_pos, 0x00, tilewidth);
            
              mask_pos += tilewidth;
            
              if (x == w)
              {
                DBG_PRINT(99,("%d=%d\n",x,(int)w));
                
                if (y++ == tileheight)
                  y = 0;
                              
                x = 0;
              }
            }
          }
        }
        else
        { 
          while(masksize-- > 0)
          {
            if (*data_pos++ & 0x8000)
            {
              m++;
              *mask_pos = 0x01;
            }
          
            mask_pos++;
          }
        }
        
        DBG_PRINT(99,("MASK: %d",m));
      }

      kos_img_free(&rv,0);

      if (0 == sets[set].mask)
      {
        DBG_PRINT(0,("failed malloc for mask %d",set));
        rc = 0;
      }
    }
    else
    {
      DBG_PRINT(0,("failed png_to_img for '%s'",file));
      rc = 0;
    }
  }
  DBG_ELSE(99,("not loading mask (mode=%d mask=%p)",mode,sets[set].mask));

  if (rc && w)
  {
    sets[set].texture_width = w;
    
    /* If we were not told the tile size, work it out */
    
    if (tilewidth == -1)
      tilewidth = tileheight = w / TIL_TILES_PER_LINE;
       
    sets[set].halftilewidth = tilewidth / 2;
    sets[set].halftileheight = tileheight / 2;

    #if defined(TIL_CALCULATE_UV)
    sets[set].width_delta = (float)tilewidth / (float)w;
    sets[set].height_delta = (float)tileheight / (float)h;
    #endif

    DBG_PRINT(99,("halftilewidth=%d helftileheight=%d",
                  sets[set].halftilewidth, sets[set].halftileheight));
  }

  sets[set].loaded++;
  
  DBG_EXIT(99,("til_set_load rc=%d (loaded=%d)",rc,sets[set].loaded));
  
  return rc;
}
/* til_set_unload ___________________________________________________________ */
/*                                                                            */
/* Clear up all resources associated with a set.                              */
/* __________________________________________________________________________ */

void til_set_unload(uint8 set)
{
  DBG_ENTRY(99,("til_set_unload(%d)",set));
 
  if (sets[set].loaded > 0 && --sets[set].loaded == 0)
  {
    DBG_PRINT(99,("actually unloading set %d",set));

    pvr_mem_free(sets[set].texture); 
    utl_free(sets[set].mask);

    memset(&sets[set],0x00,sizeof(sets[set]));
  }
  
  DBG_EXIT(99,("til_set_unload"));
  
  return;
}

/* til_obj_create_str _______________________________________________________ */
/*                                                                            */
/* Create a tile reference given an input string from a file.                 */
/* __________________________________________________________________________ */

uint32 til_obj_create_str(char *string)
{
  uint32 info = 0x00000001;
 
  uint8 set = 0;
  uint8 tile = 0;
  uint8 mode = TIL_LOAD_BOTH;
  
  char *t;
  char *e;
  
  DBG_ENTRY(99,("til_obj_create_str(%s)",string));
   
  if ((t = strtok(string,UTL_WHITESPACE))) set=(uint8)strtol(t,&e,10);
  if (t==e) info = 0;
              
  if ((t=strtok(0,UTL_WHITESPACE))) tile=(uint8)strtol(t,&e,10);
  if (t==e) info = 0;
  
  if ((t=strtok(0,UTL_WHITESPACE))) mode=(uint8)strtol(t,&e,10);
  if (t==e) info = 0;
  
  /* make sure the set is loaded appropriately */
  if (!til_set_load(set,mode)) info = 0;
              
  DBG_PRINT(99,("Tile %d/%d mode:%d info:%d",(int)set,(int)tile,(int)mode,(int)info));
                
  if (info) {
    /* info for a tile is simply set,tile and mode or'd */     
    info |= (int32)(set<<24)|(tile<<16)|(mode<<8);
  }
  
  DBG_EXIT(99,("til_obj_create_str rc=0x%08X",(unsigned int)info));
  
  return info;
}

/* til_obj_free _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void til_obj_free(uint32 info)
{
  DBG_ENTRY(99,("til_obj_free(0x%08X)",(unsigned int)info));
  
  til_set_unload((uint8)((info&0xFF000000)>>24));
  
  DBG_EXIT(99,("til_obj_free"));
}

/* til_inst_create __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 til_inst_create(uint32 info)
{
  uint32 inst = 0;
  
  DBG_ENTRY(99,("til_inst_create(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("til_inst_create rc=%d",(int)inst));
  
  return inst;
}

/* til_inst_free ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void til_inst_free(uint32 info)
{
  DBG_ENTRY(99,("til_inst_free(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("til_inst_free"));
}

/* til_render _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int til_render(uint8 set,
               uint8 tile,
               int32 map_x, 
               int32 map_y, 
               float z_order,
               int32 rotation, 
               float scale, 
               float fade,
               uint32 colour)
{
  int rc = 1;

  if (vid_gun_screen)
    return rc;

  DBG_ENTRY(99,("til_render(%d,%d,%d,%d,%f,%d,%f,%f,0x%06X)",
               (int)set,(int)tile,(int)map_x,(int)map_y,(double)z_order,
               (int)rotation,(double)scale,(double)fade,(unsigned int)colour));
  
  /* no drawing needed if no texture */
  if (sets[set].texture) 
  { 
    int halftilewidth = (int)((float)sets[set].halftilewidth*scale);
    int halftileheight = (int)((float)sets[set].halftileheight*scale);
    int tilewidth = halftilewidth<<1;
    int tileheight = halftileheight<<1;
    
    /* modify map_x and map_y positions */
    if (view_rot) 
    {
      float real_x = (float)map_x - view_mid_x;
      float real_y = (float)map_y - view_mid_y;
    
      map_x = (int32)(view_mid_x+((real_x*view_rcos)+(real_y*view_rsin)));
      map_y = (int32)(view_mid_y-((real_x*view_rsin)-(real_y*view_rcos)));
    }

    if ((map_x < (view_x-tilewidth))          ||  /* left of view */
        (map_y < (view_y-tileheight))         ||  /* above view */
        (map_x > (view_x+view_w+tilewidth))   ||  /* right of view */
        (map_y > (view_y+view_h+tileheight)))     /* below view */
    {
      rc = 0;
    }
    else
    {
      int    loop = 0;
      
      float  use_scale = scale;
      uint32 use_colour = colour;

      pvr_vertex_t vert; 
    
      int x1, y1, x2, y2, x3, y3, x4, y4;
    
      #if defined(TIL_CALCULATE_UV)
        
      /* -------------------------- */
      /* calculate texture position */
      /* -------------------------- */
    
      float u1 = (float)(tile%TIL_TILES_PER_LINE)*sets[set].width_delta;
      float u2 = u1+sets[set].width_delta;
      float v1 = (float)(tile/TIL_TILES_PER_LINE)*sets[set].height_delta;
      float v2 = v1+sets[set].height_delta;
    
      #else
        
      /* ------------------------ */
      /* look up texture position */
      /* ------------------------ */
    
      float u1 = u1_lookup[tile];
      float u2 = u2_lookup[tile];
      float v1 = v1_lookup[tile];
      float v2 = v2_lookup[tile];
        
      #endif
      
      if (colour & VID_BLACK_BORDER)
      {
        loop = 1;
        use_scale = scale + scale*0.3;
        use_colour = 0x000000;
      }

      z_order = z_order - 0.1f;

      do
      {
        if (loop == 0)
        {
          use_scale = scale;

          if (colour & VID_RAINBOW_COLOUR)
            use_colour = utl_rainbow_palette[((colour & VID_RGB_MASK) + 
                                              (colour & VID_COLOUR_CHANGE? curr_frame : 0)) % UTL_RAINBOW_PALETTE_SIZE];
          else if (colour & VID_FIRE_COLOUR)
            use_colour = utl_fire_palette[((colour & VID_RGB_MASK) +
                                           (colour & VID_COLOUR_CHANGE? curr_frame : 0)) % UTL_FIRE_PALETTE_SIZE];
          else if (colour & VID_EXPLOSION_COLOUR)
            use_colour = utl_explosion_palette[((colour & VID_RGB_MASK) +
                                               (colour & VID_COLOUR_CHANGE? curr_frame : 0)) % UTL_EXPLOSION_PALETTE_SIZE];
          else
            use_colour = colour & VID_RGB_MASK;

          z_order = z_order + 0.1f;
        }

        halftilewidth = (int)((float)sets[set].halftilewidth*use_scale);
        halftileheight = (int)((float)sets[set].halftileheight*use_scale);

        /* --------------------------------------------------------- */
        /* deal with rotation, simple cases first, then general case */
        /* --------------------------------------------------------- */
      
        rotation -= view_rot;
      
        switch(rotation) /* >360 = general case - callers beware! */
        {
          case 360:
          case 0:
            x1 = x3 = (map_x-halftilewidth)-view_x;
            y1 = y2 = (map_y-halftileheight)-view_y;
            x2 = x4 = x1+tilewidth;
            y3 = y4 = y1+tileheight;
            break;
      
          case -270:
          case 90:
            x3 = x4 = (map_x-halftilewidth)-view_x;
            y3 = y1 = (map_y-halftileheight)-view_y;
            x1 = x2 = x3+tilewidth;
            y4 = y2 = y3+tileheight;
            break;
      
          case -180:
          case 180:
            x4 = x2 = (map_x-halftilewidth)-view_x;
            y4 = y3 = (map_y-halftileheight)-view_y;
            x3 = x1 = x4+tilewidth;
            y2 = y1 = y4+tileheight;
            break;
      
          case -90:
          case 270:
            x2 = x1 = (map_x-halftilewidth)-view_x;
            y2 = y4 = (map_y-halftileheight)-view_y;
            x4 = x3 = x2+tilewidth;
            y1 = y3 = y2+tileheight;
            break;
      
          default:
          {
            float rcos = fcos((float)rotation * UTL_DEG2RAD);
            float rsin = fsin((float)rotation * UTL_DEG2RAD);
             
            // Fudge factor to fix gaps...
            float hwd = (view_rot?
                         (halftilewidth<64?
                          (float)halftilewidth*(scale*1.1):
                          (float)halftilewidth):
                         (float)halftilewidth);
            float hht = (view_rot?
                         (halftileheight<64?
                          (float)halftileheight*(scale*1.1):
                          (float)halftileheight):
                         (float)halftileheight);
   
            float rcn = rcos*-hwd;     /* pre-calculate values */
            float rcp = rcos*hwd;
            float rco = rcos*-hht;
            float rcq = rcos*hht;
            float rsn = rsin*-hwd;
            float rsp = rsin*hwd;
            float rso = rsin*-hht;
            float rsq = rsin*hht;
     
            float mid_x = map_x-view_x;
            float mid_y = map_y-view_y;
     
            x1 = mid_x+(rcn+rsq);
            y1 = mid_y-(rcq-rsn);
                                                
            x2 = mid_x+(rcp+rsq);
            y2 = mid_y-(rcq-rsp);
                                                
            x3 = mid_x+(rcn+rso);
            y3 = mid_y-(rco-rsn);
                                                
            x4 = mid_x+(rcp+rso);
            y4 = mid_y-(rco-rsp);
          }
          break;
        }
   
        if (view_scale != 1)
        {
          x1 *= view_scale;
          x2 *= view_scale;
          x3 *= view_scale;
          x4 *= view_scale;
          y1 *= view_scale;
          y2 *= view_scale;
          y3 *= view_scale;
          y4 *= view_scale;
        }
  
        pvr_prim(&sets[set].header, sizeof(pvr_poly_hdr_t));
  
        vert.flags = PVR_CMD_VERTEX;
        vert.x = x1;
        vert.y = y1;
        vert.z = z_order;
        vert.u = u1;
        vert.v = v1;
        //Lighting effects using argb on veritces?
        vert.argb = (((uint8)( fade * 255 ) ) << 24 ) | use_colour;
      
        vert.oargb = 0;
        pvr_prim(&vert, sizeof(vert));
      
        vert.x = x2;
        vert.y = y2;
        vert.u = u2;
        pvr_prim(&vert,sizeof(vert));
      
        vert.x = x3;
        vert.y = y3;
        vert.u = u1;
        vert.v = v2;
        pvr_prim(&vert,sizeof(vert));
      
        vert.flags = PVR_CMD_VERTEX_EOL;
        vert.x = x4;
        vert.y = y4;
        vert.u = u2;
        pvr_prim(&vert,sizeof(vert));
      }
      while(loop-->0);
    }
  }
  DBG_ELSE(99,("NO DRAWING REQUIRED"));
  
  DBG_EXIT(99,("til_render rc=%d",rc));

  return rc;
}

/* til_render_obj ___________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int til_render_obj(uint32 info, 
                   uint32 unused,
                   int32 map_x, 
                   int32 map_y, 
                   float z_order,
                   int32 rotation, 
                   float scale, 
                   float fade,
                   uint32 colour)
{
  return til_render((uint8)((info&0xFF000000)>>24),
                    (uint8)((info&0x00FF0000)>>16),
                    map_x, map_y,
                    z_order,
                    rotation,
                    scale,
                    fade,
                    colour);
}

/* til_getwidth _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 til_getwidth(uint32 info)
{
 uint32 rc;
 
 DBG_ENTRY(99,("til_getwidth(0x%08X)",(unsigned int)info));
 
 rc = (uint32)(sets[((info&0xFF000000)>>24)].halftilewidth*2);
 
 DBG_EXIT(99,("til_getwidth rc=%d",(int)rc));
 
 return rc;
}

/* til_getheight ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 til_getheight(uint32 info)
{
 uint32 rc;
 
 DBG_ENTRY(99,("til_getheight(0x%08X)",(unsigned int)info));
 
 rc = (uint32)(sets[((info&0xFF000000)>>24)].halftileheight*2);
 
 DBG_EXIT(99,("til_getheight rc=%d",(int)rc));
 
 return rc;
}

/* til_getmask ______________________________________________________________ */
/*                                                                            */
/* Get the mask area for the specified tile, rotated.                         */
/* __________________________________________________________________________ */

uint8 *til_getmask(uint32 info, int32 rot)
{
  uint8 set  = (uint8)((info&0xFF000000)>>24);
  uint8 tile = (uint8)((info&0x00FF0000)>>16);
  
  uint8 *mask = 0;
    
  DBG_ENTRY(99,("til_getmask(%d,%d)",(int)info,(int)rot));
  
  if (sets[set].mask)
  {
    #if defined(DEBUG_LEVEL)
    int   rcount         = 0;
    #endif
    int   tilewidth      = til_getwidth(info);
    int   tileheight     = til_getheight(info);
    float halftilewidth  = (float)tilewidth/2;
    float halftileheight = (float)tileheight/2;
    int   masksize       = UTL_MAX(tilewidth,tileheight);
    
    if (rot % 90)
    {
      masksize = fsqrt((masksize*masksize)<<1);
    }
    
    DBG_PRINT(99,("masksize: %d",masksize));
    
    /* Not using utl_malloc on purpose because of FILLVAL */
    if ((mask=utl_malloc(masksize*masksize*sizeof(uint8))))
    {
      float rcos = fcos((float)-rot * UTL_DEG2RAD);
      float rsin = fsin((float)-rot * UTL_DEG2RAD);
      
      float    x = -halftilewidth;
      float    y = -halftileheight;
      uint8 *mmp = sets[set].mask +                              /* base mask */
                   (tile%8)*tilewidth +                              /* x pos */
                   (tile/8)*tileheight*sets[set].texture_width;      /* y pos */

      DBG_PRINT(99,("MASK: %p mmp: %p",sets[set].mask,mmp));
      
      #define FILLVAL 0x00
      #define MODVAL  0x01
    
      memset(mask,FILLVAL,masksize*masksize*sizeof(uint8));
  
      if (rot == 0)
      {
        uint8 *nmp = mask;
    
        while(y<halftileheight)
        {
          while(x<halftilewidth)
          {
            *nmp++ = *mmp++;
            x+=1;
          }
    
          mmp = mmp - tilewidth + sets[set].texture_width; /* move down a line */
          x = -halftilewidth;
          y += 1;
        }
      }
      else /* Non-zero rotation */
      {
        uint8 *maskend = mask+(masksize*masksize);
        
        while(y<halftileheight)
        {
          DBG_PRINT(99,("XY: %f,%f",(double)x,(double)y));
    
          while(x<halftilewidth)
          {
            #if FILLVAL == 0x00
            if (*mmp++)
            #else
            if (!*mmp++)
            #endif
            {
              /* now we do the rotation and modify mask */
      
              int x1 = (int)(x*rcos+y*rsin);
              int y1 = (int)(y*rcos-x*rsin);
              uint8 *nmp = mask+
                          (x1+(masksize/2)) +               /* x position */
                          (y1+(masksize/2))*masksize;       /* y position */
                     
              DBG_PRINT(99,("%f,%f -> %d,%d (mask: %p nmp: %p)",(double)x,(double)y,x1,y1,mask,nmp));
      
              DBG_IF(99,((int32)nmp > (int32)maskend ||
                        x1<-(masksize/2) || x1 >(masksize/2) ||
                        y1<-(masksize/2) || y1 >(masksize/2) ||
                        (int32)nmp < (int32) mask),
                        ("MAJOR ERROR (%d) %f,%f (%fx%f) %d,%d %d,%d %d %p %p",
                        (int)((int32)mask + (masksize*masksize)-(int32)nmp),
                        (double)x,(double)y,(double)halftilewidth,(double)halftileheight,
                         x1,y1,(x1+(masksize/2)-1),(y1+(masksize/2)-1),(int)rot,mask,nmp));
        
              if (nmp > mask && nmp < maskend)
              {
                #if defined(DEBUG_LEVEL)
                if (*nmp != MODVAL)
                  rcount++;
                #endif
                
                *nmp = MODVAL;
              }
            }

            x += 1.0;
          }
      
          DBG_PRINT(99,("rcount: %d",rcount));
    
          mmp = mmp - tilewidth + sets[set].texture_width; /* start of next line */
          x = -halftilewidth;
          y += 1.0;
        }
      }    
    }
    DBG_ELSE(0,("malloc failed (MASKSIZE: %d)",masksize));
  }
  
  DBG_EXIT(99,("til_getmask rc=%p",mask));
  
  return mask;
}

#undef TIL_C
