#define IMG_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <png/png.h>

#include "utils.h"
#include "images.h"
#include "video.h"

img_t imgs[IMG_MAX_IMGS];                         /* static array of all imgs */

/* img_init _________________________________________________________________ */
/*                                                                            */
/* Initialise image support.                                                  */
/* __________________________________________________________________________ */

void img_init(void)
{
  DBG_ENTRY(99,("img_init()"));
  
  memset(imgs,0x00,sizeof(imgs));

  DBG_EXIT(99,("img_init"));
  
  return;
}

/* img_term _________________________________________________________________ */
/*                                                                            */
/* Terminate image support.                                                   */
/* __________________________________________________________________________ */

void img_term(void) {
  DBG_ENTRY(99,("img_term()"));
  
  DBG_EXIT(99,("img_term"));
  
  return;
}

/* img_load _________________________________________________________________ */
/*                                                                            */
/* Load an image.                                                             */
/* __________________________________________________________________________ */

uint8 img_load(uint8 img)
{
  uint8 rc = 1;
  
  char   file[256];
  
  uint32 w = 0;
  uint32 h = 0;
  
  int32  width = -1;
  int32  height = -1;
  
  DBG_ENTRY(99,("img_load(%d)",img));

  /* --------------- */
  /* Texture loading */
  /* --------------- */

  if (rc && !imgs[img].texture)
  {
    FILE  *fp;
    
    sprintf(file,"%s/gfx/%s%03d.%s",data_dir,IMG_DEF_PREFIX,img,IMG_DEF_FILETYPE);

    DBG_PRINT(99,("Attempting load of %s file (%s)",IMG_DEF_FILETYPE, file));

    /* Load information from an IMG file if there is one */
    
    if ((fp = fopen(file,"rb")))
    {
      char line[1024];
      
      DBG_PRINT(99,("%s file found - reading", IMG_DEF_FILETYPE));
      
   	  while (fgets(line,sizeof(line),fp))
      {   
        char *t = strtok(line, UTL_WHITESPACE);
        char *e;
      
        if (!t || *t == '#')
        {
          continue;
        }
        
        width = strtol(t,&e,10);
        
        if ((t = strtok(0," \n")))
          height = strtol(t, &e, 10);

        DBG_PRINT(99,("W1 width: %d height: %d", (int)width, (int)height));
      }
      
      fclose(fp);
    }
    
    sprintf(file,"%s/gfx/%s%03d.%s",data_dir,IMG_DEF_PREFIX,img,IMG_DEF_GFX_FILETYPE);
       
    DBG_PRINT(99,("png_load_texture for %s",file));
  
    if (0 == png_load_texture(file,
                              &imgs[img].texture,
                              PNG_FULL_ALPHA,
                              &w,
                              &h))
    {
      pvr_poly_cxt_t cxt;

      pvr_poly_cxt_txr(&cxt,
                       PVR_LIST_TR_POLY,
                       PVR_TXRFMT_ARGB4444,
                       w,h,
                       imgs[img].texture,
                       PVR_FILTER_BILINEAR);

      pvr_poly_compile(&imgs[img].header,
                       &cxt);
    }
    else
    {
      DBG_PRINT(99,("failed png_load_texture %d",img));
      rc = 0;
    }
  }
  DBG_ELSE(99,("not loading texture (tex=%p)",imgs[img].texture));
  
  if (rc && w)
  {
    /* If we were not told the width or height, use the image sizes */
    
    if (width == -1)
      imgs[img].width = w;
    else
      imgs[img].width = width;
  
    if (height == -1)
      imgs[img].height = h;
    else
      imgs[img].height = height;
      
    imgs[img].v2 = (float)imgs[img].height / (float)h;
    imgs[img].u2 = (float)imgs[img].width / (float)w;
  }

  imgs[img].loaded++;
  
  DBG_EXIT(99,("img_load rc=%d (loaded=%d)",rc,imgs[img].loaded));
  
  return rc;
}
/* img_unload _______________________________________________________________ */
/*                                                                            */
/* Clear up all resources associated with an image.                           */
/* __________________________________________________________________________ */

void img_unload(uint8 img)
{
  DBG_ENTRY(99,("img_unload(%d)",img));
 
  if (imgs[img].loaded > 0 && --imgs[img].loaded == 0)
  {
    DBG_PRINT(99,("actually unloading image %d",img));

    pvr_mem_free(imgs[img].texture); 

    memset(&imgs[img],0x00,sizeof(imgs[img]));
  }
  
  DBG_EXIT(99,("img_unload"));
  
  return;
}

/* img_obj_create_str _______________________________________________________ */
/*                                                                            */
/* Create an image reference given an input string from a file.               */
/* __________________________________________________________________________ */

uint32 img_obj_create_str(char *string)
{
  uint32 info = 0x00000001;
 
  uint8 img = 0;
  
  char *t;
  char *e;
  
  DBG_ENTRY(99,("img_obj_create_str(%s)",string));
   
  if ((t = strtok(string,UTL_WHITESPACE))) img=(uint8)strtol(t,&e,10);
  if (t==e) info = 0;
              
  /* make sure the image is loaded */
  if (!img_load(img)) info = 0;
              
  DBG_PRINT(99,("Image %d info:%d",img,(int)info));
                
  if (info)
  {
    /* info for an image is the image number + 1 */
    info += (uint32)img;
  }
  
  DBG_EXIT(99,("img_obj_create_str rc=0x%08X",(unsigned int)info));
  
  return info;
}

/* img_obj_free _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void img_obj_free(uint32 info)
{
  DBG_ENTRY(99,("img_obj_free(0x%08X)",(unsigned int)info));
  
  img_unload((uint8)(info-1));
  
  DBG_EXIT(99,("img_obj_free"));
}

/* img_inst_create __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 img_inst_create(uint32 info)
{
  uint32 inst = 0;
  
  DBG_ENTRY(99,("img_inst_create(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("img_inst_create rc=%d",(int)inst));
  
  return inst;
}

/* img_inst_free ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void img_inst_free(uint32 info)
{
  DBG_ENTRY(99,("img_inst_free(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("img_inst_free"));
}

/* img_render _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */


int img_render(uint32 img, 
               int32  map_x, 
               int32  map_y, 
               float  z_order,
               int32  rotation, 
               float  scale, 
               float  fade,
               uint32 colour)
{
  int rc = 1;
  
  if (vid_gun_screen)
    return rc;

  DBG_ENTRY(99,("img_render(%d,%d,%d,%f,%d,%f,%f,0x%06X)",
               (int)img,(int)map_x,(int)map_y,(double)z_order,
               (int)rotation,(double)scale,(double)fade,(unsigned int)colour));
  
  /* no drawing needed if no texture */
  if (imgs[img].texture) 
  { 
    int32 halfwidth = imgs[img].width / 2;
    int32 halfheight = imgs[img].height / 2;
    
    /* modify map_x and map_y positions */
    if (view_rot) 
    {
      float real_x = (float)map_x - view_mid_x;
      float real_y = (float)map_y - view_mid_y;
    
      map_x = (int32)(view_mid_x+((real_x*view_rcos)+(real_y*view_rsin)));
      map_y = (int32)(view_mid_y-((real_x*view_rsin)-(real_y*view_rcos)));
    }

    if ((map_x < (view_x-halfwidth))          ||  /* left of view */
        (map_y < (view_y-halfheight))         ||  /* above view */
        (map_x > (view_x+view_w+halfwidth))   ||  /* right of view */
        (map_y > (view_y+view_h+halfheight)))     /* below view */
   {
     rc = 0;
   }
   else
   {
     pvr_vertex_t vert; 
    
     int x1, y1, x2, y2, x3, y3, x4, y4;

     /* --------------------------------------------------------- */
     /* deal with rotation, simple cases first, then general case */
     /* --------------------------------------------------------- */
      
     rotation -= view_rot;

     switch(rotation) /* >360 = general case - callers beware! */
     {
       case 0:
         x1 = x3 = (map_x-halfwidth)-view_x;
         y1 = y2 = (map_y-halfheight)-view_y;
         x2 = x4 = x1+imgs[img].width;
         y3 = y4 = y1+imgs[img].height;
         break;
    
       case 90:
         x3 = x4 = (map_x-halfwidth)-view_x;
         y3 = y1 = (map_y-halfheight)-view_y;
         x1 = x2 = x3+imgs[img].width;
         y4 = y2 = y3+imgs[img].height;
         break;
    
       case 180:
         x4 = x2 = (map_x-halfwidth)-view_x;
         y4 = y3 = (map_y-halfheight)-view_y;
         x3 = x1 = x4+imgs[img].width;
         y2 = y1 = y4+imgs[img].height;
         break;
    
       case 270:
         x2 = x1 = (map_x-halfwidth)-view_x;
         y2 = y4 = (map_y-halfheight)-view_y;
         x4 = x3 = x2+imgs[img].width;
         y1 = y3 = y2+imgs[img].height;
         break;
    
       default:
         {
           float rcos = fcos((float)rotation * UTL_DEG2RAD);
           float rsin = fsin((float)rotation * UTL_DEG2RAD);
            // Fudge factor to fix gaps...
           float hwd = (view_rot?
                        (halfwidth<64?
                         (float)halfwidth*(scale*1.1):
                         (float)halfwidth):
                        (float)halfwidth);
           float hht = (view_rot?
                        (halfheight<64?
                         (float)halfheight*(scale*1.1):
                         (float)halfheight):
                        (float)halfheight);
 
           float rcn = rcos*-hwd;     /* pre-calculate values */
           float rcp = rcos*hwd;
           float rco = rcos*-hht;
           float rcq = rcos*hht;
           float rsn = rsin*-hwd;
           float rsp = rsin*hwd;
           float rso = rsin*-hht;
           float rsq = rsin*hht;
   
           float mid_x = map_x-view_x;
           float mid_y = map_y-view_y;
   
           x1 = mid_x+(rcn+rsq);
           y1 = mid_y-(rcq-rsn);
                                              
           x2 = mid_x+(rcp+rsq);
           y2 = mid_y-(rcq-rsp);
                                              
           x3 = mid_x+(rcn+rso);
           y3 = mid_y-(rco-rsn);
                                              
           x4 = mid_x+(rcp+rso);
           y4 = mid_y-(rco-rsp);
           
       }
       break;
     }
 
     if (view_scale != 1)
     {
       x1 *= view_scale;
       x2 *= view_scale;
       x3 *= view_scale;
       x4 *= view_scale;
       y1 *= view_scale;
       y2 *= view_scale;
       y3 *= view_scale;
       y4 *= view_scale;
     }

     DBG_PRINT(99,("a=%d,%d b=%d,%d c=%d,%d d=%d,%d", x1,y1,x2,y2,x3,y3,x4,y4));
     
     pvr_prim(&imgs[img].header, sizeof(pvr_poly_hdr_t));

      vert.flags = PVR_CMD_VERTEX;
      vert.x = x1;
      vert.y = y1;
      vert.z = z_order;
      vert.u = 0;
      vert.v = 0;
      //Lighting effects using argb on veritces?
      vert.argb = (((uint8)( fade * 255 ) ) << 24 ) | colour;;
    
      vert.oargb = 0;
      pvr_prim(&vert, sizeof(vert));
    
      vert.x = x2;
      vert.y = y2;
      vert.u = imgs[img].u2;
      pvr_prim(&vert,sizeof(vert));
    
      vert.x = x3;
      vert.y = y3;
      vert.u = 0;
      vert.v = imgs[img].v2;
      pvr_prim(&vert,sizeof(vert));
    
      vert.flags = PVR_CMD_VERTEX_EOL;
      vert.x = x4;
      vert.y = y4;
      vert.u = imgs[img].u2;
      pvr_prim(&vert,sizeof(vert));
    }
  }
  DBG_ELSE(99,("NO DRAWING REQUIRED"));
  
  DBG_EXIT(99,("img_render rc=%d",rc));

  return rc; /* <<< all done >>> */
}

/* img_render_obj ___________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int img_render_obj(uint32 info, 
                   uint32 unused,
                   int32 map_x, 
                   int32 map_y, 
                   float z_order,
                   int32 rotation, 
                   float scale, 
                   float fade,
                   uint32 colour)
{
  return(img_render(info-1,map_x,map_y,z_order,rotation,scale,fade,colour));
}

/* img_getwidth _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 img_getwidth(uint8 img)
{
 uint32 rc;
 
 DBG_ENTRY(99,("img_getwidth(%d)",(int)img));
 
 rc = imgs[img].width;
 
 DBG_EXIT(99,("img_getwidth rc=%d",(int)rc));
 
 return rc;
}

/* img_getwidth_obj _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 img_getwidth_obj(uint32 info)
{
 return(img_getwidth((uint8)info-1));
}


/* img_getheight ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 img_getheight(uint8 img)
{
 uint32 rc;
 
 DBG_ENTRY(99,("img_getheight(%d)",img));
 
 rc = imgs[img].height;
 
 DBG_EXIT(99,("img_getheight rc=%d",(int)rc));
 
 return rc;
}

/* img_getheight_obj ________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 img_getheight_obj(uint32 info)
{
 return(img_getheight((uint8)info-1)); 
}

#undef IMG_C
