#define SND_C

#include <kos.h>
#include <dc/spu.h>
#include <dc/sound/sound.h>
#include <stdlib.h>
#include <string.h>

#if defined(SND_BG_MOD)
#include <modplug/modplug.h>
#endif 

#if defined(SND_BG_MP3)
#include <sndmp3.h>
#include <sndserver.h>
#endif

#include "utils.h"
#include "sound.h"
#include "video.h"

#include <sound/arm/aica_cmd_iface.h>

/* Background music information */

BGTYPE bg_type[MAX_BG] = {BG_NONE};
void *bg_info[MAX_BG] = {0};

int curr_bg = -1;
int prev_bg = -1;

uint16 bg_buffer[65536] = {0};

kthread_t       *bg_thread;
semaphore_t     *bg_poke;
snd_stream_hnd_t bg_stream = -1;

/* Sound effects information */

sound_t fx_info[MAX_FX];
channel_t ch_info[CHANNELS];
int next_fx_channel = FIRST_FREE_FX_CHANNEL;

/* snd_bg_name_to_type ========================================== */
/*                                                                */
/* ============================================================== */

BGTYPE snd_bg_name_to_type(char *name)
{
  BGTYPE bg_type = BG_NONE;

  char *local_name = strdup(name);

  DBG_ENTRY(99,("snd_bg_name_to_type(%s)",name));

  if (local_name)
  {
    #if defined(K13)
    strupr(local_name);
    #else
    _strupr(local_name);
    #endif
    
    #if defined(SND_BG_MP3)
    if (bg_type == BG_NONE && strstr(local_name,".MP3"))
      bg_type = BG_MP3;
    #endif
    
    #if defined(SND_BG_OGG)
    if (bg_type == BG_NONE && strstr(local_name, ".OGG"))
      bg_type = BG_OGG;
    #endif
    
    #if defined(SND_BG_MOD)
    if (bg_type == BG_NONE && (strstr(local_name,".669") ||
          strstr(local_name,".AMF") ||
          strstr(local_name,".AMS") ||
          strstr(local_name,".DBM") ||
          strstr(local_name,".DMF") ||
          strstr(local_name,".DSM") ||
          strstr(local_name,".FAR") ||
          strstr(local_name,".IT")  ||
          strstr(local_name,".J2B") ||
          strstr(local_name,".MDL") ||
          strstr(local_name,".MED") || 
          strstr(local_name,".MOD") || 
          strstr(local_name,".MT2") || 
          strstr(local_name,".MTM") || 
          strstr(local_name,".OKT") || 
          strstr(local_name,".PSM") ||
          strstr(local_name,".PTM") ||
          strstr(local_name,".S3M") ||
          strstr(local_name,".STM") ||
          strstr(local_name,".ULT") ||
          strstr(local_name,".UMX") ||
          strstr(local_name,".WAV") ||
          strstr(local_name,".XM")))
      bg_type = BG_MOD;
    #endif
    
    free(local_name);
  }

  DBG_EXIT(99,("snd_bg_name_to_type rc=%d",bg_type));

  return bg_type;
}

/* snd_bg_callback ============================================== */
/*                                                                */
/* ============================================================== */

void *snd_bg_callback(snd_stream_hnd_t hnd, int len, int *actual)
{
  *actual = 0;
  
  if (curr_bg != -1)
  {
  	switch(bg_type[curr_bg])
  	{
  	  case BG_MOD:
        #if defined(SND_BG_MOD)
  	    *actual = ModPlug_Read((ModPlugFile *)bg_info[curr_bg],
  	                           bg_buffer,
  	                           len);
	
        if (*actual<len)
        {
          memset(&bg_buffer[*actual],0x00,len-*actual);
                                 
          /* This was an insert... so revert to prev */
          if (prev_bg != -1)
          {
          	curr_bg = prev_bg;
          	prev_bg = -1;
          }
          else /* otherwise... loop */
          {
            ModPlug_Seek((ModPlugFile *)bg_info[curr_bg],0);
          }
          
          *actual += ModPlug_Read((ModPlugFile *)bg_info[curr_bg],
                                  &bg_buffer[*actual],
                                  len-*actual);
        }
        #endif
        break;
      default:
        break;
  	}
  }

  /* Silence */
  if (*actual == 0)
  {
  	memset(bg_buffer,0x00,len);
    *actual = len;
  }
  	
  return bg_buffer;
}

/* snd_all_init ================================================= */
/*                                                                */
/* ============================================================== */

void snd_all_init(void)
{
  DBG_ENTRY(99,("snd_all_init()"));
  
  memset(bg_info,0x00,sizeof(bg_info));	
  
  memset(fx_info,0x00,sizeof(fx_info)); 
  
  //Fixing MP3 Init (not Chankast time anymore!)
  #if defined(SND_BG_MP3)
  mp3_init();
  #elif defined(SND_BG_OGG)
  sndoggvorbis_init();
  #else
  snd_init();
  snd_stream_init();
  #endif
  
  DBG_EXIT(99,("snd_all_init"));
  
  return;
}

/* snd_all_term ================================================= */
/*                                                                */
/* ============================================================== */

void snd_all_term(void)
{
  DBG_ENTRY(99,("snd_all_term()"));
  
  #if defined(SND_BG_MP3)
  mp3_shutdown();
  #elif defined(SND_BG_OGG)
  sndoggvorbis_shutdown();
  #else
  if (bg_stream != -1)
    snd_stream_destroy(bg_stream);

  bg_stream = -1;

  snd_stream_shutdown();
  snd_shutdown();
  #endif
  
  DBG_EXIT(99,("snd_all_term"));
  
  return;
}

/* snd_bg_load ============================================= */
/*                                                                */
/* ============================================================== */

int snd_bg_load(char *filename)
{
  int    bg_slot;

  DBG_ENTRY(99,("snd_bg_load(%s)",filename));
  
  for(bg_slot=0;bg_slot<MAX_BG;bg_slot++)
  {
  	if (bg_type[bg_slot] == BG_NONE)
  	  break;
  }
  
  if (bg_slot != MAX_BG)
  {
    uint32 hnd;
    unsigned char name[256];

    sprintf(name,"%s/sound/%s",data_dir,filename);

    DBG_PRINT(99,("Loading bg sound '%s'",name));

    if ((hnd = fs_open(name, O_RDONLY)) != -1)
    {
      BGTYPE bg_type_local = snd_bg_name_to_type(name);

      DBG_PRINT(99,("bg_type_local=%d",bg_type_local));

      switch(bg_type_local)
      {
        case BG_MOD:
        #if defined(SND_BG_MOD)
        {
          uint32 size = fs_total(hnd);
          uint8 *buffer = (uint8 *)utl_malloc(size);

          if (buffer)
          {
            if (fs_read(hnd,buffer,size) == size)
            {
 	            ModPlug_Settings settings;

              ModPlug_GetSettings(&settings);

              settings.mLoopCount = -1;
              settings.mResamplingMode = MODPLUG_RESAMPLE_LINEAR;
              settings.mFlags |= MODPLUG_ENABLE_OVERSAMPLING;
              settings.mBits = 16;

              ModPlug_SetSettings(&settings);

           	  bg_info[bg_slot] = ModPlug_Load(buffer,size);
   	        }
          }

          if (!bg_info[bg_slot])
          {
            utl_free(buffer); 
            buffer = 0;
            bg_slot = -1;
          }
        }
        #endif
        break;

        case BG_MP3:
        case BG_OGG:
          #if defined(SND_BG_MP3) || defined(SND_BG_OGG)
          bg_info[bg_slot] = strdup(name);
          #endif
          break;

        case BG_NONE:
        default:
          bg_slot = -1;
          break;
      }
      
      if (bg_slot != -1)
        bg_type[bg_slot] = bg_type_local;

      fs_close(hnd);
    }
    else
      bg_slot = -1;
  }
  else
  	bg_slot = -1;
  
  DBG_EXIT(99,("snd_bg_load rc=%d", bg_slot));
  
  return(bg_slot);
}

/* snd_bg_unload ================================================ */
/*                                                                */
/* ============================================================== */

void snd_bg_unload(int bg_slot)
{
  DBG_ENTRY(99,("snd_bg_unload(%d)", bg_slot));
  
  if (bg_slot >= 0 && bg_slot < MAX_BG)
  {
    if (bg_slot == curr_bg)
    {
      snd_bg_stop();
      curr_bg = -1;
    }

    switch(bg_type[bg_slot])
    {
      case BG_MOD:
        #if defined(SND_BG_MOD)
        ModPlug_Unload((ModPlugFile *)bg_info[bg_slot]);
        #endif
        break;
      case BG_MP3:
      case BG_OGG:
        #if defined(SND_BG_MP3) || defined(SND_BG_OGG)
        free(bg_info[bg_slot]);
        #endif
        break;
      default:
        break;
    }
    
    bg_info[bg_slot] = 0;
    bg_type[bg_slot] = BG_NONE;
  }

  DBG_EXIT(99,("snd_bg_unload"));
  
  return;
}

/* snd_bg_obj_create_str ____________________________________________________ */
/*                                                                            */
/* Create a background music given an input string.                           */
/* __________________________________________________________________________ */

uint32 snd_bg_obj_create_str(char *string)
{
  uint32 info = 0;
  int bg_slot;

  char *t = strtok(string,UTL_WHITESPACE);
   
  DBG_ENTRY(99,("snd_bg_obj_create_str(%s)",string));
   
  bg_slot = snd_bg_load(t);
              
  DBG_PRINT(99,("bg_slot=%d",bg_slot));
                
  info = bg_slot+1;

  DBG_EXIT(99,("snd_bg_obj_create_str rc=0x%08X",(unsigned int)info));
  
  return info;
}

/* snd_bg_obj_free __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void snd_bg_obj_free(uint32 info)
{
  int bg_slot = (int)info-1;
  
  DBG_ENTRY(99,("snd_bg_obj_free(0x%08X)",(unsigned int)info));
  
  snd_bg_unload(bg_slot);
  
  DBG_EXIT(99,("snd_bg_obj_free"));
}

/* snd_bg_thread ================================================ */
/*                                                                */
/* ============================================================== */

#if defined(SND_BG_USE_POLL_THREAD)
static void snd_bg_thread(void *v)
{
  while(1)
  {
    thd_sleep(100);
    
    snd_bg_poll();
  }
}
#endif

/* snd_bg_start ================================================= */
/*                                                                */
/* ============================================================== */
void* xing_callback(snd_stream_hnd_t hnd, int size, int * actual);

void snd_bg_start(int bg_slot)
{
  DBG_ENTRY(99,("snd_bg_start()"));
  
  snd_bg_stop();

  curr_bg = bg_slot;

  if (curr_bg != -1)
  {
    switch (bg_type[curr_bg])
    {
      case BG_MOD:
        #if defined(SND_BG_MOD)
        bg_stream = snd_stream_alloc((snd_stream_callback_t)snd_bg_callback, 16384);

        DBG_PRINT(99,("bg_stream=%d",bg_stream));
        
        snd_bg_modify(BG_MODIFY_SETVOLUME, snd_bg_volume, 0);
        snd_bg_modify(BG_MODIFY_RESTART, 0, 0);
        #endif
        break;
      case BG_MP3:
        #if defined(SND_BG_MP3)
        if (sndmp3_start(bg_info[curr_bg],1) >= 0)
        {
          bg_stream = snd_stream_alloc((snd_stream_callback_t)snd_bg_callback, 16384);

          snd_bg_modify(BG_MODIFY_SETVOLUME, snd_bg_volume, 0);
          snd_bg_modify(BG_MODIFY_RESTART, 0, 0);
        }
        else
          curr_bg = -1;
        #endif
        break;
      case BG_OGG:
        #if defined(SND_BG_OGG)
        sndoggvorbis_start(bg_info[curr_bg],1);
        snd_bg_modify(BG_MODIFY_SETVOLUME, snd_bg_volume, 0);
        #endif
        break;
      default:
        break;
    }
  }
  
  #if defined(SND_BG_USE_POLL_THREAD)
  bg_thread = thd_create(1,snd_bg_thread,0);
  #endif
  
  DBG_EXIT(99,("snd_bg_start"));
  
  return;
}

/* snd_bg_modify ================================================ */
/*                                                                */
/* ============================================================== */

void snd_bg_modify(int modifytype, int op1, int op2)
{
  DBG_ENTRY(99,("snd_bg_modify(%d, %d, %d)", modifytype, op1, op2));
 
  /* check operator */
  switch(modifytype)
  {
  	case BG_MODIFY_INSERT:
    case BG_MODIFY_SWITCH:
  	  if (op1 == -1 || bg_type[op1] == BG_NONE)
  	    return;
      break;
    case BG_MODIFY_PAUSE:
    case BG_MODIFY_RESTART:
    case BG_MODIFY_SETVOLUME:
      if (curr_bg == -1)
        return;
      break;
  }
  	    
  switch(modifytype)
  {
    case BG_MODIFY_INSERT: /* insert */
      prev_bg = curr_bg;
      break;
        
    case BG_MODIFY_SWITCH: /* switch */
      prev_bg = -1;
      break;
  }
    
  switch(modifytype)
  {
  	case BG_MODIFY_INSERT:
  	case BG_MODIFY_SWITCH:
      /* start new bg at the beginning */  
      switch(bg_type[op1])
      {
        case BG_MOD:
          #if defined(SND_BG_MOD)
          ModPlug_Seek((ModPlugFile *)bg_info[op1],0);
          #endif
          break;
        default:
          break;
      }

      snd_bg_start(op1);
      break;

    case BG_MODIFY_SETVOLUME:
    {
      switch(bg_type[curr_bg])
      {
        case BG_MP3:
        case BG_MOD:
          #if defined(SND_BG_MP3) || defined(SND_BG_MOD)
          if (bg_stream != -1)
            snd_stream_volume(bg_stream, op1);
          #endif
          break;
        case BG_OGG:
          #if defined(SND_BG_OGG)
          sndoggvorbis_volume(op1);
          #endif
          break;
        default:
          break;
      }
    }
    break;

    case BG_MODIFY_PAUSE:
    {
      switch(bg_type[curr_bg])
      {
        case BG_MP3:
        case BG_MOD:
          #if defined(SND_BG_MP3) || defined(SND_BG_MOD)
          if (bg_stream != -1)
            snd_stream_stop(bg_stream);
          #endif
          break;
        default:
          break;
      }
    }
    break;

    case BG_MODIFY_RESTART:
    {
      switch(bg_type[curr_bg])
      {
        case BG_MOD:
          #if defined(SND_BG_MOD)
          snd_stream_prefill(bg_stream);
          snd_stream_start(bg_stream, 44100, 1);
          #endif
          break;
        case BG_MP3:
          #if defined(SND_BG_MP3)
          snd_stream_prefill(bg_stream);
          snd_stream_start(bg_stream, 44100, 1);
          #endif
          break;
        default:
          break;
      }
    }
    break;
  }
  
  DBG_EXIT(99,("snd_bg_modify"));
}

/* snd_bg_poll ================================================== */
/*                                                                */
/* ============================================================== */

void snd_bg_poll(void)
{
  if (curr_bg != -1)
  {
    switch(bg_type[curr_bg])
    {
      case BG_MOD:
        #if defined(SND_BG_MOD)
        snd_stream_poll(bg_stream);
        #endif
        break;
      case BG_MP3:
        #if defined(SND_BG_MP3) 
        if (snd_stream_poll(bg_stream) < 0)
          sndmp3_start(bg_info[curr_bg],0);
        #endif
        break;
      default:
        break;
    }
  }

  return;
}

/* snd_bg_stop ================================================== */
/*                                                                */
/* ============================================================== */

void snd_bg_stop(void)
{
  DBG_ENTRY(99,("snd_bg_stop()"));
  
  //thd_destroy(bg_thread);
  
  if (curr_bg != -1)
  {
    switch(bg_type[curr_bg])
    {
      case BG_MP3:
        #if defined(SND_BG_MP3)
        sndmp3_stop();
        #endif
        break;
      case BG_OGG:
        #if defined(SND_BG_OGG)
        sndoggvorbis_stop();
        #endif
        break;
      default:
        break;
    }

    if (bg_stream != -1)
    {
      snd_stream_stop(bg_stream);
      snd_stream_destroy(bg_stream);
      bg_stream = -1;
    }

    curr_bg = -1;
  }
  
  DBG_EXIT(99,("snd_bg_stop"));
  
  return;
}

/* snd_fx_load ================================================== */
/*                                                                */
/* ============================================================== */

int snd_fx_load(char *filename)
{
  int fx_slot = 0;

  DBG_ENTRY(99,("snd_fx_load(%s)",filename));
  
  for(fx_slot=0;fx_slot<MAX_FX;fx_slot++)
  {
  	if (fx_info[fx_slot].left_addr == 0)
  	  break;
  }
  
  if (fx_slot != MAX_FX)
  {
  	uint32 hnd;
    unsigned char name[256];

    sprintf(name,"%s/sound/%s",data_dir,filename);

    DBG_PRINT(99,("Loading fx file '%s'",name));

    if ((hnd = fs_open(name, O_RDONLY)) != -1)
    {
      uint32 hz = 0;
      
      /* Check header */
      fs_seek(hnd,0x8,SEEK_SET);
      fs_read(hnd,&hz,4);      /* YUCK! */
      
      if (!strncmp((char *)&hz,"WAVE",4))
      { 
      	uint8 *buffer;
      	uint16 fmt,chn,bitsize;
      	uint32 len;
      	uint8  malloced = 0;
      	
        /* Read WAV header info */
        fs_seek(hnd, 0x14, SEEK_SET);
        fs_read(hnd, &fmt, 2);
        fs_read(hnd, &chn, 2);
        fs_read(hnd, &hz, 4);
        fs_seek(hnd, 0x22, SEEK_SET);
        fs_read(hnd, &bitsize, 2);
	    
      	/* Read WAV data */
        fs_seek(hnd, 0x28, SEEK_SET);
        fs_read(hnd, &len, 4);
      	
      	if (!(buffer=fs_mmap(hnd))) /* try fs_mmap */
      	{
      	  if ((buffer=utl_malloc(len)))
      	  {
      	  	fs_read(hnd,buffer,len);
      	  	malloced = 1;
      	  }
      	}
      	else
      	{
      	  buffer = buffer + fs_tell(hnd);
      	}
      	
      	/* Now we have a buffer pointing at the data */
      	if (buffer)
      	{
      	  fx_info[fx_slot].rate = hz;
          fx_info[fx_slot].stereo = chn-1;
     	  
      	  if (chn == 1) /* mono... */
      	  {
      	  	fx_info[fx_slot].len = len/2;
      	  	
            fx_info[fx_slot].left_addr  = snd_mem_malloc(len);
            fx_info[fx_slot].right_addr = 0;
     	    
      	  	spu_memload(fx_info[fx_slot].left_addr,buffer,len);
      	  	
      	  	if (fmt == 20)
      	  	{
      	  		fx_info[fx_slot].fmt = AICA_SM_ADPCM;
              fx_info[fx_slot].len *= 4;	/* 4-bit packed samples */
      	  	}
      	  	else
      	  	{
              fx_info[fx_slot].fmt = AICA_SM_16BIT;
      	  	}
      	  }
      	  else if (fmt == 1) /* stereo PCM */
      	  {
      	  	int i;
      	  	uint16 *sepbuf1 = (uint16 *)buffer;
            uint16 *sepbuf2;

            len = len/2;     /* stereo */
            
            fx_info[fx_slot].len = len/2; /* 16-bit */
            
            sepbuf2 = utl_malloc(len);
		    
            for (i=0; i<len; i+=2)
            {
              sepbuf2[i/2] = sepbuf1[i+1];
              sepbuf1[i/2] = sepbuf1[i];
            }
		    
            fx_info[fx_slot].left_addr  = snd_mem_malloc(len);
            fx_info[fx_slot].right_addr = snd_mem_malloc(len);

            spu_memload(fx_info[fx_slot].left_addr,sepbuf1,len);
            spu_memload(fx_info[fx_slot].right_addr,sepbuf2,len);
            
            fx_info[fx_slot].fmt = AICA_SM_16BIT;
            
            utl_free(sepbuf2);
      	  }
      	  else if (fmt == 20) /* stereo ADPCM */
      	  {
      	  	uint8 *buffer2 = utl_malloc(len/2);
      	  	
      	  	fx_info[fx_slot].len = len;
      	  	
      	  	len = len/2;
      	  	
      	  	memcpy(buffer2,buffer+len,len);
      	  	
            fx_info[fx_slot].left_addr  = snd_mem_malloc(len);
            fx_info[fx_slot].right_addr = snd_mem_malloc(len);

            spu_memload(fx_info[fx_slot].left_addr,buffer,len);
            spu_memload(fx_info[fx_slot].right_addr,buffer2,len);
            
      	  	fx_info[fx_slot].fmt = AICA_SM_ADPCM;
      	  	
      	  	utl_free(buffer2);
      	  }
      	  else
      	    fx_slot = -1;
      	    
      	  if (malloced)
      	    utl_free(buffer);
      	}
      	else
      	  fx_slot = -1;
      }
      
      fs_close(hnd); /* move to end */
    }
  }
  else
  {
  	fx_slot = -1;
  }

  DBG_EXIT(99,("snd_fx_load rc=%d", fx_slot));
    
  return(fx_slot);
}

/* snd_fx_unload ================================================ */
/*                                                                */
/* ============================================================== */

void snd_fx_unload(int fx_slot)
{
  DBG_ENTRY(99,("snd_fx_unload(%d)",fx_slot));
  
  if ((fx_slot != -1) && (fx_info[fx_slot].left_addr != 0))
  {
    snd_mem_free(fx_info[fx_slot].left_addr);
    snd_mem_free(fx_info[fx_slot].right_addr);

  	memset(&fx_info[fx_slot],0x00,sizeof(fx_info[fx_slot]));
  }
  
  DBG_EXIT(99,("snd_fx_unload"));
  
  return;
}

/* snd_fx_obj_create_str ____________________________________________________ */
/*                                                                            */
/* Create a sound effect given an input string.                               */
/* __________________________________________________________________________ */

uint32 snd_fx_obj_create_str(char *string)
{
  uint32 info = 0;
  int fx_slot;

  char *t = strtok(string,UTL_WHITESPACE);
   
  DBG_ENTRY(99,("snd_fx_obj_create_str(%s)",string));
   
  fx_slot = snd_fx_load(t);
              
  DBG_PRINT(99,("fx_slot=%d",fx_slot));
                
  if (fx_slot != -1)
  {
    int loop = -1,loopstart = -1,loopend = -1;
    
    info = fx_slot+1;
    
    if ((t = strtok(0,UTL_WHITESPACE))) loop = strtol(t,0,10);
    if ((t = strtok(0,UTL_WHITESPACE))) loopstart = strtol(t,0,10);
    if ((t = strtok(0,UTL_WHITESPACE))) loopend = strtol(t,0,10);
    
    if (loop == -1)
      loop = 0;
    if (loopstart == -1)
      loopstart = 0;
    if (loopend == -1)
      loopend = fx_info[fx_slot].len;
      
    fx_info[fx_slot].loop = loop;
    fx_info[fx_slot].loopstart = loopstart;
    fx_info[fx_slot].loopend = loopend;
  }
  else
    info = 0;
    
  DBG_EXIT(99,("snd_fx_obj_create_str rc=0x%08X",(unsigned int)info));
  
  return info;
}

/* snd_fx_obj_free __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void snd_fx_obj_free(uint32 info)
{
  int fx_slot = (int)info-1;
  
  DBG_ENTRY(99,("snd_fx_obj_free(0x%08X)",(unsigned int)info));
  
  snd_fx_unload(fx_slot);
  
  DBG_EXIT(99,("snd_fx_obj_free"));
}

/* snd_fx_inst_create _______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 snd_fx_inst_create(uint32 info)
{
  uint32 inst = 0;
  
  DBG_ENTRY(99,("snd_fx_inst_create(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("snd_fx_inst_create rc=%u",(unsigned int)inst));
  
  return inst;
}

/* snd_fx_inst_free _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void snd_fx_inst_free(uint32 info)
{
  DBG_ENTRY(99,("snd_fx_inst_free(0x%08X)",(unsigned int)info));
  
  DBG_EXIT(99,("snd_fx_inst_free"));
}


/* snd_fx_render ================================================ */
/*                                                                */
/* ============================================================== */

int snd_fx_render(uint32 fx_num, uint32 class, int32 map_x, int32 map_y)
{
  int fx_slot = fx_num - 1;
  /* pick any channel*/
  int channel = -1;
  /* and don't lock it */
  int lock_for = FX_NOLOCK;
  /* max volume */
  int volume = snd_fx_volume;
  /* pan to middle - calculate in a moment */
  int pan = 128;
    
  DBG_ENTRY(99,("snd_fx_render(%u,%u,%d,%d)",
                (unsigned int)fx_num, (unsigned int)class,
                (int)map_x, (int)map_y));
  
  if ((fx_slot != -1) && (fx_info[fx_slot].left_addr != 0))
  {
  	int size = fx_info[fx_slot].len;

    uint32 now = GET_TIME_SECONDS; /* Get current time */
  	
  	AICA_CMDSTR_CHANNEL(tmp, cmd, chan);
  	
  	if (size >= 65535)
  	  size = 65534;

    /* Find channel(s) that is not locked */
  	
  	if (channel == -1)
  	{
      int cur_channel;
      
  	  if (next_fx_channel > 63-fx_info[fx_slot].stereo)
  	    next_fx_channel = FIRST_FREE_FX_CHANNEL;
  	    
  	  cur_channel = next_fx_channel;
     
      while(channel == -1)
      {
        if ((ch_info[cur_channel].lock_until == 0 ||
             now > ch_info[cur_channel].lock_until) &&
            (fx_info[fx_slot].stereo == 0 ||
             (ch_info[cur_channel+1].lock_until == 0 ||
             now > ch_info[cur_channel+1].lock_until)))
        {
          channel = cur_channel;
          
          //printf("ch_info[%d].lock_until=%d\n",cur_channel,ch_info[cur_channel].lock_until);
          
          ch_info[channel].lock_until = (lock_for == FX_NOLOCK)?0:now+lock_for;
          
          next_fx_channel = channel+1;
          
          if (fx_info[fx_slot].stereo)
          {
            ch_info[channel+1].lock_until = ch_info[channel].lock_until;
            
            next_fx_channel++;
          }
            
        }
  	    else
        {
          cur_channel++;
          
          if (cur_channel == next_fx_channel)
          {
            //printf("NO FREE CHANNEL\n");
            return -1;
          }
          else if (cur_channel > 63-fx_info[fx_slot].stereo)
            cur_channel = FIRST_FREE_FX_CHANNEL;
        }
      }
  	}

  	if (!fx_info[fx_slot].stereo)
  	{
      if (auto_pan)
      {      
        /* calculate pan based on map_x, view_x and view_w!!! */
        float pos = (256.0 / (float)view_w) * (float)(map_x-view_x);

        if (pos<0)
          pan = 0;
        if (pos>255)
          pan = 255;
        else
          pan = (int)pos;
      }
      else 
        pan = 128;
        
  	  cmd->cmd = AICA_CMD_CHAN;
  	  cmd->timestamp = 0;
      cmd->size = AICA_CMDSTR_CHANNEL_SIZE;
      cmd->cmd_id = channel;
      chan->cmd = AICA_CH_CMD_START;
      chan->base = fx_info[fx_slot].left_addr;
      chan->type = fx_info[fx_slot].fmt;
      chan->length = size;
      chan->loop = fx_info[fx_slot].loop;
      chan->loopstart = fx_info[fx_slot].loopstart;
      chan->loopend = fx_info[fx_slot].loopend;
      chan->freq = fx_info[fx_slot].rate;
      chan->vol = volume;
      chan->pan = pan;
      snd_sh4_to_aica(tmp, cmd->size);
    }
    else 
    {
      cmd->cmd = AICA_CMD_CHAN;
      cmd->timestamp = 0;
      cmd->size = AICA_CMDSTR_CHANNEL_SIZE;
      cmd->cmd_id = channel;
      chan->cmd = AICA_CH_CMD_START;
      chan->base = fx_info[fx_slot].left_addr;
      chan->type = fx_info[fx_slot].fmt;
      chan->length = size;
      chan->loop = fx_info[fx_slot].loop;
      chan->loopstart = fx_info[fx_slot].loopstart;
      chan->loopend = fx_info[fx_slot].loopend;
      chan->freq = fx_info[fx_slot].rate;
      chan->vol = volume;
      chan->pan = 0;

      snd_sh4_to_aica_stop();
      snd_sh4_to_aica(tmp, cmd->size);

  	  cmd->cmd_id = channel+1;
      chan->base = fx_info[fx_slot].right_addr;
      chan->pan = 255;

      snd_sh4_to_aica(tmp, cmd->size);
      snd_sh4_to_aica_start();
    }
  }

  DBG_EXIT(99,("snd_fx_render"));

  return channel;
}
