#ifndef OBJ_H
#define OBJ_H

/* -------------------------------------------------------------------- */
/* Uncomment the following for use of real_x and real_y in calculations */
/* -------------------------------------------------------------------- */
#define OBJ_ACCURATE_MOVEMENT

typedef struct obj_info
{
  uint32 class;
  uint32 var;
  uint32 flags;
  uint8 type;
  int    wid;
  int    hgt;
  uint8 *mask;
  int8   hits_needed;
}
obj_info_t;

#define OBJ_INFO_FLAG_COPY 0x00000001   /* this obj_info is a copy of another */

typedef struct obj_inst                                 /* instance structure */
{
  struct obj_inst *next;                             /* next instance in list */
  struct obj_inst *prev;                             /* prev instance in list */
  int obj;                                          /* object this represents */
  uint32 var;                                           /* instance variables */
  uint8 *mask;                                      /* current collision mask */
  #ifdef OBJ_ACCURATE_MOVEMENT
  float real_x;                                            /* real x position */
  float real_y;                                            /* real y position */
  #endif
  int32 init_x;                                         /* initial x position */
  int32 x;                                                      /* x position */
  int32 init_y;                                         /* initial y position */
  int32 y;                                                      /* y position */
  float init_z;                                         /* initial z position */
  float z;                                                      /* z position */
  int32 init_rot;                         /* initial rotation (DEG_INC based) */
  int32 rot;                                      /* rotation (DEG_INC based) */
  int32 init_dir;                        /* initial direction (DEG_INC based) */
  int32 dir;                                     /* direction (DEG_INC based) */
  int32 init_speed;                                          /* initial speed */
  int32 speed;                                                       /* speed */
  float init_scale;                                          /* initial scale */
  float scale;                                                       /* scale */
  float init_fade;                                            /* initial fade */
  float fade;                                                         /* fade */
  uint32 init_colour;                                       /* initial colour */
  uint32 colour;                              /* Colour to use when rendering */
  int8 hits_needed;                                    /* hits needed to kill */
  int32 eol;                                                   /* end-of-life */
  struct obj_inst *coll;                            /* instance collided with */
}
obj_inst_t;

#define OBJ_INITIAL_OBJS 300

#define OBJ_SPECIAL_START 1000
#define OBJ_SPECIAL_SCORE OBJ_SPECIAL_START + 1

#define OBJ_EXTENSION "odf"                         /* Object Definition File */
#define OBJ_INST_LIST_EXTENSION "oil"            /* Object Instance List File */

#define OBJ_NONE 0                                            /* Empty object */
#define OBJ_TILE   'T'                             /* A tile object (tiles.c) */
#define OBJ_IMAGE  'I'                    /* A static image object (images.c) */
#define OBJ_ANIM   'A'           /* A sprite (with associated animations etc) */
#define OBJ_SFX    'S'                                      /* A sound effect */
#define OBJ_COPY   'C'                                 /* Copy another object */
#define OBJ_MUSIC  'M'                                    /* Background music */
#define OBJ_FONT   'F'                                   /* Text as an object */

#define OBJ_COLL_BBONLY    0x00                    /* Bounding box collision only */
#define OBJ_COLL_YESNO     0x01      /* PPC but no detail 'am i hitting ANYTHING' */
#define OBJ_COLL_FULL      0x02                  /* PPC returning list of objects */

#define OBJ_COLL_POINT     0x04                   /* Or'd with others, point only */
#define OBJ_COLL_CHECKONE  0x08                                 /* Check only one */
#define OBJ_COLL_LOOKAHEAD 0x10                /* Look where the object *will* be */

#define OBJ_ATTR_DIRCHANGE 1      /* Attribute about the object following P1 */
#define OBJ_ATTR_SCORE     2          /* Standard score achieved for killing */

#define OBJ_MAX_LISTS   50

#if defined(OBJ_C)
obj_info_t *obj_infos = 0;
obj_inst_t *obj_lists[OBJ_MAX_LISTS];
#else 
extern obj_info_t *obj_infos;
extern obj_inst_t *obj_lists[OBJ_MAX_LISTS];
#endif

#define OBJ_LIST_BACKDROP           0
#define OBJ_LIST_PLAYER_START       1
#define OBJ_LIST_PLAYER_1           1
#define OBJ_LIST_PLAYER_2           2
#define OBJ_LIST_PLAYER_3           3
#define OBJ_LIST_PLAYER_4           4
#define OBJ_LIST_PLAYER_END         4
#define OBJ_LIST_SHOT_START         5
#define OBJ_LIST_SHOT_1             5
#define OBJ_LIST_SHOT_2             6
#define OBJ_LIST_SHOT_3             7
#define OBJ_LIST_SHOT_4             8
#define OBJ_LIST_SHOT_END           8
#define OBJ_LIST_BADDIES            9
#define OBJ_LIST_GENERATORS         10
#define OBJ_LIST_COLLECTABLES       11
#define OBJ_LIST_INCIDENTAL         12
#define OBJ_LIST_BOUNDARY_START     13
#define OBJ_LIST_BOUNDARY_1         13
#define OBJ_LIST_BOUNDARY_2         14
#define OBJ_LIST_BOUNDARY_3         15
#define OBJ_LIST_BOUNDARY_4         16
#define OBJ_LIST_BOUNDARY_5         17
#define OBJ_LIST_BOUNDARY_END       17

#define OBJ_LIST_UNIQUE_START       20
#define OBJ_LIST_TILES              20
#define OBJ_LIST_UNIQUE_END         49

#define OBJ_EVENT_SHOOT             0
#define OBJ_EVENT_HIT               1
#define OBJ_EVENT_DEAD              2
#define OBJ_EVENT_BOUNDARY_1        3
#define OBJ_EVENT_BOUNDARY_2        4
#define OBJ_EVENT_BOUNDARY_3        5
#define OBJ_EVENT_BOUNDARY_4        6
#define OBJ_EVENT_BOUNDARY_5        7

#define OBJ_EVENT_UNIQUE_START      10
#define OBJ_EVENT_REVIVE            10
#define OBJ_EVENT_UNIQUE_END        20

/* FUNCTIONS */

void obj_init(void);
void obj_term(void);

int  obj_load(char *obj_file);
void obj_unload(char *obj_file);

void obj_set(int num,
             int class,
             uint8 type,
             int8 hits_needed,
             uint32 var,
             uint32 flags,
             int wid,
             int hgt,
             uint8 *mask);

int    obj_getwidth(int num);
int    obj_getheight(int num);

void   obj_render(int obj, int x, int y, float z);
int    obj_inst_render(obj_inst_t *inst, int animate);
void   obj_inst_render_all(int animate);
void   obj_inst_move(obj_inst_t *inst, int reverse);
int    obj_inst_animate(obj_inst_t *inst);
void   obj_inst_reset(obj_inst_t *inst, int position);
int    obj_inst_getwidth(obj_inst_t *inst);
int    obj_inst_getheight(obj_inst_t *inst);
uint8 *obj_inst_getmask(obj_inst_t *inst);
void   obj_inst_setevent(obj_inst_t *inst, int event);
int32  obj_inst_getattribute(obj_inst_t *inst, int attribute);

obj_inst_t *obj_inst_create(int num,
                            int32 x,
                            int32 y,
                            float z,
                            int32 rot,
                            int32 dir,
                            int32 speed,
                            float scale,
                            float fade,
                            uint32 colour);

obj_inst_t *obj_inst_collide_fs(obj_inst_t *inst, obj_inst_t *list, uint32 coll_mode);
obj_inst_t *obj_inst_collide(obj_inst_t *inst, obj_inst_t *list, uint32 coll_mode);

obj_inst_t *obj_inst_add(obj_inst_t *list, obj_inst_t *inst);
void        obj_inst_add_default(obj_inst_t *inst);
obj_inst_t *obj_inst_add_simple(int num, int32 x, int32 y, float z);

obj_inst_t *obj_inst_remove(obj_inst_t *list, obj_inst_t *inst, int free);
void        obj_inst_remove_default(obj_inst_t *inst, int free);

void        obj_inst_free(obj_inst_t *inst);
void        obj_inst_free_all(void);

void obj_inst_list_load(char *file);
void obj_inst_list_save(char *file);

#endif
