#define JOGGLE_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <math.h>
#include <arch/timer.h>

#include "utils.h"
#include "video.h"
#include "cont.h"
#include "sound.h"
#include "object.h"
#include "font.h"
#include "tiles.h"
#include "images.h"
#include "joggle.h"
#include "dict.h"

KOS_INIT_FLAGS(INIT_DEFAULT);

#if defined(USE_ROMDISK)
extern uint8 romdisk[];
KOS_INIT_ROMDISK(romdisk);
#endif

int players = 0;
player_info_t player_info[MAPLE_PORT_COUNT];
cnt_info_t    cont[MAPLE_PORT_COUNT+1];
int sa_bars[16] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

/* round attributes _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int             grid_wid         = 0;
static int             grid_hgt         = 0;
grid_square_t        **grid             = 0;

static unsigned char  *round_name                             = 0;
static uint32          round_millis                           = 0;
static int             round_background_colour                = 0x000000;
static int             round_background_music                 = -1;
static uint32          round_player_colour[MAPLE_PORT_COUNT]  = {0xFFFF00, 0xFF9900, 0xCCFF99, 0x3366FF};
static int             round_player_arrow_x[MAPLE_PORT_COUNT] = {16, 464, 16, 464};
static int             round_player_arrow_y[MAPLE_PORT_COUNT] = {16, 464, 16, 464};
static uint32          round_fade_action                      = 0;
static uint8           round_tile_set                         = 0;

static int             round_objective_type                   = OBJ_MIN_SCORE;
static int             round_objective_count                  = 500;
static unsigned char   round_objective_value                  = ' ';

static uint32          round_bonuses                          = BON_NONE;
static unsigned char   round_sc_collectable                   = 0;
static unsigned char   round_fc_collectable                   = 0;

static unsigned char  *round_object_file                      = 0;
static int             round_item[MAX_ITEMS];

static joggword_info_t joggword_info[MAX_JOGGWORDS];
static int             round_joggwords                        = 0;

/* game attributes __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

fnt_info_t          clock_font;
fnt_info_t          font;
dict_t              basic_dictionary = 0;
unsigned char     **text_string      = 0;
unsigned char     **random_string    = 0;
unsigned char      *round_qualifier  = 0;
uint8               letter_tile_set  = 0;
int                 letter_value[26];
static int          last_hiscore     = 0;

static int            game_loaded                           = 0;
static uint32         game_background_colour                = 0x000000;
static int            game_background_music                 = -1;
static uint32         game_player_colour[MAPLE_PORT_COUNT]  = {0xFFFF00, 0xFF9900, 0xCCFF99, 0x3366FF};
static int            game_player_arrow_x[MAPLE_PORT_COUNT] = {16, 464, 16, 464};
static int            game_player_arrow_y[MAPLE_PORT_COUNT] = {16, 464, 16, 464};
static uint8          game_tile_set                         = 0;

static unsigned char *game_object_file                      = 0;
static int            game_item[MAX_ITEMS];

/* data_load ________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static void data_load(FILE *fp, unsigned char *message)
{
  char line[1024];
  //int  square;
  int  i = 0;

  DBG_ENTRY(99,("data_load(%p,%s)",fp,message));

  while (fgets(line,sizeof(line),fp) && check_win32())
  {
    char *t;
     
    DBG_PRINT(99,("'%s'",line));

    t = strtok(line, UTL_WHITESPACE);

    if (!t || *t == '#')
      continue;

    if (message)
    {
      vid_screen_begin_rgb(0x000000, 0);

      fnt_write_simple(&font,
                       message,
                       FNT_CENTRAL, FNT_CENTRAL,
                       1,
                       1.5,
                       1,
                       (i++/5) | 0x01000000);

      vid_screen_end();
    }

    switch(*t)
    {
      /* ------------ */
      /* RG SOUNDS: S */
      /* ------------ */
      case 'S':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* ------------- */
            /* BACKGROUND: B */
            /* ------------- */
            case 'B':
            {
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                if (game_loaded)
                  round_background_music = snd_bg_load(t);
                else
                  game_background_music = snd_bg_load(t);
              }
            }
            break;
          }
        }
        break;

      /* ------------- */
      /* RG COLOURS: C */
      /* ------------- */
      case 'C':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* ------------- */
            /* BACKGROUND: B */
            /* ------------- */
            case 'B':
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                if (game_loaded)
                  round_background_colour = (int)strtoul(t,0,0);
                else
                  game_background_colour = (int)strtoul(t,0,0);
              }                
              break;

            /* ------------ */
            /* PLAYERS: 1-4 */
            /* ------------ */
            case 'P':             
              t++;
              if (*t > '0' && *t <= '0' + MAPLE_PORT_COUNT)
              {
                int i = *t - '1';

                if ((t = strtok(0, UTL_WHITESPACE)))
                {
                  if (game_loaded)
                    round_player_colour[i] = (int)strtoul(t,0,0);
                  else
                    game_player_colour[i] = (int)strtoul(t,0,0);
                }
              }
              break;
          }
        }
        break;

      /* ----------------- */
      /* RG X POSITIONS: X */
      /* ----------------- */
      case 'X':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* -------------------- */
            /* PLAYER ARROWS: A1-A4 */
            /* -------------------- */
            default:
              if (*t++ == 'A' && (*t > '0' && *t <= '0' + MAPLE_PORT_COUNT))
              {
                int i = *t - '1';

                if ((t = strtok(0, UTL_WHITESPACE)))
                {
                  if (game_loaded)
                    round_player_arrow_x[i] = (int)strtoul(t,0,0);
                  else
                    game_player_arrow_x[i] = (int)strtoul(t,0,0);                
                }
              }
              break;
          }
        }
        break;

      /* ----------------- */
      /* RG Y POSITIONS: Y */
      /* ----------------- */
      case 'Y':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* -------------------- */
            /* PLAYER ARROWS: A1-A4 */
            /* -------------------- */
            default:
              if (*t++ == 'A' && (*t > '0' && *t <= '0' + MAPLE_PORT_COUNT))
              {
                int i = *t - '1';

                if ((t = strtok(0, UTL_WHITESPACE)))
                {
                  if (game_loaded)
                    round_player_arrow_y[i] = (int)strtoul(t,0,0);
                  else
                    game_player_arrow_y[i] = (int)strtoul(t,0,0);
                }
              }
              break;
          }
        }
        break;

      /* ---------------- */
      /* RG DICTIONARY: D */
      /* ---------------- */
      case 'D':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* ----------- */
            /* GO BASIC: B */
            /* ----------- */
            case 'B':
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                if (basic_dictionary)
                  dct_free(basic_dictionary);

                basic_dictionary = dct_load(t);              
              }
              break;
          }
        }
        break;

      /* ----------------- */
      /* RG RANDOM SETS: R */
      /* ----------------- */
      case 'R':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          int rst_num = (int)strtoul(t,0,0);

          if (rst_num < RST_MAX)
          {
            utl_free(random_string[rst_num]);
            random_string[rst_num] = strdup(strtok(0, UTL_WHITESPACE));
          }
          DBG_ELSE(99,("Random string number %d out of range (max=%d)", rst_num, RST_MAX));
        }
        break;

      /* ------------------- */
      /* RG LETTER VALUES: V */
      /* ------------------- */
      case 'V':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          int letter_num = 0;

          while(t && letter_num < 26)
          {
            letter_value[letter_num++] = (int)strtoul(t,0,0);

            t = strtok(0, UTL_WHITESPACE);
          }
        }
        break;

      /* ----------------- */
      /* RG OBJECT INFO: O */
      /* ----------------- */
      case 'O':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* ---- */
            /* FILE */
            /* ---- */
            case 'F':
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                if (obj_load(t) == 0)
                {
                  if (game_loaded)
                    round_object_file = strdup(t);
                  else
                    game_object_file = strdup(t);
                }
              }
              break;

            /* ------------ */
            /* ITEM MAPPING */
            /* ------------ */
            case 'I':
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                int item = (int)strtoul(t,0,10);

                if ((t = strtok(0, UTL_WHITESPACE)))
                {
                  if (game_loaded)
                    round_item[item] = (int)strtoul(t,0,10);
                  else
                    game_item[item] = (int)strtoul(t,0,10);
                }
              }
              break;

            /* ------------- */
            /* INSTANCE LIST */
            /* ------------- */
            case 'L':
              if ((t = strtok(0, UTL_WHITESPACE)))
                obj_inst_list_load(t);
              break;
          }
        }
        break;

      /* ------------------- */
      /* RG GAME TILE-SET: G */
      /* ------------------- */
      case 'Z':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          uint8 set_num = (uint8)strtoul(t,0,0);
          
          if (game_loaded)
            round_tile_set = set_num;
          else
            game_tile_set = set_num;

          til_set_load(set_num, TIL_LOAD_TEXTURE);
        }
        break;

      /* -------------- */
      /* RG PLAYAREA: P */
      /* -------------- */
      case 'P':
      {
        int w = VID_INIT_PLAYAREA_W;
        int h = VID_INIT_PLAYAREA_H;

        if ((t=strtok(0, UTL_WHITESPACE)))
          w = (int)strtol(t,0,0);
        if ((t=strtok(0, UTL_WHITESPACE)))
          h = (int)strtol(t,0,0);

        vid_playarea_set(w,h);
      }
      break;

      /* ---------- */
      /* RO NAME: N */
      /* ---------- */
      case 'N':
        round_name = strdup(strtok(t+strlen(t)+1,UTL_CRLF));
        break;

      /* ---------------- */
      /* RO DIMENSIONS: M */
      /* ---------------- */
      case 'M':
        if ((t = strtok(0, UTL_WHITESPACE)))
          grid_wid = (int)strtoul(t,0,0);
        if ((t = strtok(0, UTL_WHITESPACE)))
          grid_hgt = (int)strtoul(t,0,0);

        if ((grid = utl_malloc(grid_wid*grid_hgt*sizeof(grid_square_t *))))
          memset(grid,0x00,grid_wid*grid_hgt*sizeof(grid_square_t *));

        DBG_PRINT(99,("gw=%d gh=%d",grid_wid,grid_hgt));
        break;

      /* ------------------- */
      /* RO ENTRY IN GRID: E */
      /* ------------------- */
      case 'E':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          int square = (int)strtoul(t,0,10);

          if ((grid[square] = utl_malloc(sizeof(grid_square_t))))
          {
            memset(grid[square],0x00,sizeof(grid_square_t));
  
            /* ------------------ */
            /* calculate position */
            /* ------------------ */
  
            grid[square]->x = LEFT_MARGIN + (GRID_SIZE/2) + (GRID_SIZE*(square%grid_wid));
            grid[square]->y = TOP_MARGIN + (GRID_SIZE/2) + (GRID_SIZE*(square/grid_wid));
            
            if ((t = strtok(0, UTL_WHITESPACE)))
            {
              if (*t == 'r')
              {
                int rst_num = (int)strtoul(&t[1],0,0);
                int letter_num = (int)(RANDOM_NUMBER * (strlen(random_string[rst_num])-2));
  
                grid[square]->letter = random_string[rst_num][letter_num];
              }
              else
                grid[square]->letter = toupper(*t);
            }
  
            /* ------------------------------- */
            /* NOTE: overloaded use of 'owner' */
            /* ------------------------------- */
            if ((t = strtok(0, UTL_WHITESPACE)))
              grid[square]->owner = (int)strtoul(t,0,0);
  
            grid[square]->orig_square = square;

            if (round_objective_type & OBJ_FLAG_LAYOUT && (t = strtok(0, UTL_WHITESPACE)))
              if (*t == 'J')
                grid[square]->letter = BLANK_LETTER;
  
            DBG_PRINT(99,("sq=%d x=%d y=%d l=%c",(int)square,(int)grid[square]->x,(int)grid[square]->y,grid[square]->letter));
          }
        }
        break;

      /* --------------------- */
      /* RO ROUND OBJECTIVE: G */
      /* --------------------- */      
      case 'G':
        if ((t = strtok(0, UTL_WHITESPACE)))
          round_objective_type = (int)strtoul(t,0,0);

        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          round_millis = (int)strtoul(t,0,0);

          if (!(round_objective_type & OBJ_FLAG_INFINITE))
            round_millis *= 1000;
        }

        switch(round_objective_type & OBJ_TYPE_MASK)
        {
          default:
            if ((t = strtok(0, UTL_WHITESPACE)))
              round_objective_count = (int)strtoul(t,0,0);
            if ((t = strtok(0, UTL_WHITESPACE)))
              round_objective_value = *t;
            break;
        }
        break;

      /* ------------- */
      /* RO BONUSES: B */
      /* ------------- */
      case 'B':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          switch(*t)
          {
            /* ------ */
            /* LENGTH */
            /* ------ */
            case 'L':
            {
              int wm_num = 0;

              round_bonuses |= BON_WORD_LENGTH;

              while((t = strtok(0, UTL_WHITESPACE)))
                word_multiplier[wm_num++] = (int)strtoul(t,0,0);
            }
            break;

            /* ------------ */
            /* SAME COLOURS */
            /* ------------ */
            case 'S':
              round_bonuses |= BON_SAME_COLOUR;

              if ((t = strtok(0, UTL_WHITESPACE)))
                round_sc_collectable = *t;
              else
                round_sc_collectable = COL_MULT_TWO;

              break;

            /* --------------- */
            /* FIRST COMPLETER */
            /* --------------- */
            case 'F':
              round_bonuses |= BON_FIRST_COMPLETER;

              if ((t = strtok(0, UTL_WHITESPACE)))
                round_fc_collectable = *t;
              else
                round_fc_collectable = COL_MULT_TWO;

              break;

            /* ------------ */
            /* COLLECTABLES */
            /* ------------ */
            case 'C':
              if ((t = strtok(0, UTL_WHITESPACE)))
              {
                int collectables = (int)strtoul(t,0,0);

                if ((t = strtok(0, UTL_WHITESPACE)))
                {
                  for(i=0;i<collectables;i++)
                  {
                    int target = RANDOM_NUMBER * (grid_wid * grid_hgt);
                    int type = RANDOM_NUMBER * strlen(t);

                    if (!grid[target]->collectable)
                      grid[target]->collectable = t[type];
                  }
                }
              }
              break;
          }
        }
        break;

      /* ------------------- */
      /* RO JOGGWORD CLUE: J */
      /* ------------------- */
      case 'J':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {                                   
          int joggword = (int)strtoul(t,0,10);

          if ((t = strtok(0, UTL_WHITESPACE)))
          {
            joggword_info[joggword].length = (int)strtoul(t,0,10);

            for(i=0; i<joggword_info[joggword].length; i++)
              if ((t = strtok(0, UTL_WHITESPACE)))
                joggword_info[joggword].chain[i] = (int)strtoul(t,0,10);

            t += strlen(t)+1;

            while(*t == ' ')
              t++;

            joggword_info[joggword].clue = strdup(strtok(t,UTL_CRLF));

            if (joggword >= round_joggwords)
              round_joggwords = joggword+1;
          }
        }
        break;

      /* --------------- */
      /* GO TEXT FONT: F */
      /* --------------- */
      case 'F':
        if ((t = strtok(0, UTL_WHITESPACE)))
          fnt_load(t,&font);
        break;

      /* ---------------- */
      /* GO CLOCK FONT: K */
      /* ---------------- */
      case 'K':
        if ((t = strtok(0, UTL_WHITESPACE)))
          fnt_load(t,&clock_font);
        break;

      /* ------------------ */
      /* GO TEXT STRINGS: T */
      /* ------------------ */
      case 'T':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          int txt_num = (int)strtoul(t,0,0);

          if (txt_num < TXT_MAX)
          {
            utl_free(text_string[txt_num]);
            text_string[txt_num] = strdup(strtok(t+strlen(t)+1, UTL_CRLF));
          }            
          DBG_ELSE(99,("Text string number %d out of range (max=%d)", txt_num, TXT_MAX));
        }
        break;

      /* --------------------- */
      /* GO LETTER TILE-SET: L */
      /* --------------------- */
      case 'L':
        if ((t = strtok(0, UTL_WHITESPACE)))
        {
          uint8 set_num = (uint8)strtoul(t,0,0);

          letter_tile_set = set_num;

          til_set_load(letter_tile_set, TIL_LOAD_TEXTURE);
        }
        break;

      /* --------------------- */
      /* GO ROUND QUALIFIER: Q */
      /* --------------------- */
      case 'Q':
        if ((t = strtok(0, UTL_WHITESPACE)))
          round_qualifier = strdup(t);
        break;

      default:
        break;
    }
  }

  DBG_EXIT(99,("data_load"));
}

/* round_load _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void round_load(FILE *fp, int round, fnt_info_t *font)
{
  int i;
  unsigned char local_round_name[50];

  DBG_ENTRY(99,("round_load(%p)", fp));
 
  SEED_RANDOM;

  round_background_colour = game_background_colour;
  round_background_music  = game_background_music;
  round_fade_action       = 0;
  round_tile_set          = game_tile_set;
  round_object_file       = 0;
  round_bonuses           = BON_NONE;
  round_sc_collectable    = COL_NONE;
  round_fc_collectable    = COL_NONE;

  for(i=0;i<players;i++)
  {
    round_player_arrow_x[i] = game_player_arrow_x[i];
    round_player_arrow_y[i] = game_player_arrow_y[i];
    round_player_colour[i] = game_player_colour[i];
  }

  round_joggwords = 0;
  memset(&joggword_info,0x00,sizeof(joggword_info));

  memcpy(&round_item, &game_item, sizeof(round_item));

  if (round)
  {
    sprintf(local_round_name,text_string[TXT_ROUND_NUM], round);
    data_load(fp, local_round_name);
  }
  else
    data_load(fp, 0);

  /* --------------------------------------------- */
  /* Set round_fade action based on objective type */
  /* --------------------------------------------- */
  switch(round_objective_type & OBJ_DROP_MASK)
  {
    case OBJ_REPLACE_LETTERS:
      round_fade_action |= ACT_REPLACE;
      break;
    case OBJ_LEAVE_LETTERS:
      round_fade_action |= ACT_LEAVE;
      break;
  }

  if (round_objective_type & OBJ_FLAG_RANDOM_DELAY)
    round_fade_action |= ACT_RANDOM_DELAY;  

  DBG_EXIT(99,("round_load"));
}

/* round_unload _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void round_unload(void)
{
  int square;

  DBG_ENTRY(99,("round_unload()"));
 
  for(square=0;square<grid_wid*grid_hgt;square++)
    utl_free(grid[square]);

  for(square=0;square<MAX_JOGGWORDS;square++)
    utl_free(joggword_info[square].clue);

  utl_free(grid);
  grid = 0;

  utl_free(round_name);
  round_name = 0;

  if (round_tile_set != game_tile_set)
    til_set_unload(round_tile_set);

  if (round_object_file)
  {
    obj_unload(round_object_file);
    free(round_object_file);
    round_object_file = 0;
  }

  if (round_background_music != game_background_music)
    snd_bg_unload(round_background_music);

  memset(&round_item, 0x00, sizeof(round_item));

  DBG_EXIT(99,("round_unload"));
}

/* get_millis _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 get_millis(void)
{
  uint32 s, m;

  timer_ms_gettime(&s,&m);

  return (s*1000)+m;
}

/* collect __________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 collect(unsigned char collectable, player_info_t *player, int *word_score)
{
  uint32 rc = (uint32)collectable;

  DBG_ENTRY(99, ("collect('%c',%p,%p)", collectable?collectable:' ', player, word_score));

  switch(collectable)
  {
    case COL_MULT_TWO:
      *word_score = *word_score * 2;
      break;
    case COL_MULT_THREE:
      *word_score = *word_score * 3;
      break;
    default:
      rc = 0;
      break;
  }

  /* -------------------------------------------- */
  /* Can do something here to indicate collection */
  /* -------------------------------------------- */

  DBG_EXIT(99,("collect rc=%d",(int)rc));

  return rc;
}

/* hiscore_name _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void hiscore_name(int p, char *name, fnt_info_t *font)
{
  int i;
  int initial = 0;
  int namelen = strlen(name);

  int x,y;

  int yf[10] = {0,0,0,0,0,0,0,0,0,0};

  int xd = (int)(1 * (float)til_getwidth(round_tile_set));
  int yd = (int)(1 * (float)til_getheight(round_tile_set));

  char temp_string[50];

  memset(name, 'A', namelen);

  obj_inst_add_simple(round_item[ITM_TITLE_BACKDROP_BOARD],
                      view_x + (512/2), view_mid_y, 2);
  obj_inst_add_simple(round_item[ITM_TITLE_BACKDROP_STATUS],
                      (view_x+view_w) - (128/2), view_mid_y, 2);

  player_info[p].colour        = round_player_colour[p];
  player_info[p].cur_delay     = 0;
  player_info[p].inactivity    = 0;

  while(check_win32())
  {
    cnt_state_all((cnt_info_t *)&cont);

    if (!player_info[p].cur_delay)
    {
      if (cont[player_info[p].cont].y < -64)
        cont[player_info[p].cont].buttons |= CONT_DPAD_UP;
      else if (cont[player_info[p].cont].y > 64)
        cont[player_info[p].cont].buttons |= CONT_DPAD_DOWN;
      else if (cont[player_info[p].cont].x > 64)
        cont[player_info[p].cont].buttons |= CONT_DPAD_RIGHT;
      else if (cont[player_info[p].cont].x < -64)
        cont[player_info[p].cont].buttons |= CONT_DPAD_LEFT;

      if (cont[player_info[p].cont].buttons & 0x00F0)
      {
        if (cont[player_info[p].cont].buttons & CONT_DPAD_UP)
        {
          name[initial] = name[initial]+1;
          yf[initial] = 0;
        }
        else if (cont[player_info[p].cont].buttons & CONT_DPAD_DOWN)
        {
          name[initial] = name[initial]-1;
          yf[initial] = 0;
        }
        else if (cont[player_info[p].cont].buttons & CONT_DPAD_LEFT)
          initial -= 1;
        else if (cont[player_info[p].cont].buttons & CONT_DPAD_RIGHT)
          initial += 1;

        player_info[p].cur_delay = 6;
        player_info[p].inactivity = 0;
      }
      else
        player_info[p].inactivity++;

      if (initial < 0)
        initial = 0;
      else if (initial >= namelen)
        initial = namelen-1;

      if (name[initial] > 'Z')
        name[initial] = 'A';
      else if (name[initial] < 'A')
        name[initial] = 'Z';

      if (cont[player_info[p].cont].buttons & CONT_B)
        break;
    }
    else
      player_info[p].cur_delay--;

    vid_screen_begin_rgb(round_background_colour,0);

    #if defined(_WIN32)
    obj_inst_render_all(1);
    #endif

    fnt_write_simple(font,
                     round_name,
                     FNT_CENTRAL, 30,
                     10,
                     2,
                     1,
                     0xFFFFFF);

    sprintf(temp_string, text_string[TXT_GOT_HIGH_SCORE], p+1);

    fnt_write_simple(font,
                     temp_string,
                     FNT_CENTRAL, 150,
                     10,
                     0.75,
                     1,
                     round_player_colour[p]);

    fnt_write_simple(font,
                     text_string[TXT_ENTER_NAME],
                     FNT_CENTRAL, 180,
                     10,
                     0.75,
                     1,
                     0xFFFFFF);
    
    y = 250;

    x = view_mid_x - (namelen*xd)/2;

    /* ----------------- */
    /* Draw the initials */
    /* ----------------- */
    for(i=0; i<namelen; i++)
    {
      til_render(game_tile_set,
                 0,
                 x, y+yf[i],
                 14,
                 0,
                 1,
                 1,
                 hiscore_colours[name[i]%HISCORE_COLOURS]);

      til_render(letter_tile_set, 
                 (uint8)(name[i]-'A'),
                 x, y+yf[i],
                 15,
                 0,
                 1,
                 1,
                 0xFFFFFF);

      if (yf[i] > 0)
        yf[i]--;

      if (i == initial)
      {
        int rot = 0;
        float scale = 1;

        /* --------------------------------- */
        /* JIGGLE player marker if inactive! */
        /* --------------------------------- */
        if ((player_info[p].inactivity/50)%5 == 4)
        {
          rot = (int)(RANDOM_NUMBER * 20)-10;
          scale += RANDOM_NUMBER;
        }

        til_render(round_tile_set,
                   (uint8)(2+p),
                   x, y,
                   16,
                   rot,
                   scale,
                   1,
                   round_player_colour[p]);

      }

      x+=xd;
    }

    for(x=0; x<16; x++)
    {
      i = sa_bars[x] * 2;

      vid_box(20+x*40,
              y,
              36,
              i,
              0.8,
              1,
              utl_fire_palette[UTL_FIRE_PALETTE_SIZE-i]);
    }
    
    y += yd + 20;

    sprintf(temp_string, "%s: %d %s: %d %s: %d", 
        text_string[TXT_SCORE],
        player_info[p].score,
        text_string[TXT_WORD_SCORE],
        player_info[p].hi_word_score,
        text_string[TXT_WORD_LENGTH],
        player_info[p].hi_word_length);

    fnt_write_simple(font,
                     temp_string,
                     FNT_CENTRAL, y,
                     10,
                     0.5,
                     1,
                     0xFF6535);

    #if !defined(_WIN32)
    obj_inst_render_all(1);
    #endif
    
    snd_bg_poll();

    vid_screen_end();
  }

  return;
}

/* start_round ______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int start_round(FILE *fp, int round, fnt_info_t *font)
{ 
  activity_t activity = RND_ACT_SHOW_OBJECTIVE;
  activity_t prev_activity = activity;

  int           cur_joggword = 0;
  unsigned char layout_chain[50];
  int           layout_chain_pos = 0;
  fnt_info_t *cfont = &clock_font;
  uint32 start_time = get_millis();
  uint32 curr_time;
  uint32 end_time;
  uint32 sec;
  uint32 prev_sec;
  char   temp_string[50];
  float  time_scale = 2.5;

  uint32 tiles_left = 0;
  uint32 players_left = 0;
  uint32 tiles_left_value = 0;

  float darken = 0;
  int   i,j = 0;

  int new_round = round;

  round_load(fp, round, font);

  end_time = start_time + round_millis;
  curr_time = start_time;

  prev_sec = (end_time-curr_time)/1000;

  /* ----------------------------- */
  /* Set the player initial square */
  /* ----------------------------- */

  switch(players)
  {
    case 1:
      player_info[0].square = ((grid_hgt/2)*grid_wid)+(grid_wid/2);
      break;
    case 2:
      player_info[0].square = ((grid_hgt/4)*grid_wid)+(grid_wid/4);
      player_info[1].square = (((grid_hgt*3)/4)*grid_wid)+((grid_wid*3)/4);
      break;
    case 3:
      player_info[0].square = ((grid_hgt/4)*grid_wid)+(grid_wid/4);
      player_info[1].square = ((grid_hgt/4)*grid_wid)+((grid_wid*3)/4);
      player_info[2].square = (((grid_hgt*3)/4)*grid_wid)+(grid_wid/2);
      break;
    case 4:
      player_info[0].square = ((grid_hgt/4)*grid_wid)+(grid_wid/4);
      player_info[1].square = ((grid_hgt/4)*grid_wid)+((grid_wid*3)/4);
      player_info[2].square = (((grid_hgt*3)/4)*grid_wid)+(grid_wid/4);
      player_info[3].square = (((grid_hgt*3)/4)*grid_wid)+((grid_wid*3)/4);
      break;
  }

  /* ------------------------------- */
  /* For each grid square add a tile */
  /* ------------------------------- */
  for(i=(grid_wid*grid_hgt)-1; i>=0; i--)
  {
    grid[i]->tile = obj_inst_add_simple(round_item[ITM_TILE],
                                        grid[i]->x, grid[i]->y, GRID_Z);

    if (grid[i]->tile)
    {
      grid[i]->tile->colour = grid[i]->tile->init_colour = grid[i]->owner | VID_BLACK_BORDER;

      obj_inst_animate(grid[i]->tile);
    }

    grid[i]->owner = 0;
  }

  for(i=0; i<players; i++)
  {
    if (player_info[i].type >= GOING)
    {
      player_info[i].type = GONE;
    }
    else
    {
      if (player_info[i].type == RETIRED)
        player_info[i].type = HUMAN;

      player_info[i].colour          = round_player_colour[i];
      player_info[i].cur_delay       = 0;
      memset(&player_info[i].chain,0xFF,sizeof(player_info[i].chain));
      player_info[i].chain_pos       = 0;
      player_info[i].x               = grid[player_info[i].square]->x;
      player_info[i].y               = grid[player_info[i].square]->y;
      player_info[i].objective_count = 0;
      player_info[i].word[0]         = 0;
      player_info[i].inactivity      = 0;

      if (round_item[ITM_STATUS_AREA])
      {
        player_info[i].status_area = obj_inst_add_simple(round_item[ITM_STATUS_AREA],
                                                         STATUS_AREA_X, STATUS_AREA_Y + (74*i), 2);
      }
    }
  }

  for(i=0;i<CLOCK_SEGMENTS;i++)
    clock_info[i].colour = i;

  obj_inst_add_simple(round_item[ITM_INGAME_BACKDROP_BOARD],
                      view_x + (512/2), view_mid_y, 1);
  obj_inst_add_simple(round_item[ITM_INGAME_BACKDROP_STATUS],
                      (view_x+view_w) - (128/2), view_mid_y, 1);

  /* -------------------------------------------------------- */
  /* Make sure the 'clock' font for infinite include letters! */
  /* -------------------------------------------------------- */
  if (round_objective_type & OBJ_FLAG_INFINITE)
    cfont = font;

  if (round_objective_type & OBJ_FLAG_LAYOUT)
    cur_joggword = round_joggwords;

  while(new_round == round  && check_win32())
  {
    cnt_state_all((cnt_info_t *)&cont);

    vid_screen_begin_rgb(round_background_colour,0);

    /* ---------------------------------------------------- */
    /* What we do here depends on the current game activity */
    /* ---------------------------------------------------- */
    switch(activity)
    {
      /* Show Objective ------------------------------------------------------ */

      case RND_ACT_SHOW_OBJECTIVE:
      {
        unsigned char *string;

        darken = 0.5;

        for(i=0;i < players;i++)
        {
          if (player_info[i].type == HUMAN)
          {
            if (cont[player_info[i].cont].buttons & CONT_START && 
              !(cont[player_info[i].cont].prev_buttons & CONT_START))
            {
              activity = RND_ACT_PLAY;
              start_time = get_millis();
              end_time = start_time + round_millis;
              curr_time = start_time;
            }
          }
        }

        fnt_write_simple(font,
                         round_name,
                         FNT_CENTRAL, 20,
                         10,
                         2,
                         1,
                         0xFFFFFF);

        fnt_write_simple(font,
                         text_string[TXT_OBJECTIVE],
                         FNT_CENTRAL, 180,
                         10,
                         1,
                         1,
                         0xFFFFFF);

        sprintf(temp_string, 
                text_string[TXT_OBJ_MIN_SCORE+(round_objective_type & OBJ_TYPE_MASK)],
                round_objective_count, round_objective_value);
        fnt_write_simple(font,
                         temp_string,
                         FNT_CENTRAL, 220,
                         10,
                         1,
                         1,
                         0xFFFFFF);

        if (!(round_objective_type & OBJ_FLAG_INFINITE))
          sprintf(temp_string, text_string[TXT_TIME_LIMIT], round_millis/1000);
        else
          sprintf(temp_string, text_string[TXT_TILE_LIMIT], round_millis);

        fnt_write_simple(font,
                         temp_string,
                         FNT_CENTRAL, 260,
                         10,
                         1,
                         1,
                         0xFFFFFF);

        i = 300;

        /* ----------------------------------- */
        /* Explain non-default letter movement */
        /* ----------------------------------- */
        switch(round_objective_type & OBJ_DROP_MASK)
        {
          case OBJ_REPLACE_LETTERS:
            string = text_string[TXT_REPLACE_LETTERS];
            break;
          case OBJ_LEAVE_LETTERS:
            string = text_string[TXT_LEAVE_LETTERS];
            break;
          default:
            string = 0;
            break;
        }
          
        if (string)
        {
          fnt_write_simple(font,
                           string,
                           FNT_CENTRAL, i,
                           10,
                           0.75,
                           1,
                           0xFFC835);
 
          i+=30;
        }

        /* ------------- */
        /* Explain flags */
        /* ------------- */
        if (round_objective_type & OBJ_FLAG_INVALID_DEDUCT)
        {
          fnt_write_simple(font,
                           text_string[TXT_INVALID_DEDUCT],
                           FNT_CENTRAL, i,
                           10,
                           0.75,
                           1,
                           0xFF6535);

          i+=30;
        }

        sprintf(temp_string, text_string[TXT_START_TO], text_string[TXT_PLAY]);
        fnt_write_simple(font,
                         temp_string,
                         FNT_CENTRAL, 430,
                         10,
                         1,
                         1,
                         0xFFFFFF);
      }
      break;

      /* Play ---------------------------------------------------------------- */

      case RND_ACT_PLAY:
      {
        unsigned char test_word[50];

        int word_score = 0;

        players_left = 0;
        darken = 0;

        curr_time = get_millis();

        for(i=0;i < players;i++)
        {
          if (!player_info[i].cur_delay && player_info[i].type < RETIRED)
          {
            int32 word_min_x = view_x + view_w;
            int32 word_max_x = view_x;
            int32 word_min_y = view_y + view_h;
            int32 word_max_y = view_y;

            int32 x = player_info[i].x;
            int32 y = player_info[i].y;

            players_left++;

            /* -------------- */
            /* MOVE - DIGITAL */
            /* -------------- */

            if (cont[player_info[i].cont].buttons & 0x00F0)
            {
              if (cont[player_info[i].cont].buttons & CONT_DPAD_UP)
                y -= DIGITAL_MOVEMENT;
              else if (cont[player_info[i].cont].buttons & CONT_DPAD_DOWN)
                y += DIGITAL_MOVEMENT;

              if (cont[player_info[i].cont].buttons & CONT_DPAD_LEFT)
                x -= DIGITAL_MOVEMENT;
              else if (cont[player_info[i].cont].buttons & CONT_DPAD_RIGHT)
                x += DIGITAL_MOVEMENT;
            }

            /* ------------- */
            /* MOVE - ANALOG */
            /* ------------- */

            if (cont[player_info[i].cont].type == CNT_CONT)
            {
              x += cont[player_info[i].cont].x/12;
              y += cont[player_info[i].cont].y/12;
            }
  
            /* --------------------------------------------------------- */
            /* If the player has actually moved, check still in the grid */
            /* --------------------------------------------------------- */

            if (player_info[i].x != x || player_info[i].y != y)
            {
              /* Still within the grid */
              if (x > LEFT_MARGIN  &&
                  x < RIGHT_MARGIN)
                player_info[i].x = x;

              if (y > TOP_MARGIN   &&
                  y < BOTTOM_MARGIN)
                player_info[i].y = y;

              player_info[i].square     = XY_TO_SQUARE(x, y, grid_wid);
              player_info[i].inactivity = 0;
            }
            else
              player_info[i].inactivity++;

            /* ------ */
            /* SELECT */
            /* ------ */
  
            if (cont[player_info[i].cont].buttons & CONT_A && 
              !(cont[player_info[i].cont].prev_buttons & CONT_A))
            {
              int check_square = player_info[i].square;

              if (grid[check_square]->action != ACT_NOTHING)
              {
                /* Square not available */
              }
              else if (grid[check_square]->letter == BLANK_LETTER)
              {
                /* Not on a valid letter */
              }
              else if (grid[check_square]->owner == i+1)
              {
                /* This is an 'undo' attempt */
            
                if (player_info[i].chain[player_info[i].chain_pos-1] == check_square)
                {
                  player_info[i].chain_pos--;
                  grid[check_square]->owner = 0;
                  grid[check_square]->tile->colour = grid[check_square]->tile->init_colour;
                }
              }
              else if (grid[check_square]->owner != 0)
              {
                /* This is owned by someone else */
              }
              else
              {
                if (player_info[i].chain_pos != 0)
                { 
                  int check_x = player_info[i].square % grid_wid;
                  int check_y = player_info[i].square / grid_wid;
                  int last_x  = player_info[i].chain[player_info[i].chain_pos-1] % grid_wid;
                  int last_y  = player_info[i].chain[player_info[i].chain_pos-1] / grid_wid;
  
                  if (check_x < last_x-1 || check_x > last_x+1 ||
                      check_y < last_y-1 || check_y > last_y+1)
                  {
                    /* not adjacent */
  
                    check_square = -1;
                  }
                }
  
                if (check_square != -1)
                {
                  player_info[i].chain[player_info[i].chain_pos++] = check_square;
                  grid[check_square]->owner = i+1;
                  grid[check_square]->tile->colour = player_info[i].colour | VID_BLACK_BORDER;
                }
              }
            }
  
            /* ----- */
            /* ENTER */
            /* ----- */

            for(j=0;j<player_info[i].chain_pos;j++)
            {
              int32 letter_x = grid[player_info[i].chain[j]]->x;
              int32 letter_y = grid[player_info[i].chain[j]]->y;

              test_word[j] = grid[player_info[i].chain[j]]->letter;
              word_score += letter_value[test_word[j]-'A'];

              if (j)
                player_info[i].word[j] = tolower(test_word[j]);
              else
                player_info[i].word[j] = test_word[j];

              if (letter_x < word_min_x)
                word_min_x = letter_x;
              
              if (letter_x > word_max_x)
                word_max_x = letter_x;

              if (letter_y < word_min_y)
                word_min_y = letter_y;
              
              if (letter_y > word_max_y)
                word_max_y = letter_y;
            }

            test_word[j] = player_info[i].word[j] = 0;

            if (cont[player_info[i].cont].buttons & CONT_B && 
              !(cont[player_info[i].cont].prev_buttons & CONT_B))
            {
              if (round_bonuses & BON_WORD_LENGTH)
              {
                if (j-3 < WML_MAX)
                  word_score *= word_multiplier[j-3];
                else
                  word_score *= word_multiplier[WML_MAX-1];
              }

              if (player_info[i].chain_pos < DCT_MIN_WORD_LENGTH)
              {
                /* Word too short */
              }
              else
              {
                int found = 0;
                uint32 colour;
                
                if ((round_objective_type & OBJ_TYPE_MASK) == OBJ_JOGGWORD)
                {
                  /* -------------------------------------------------- */
                  /* Is this the word we're looking for? If it's the    */
                  /* right word in the wrong position give them a hint! */
                  /* -------------------------------------------------- */

                  if (round_objective_type & OBJ_FLAG_LAYOUT)
                  {
                    memcpy(&layout_chain,&player_info[i].chain,sizeof(player_info[i].chain));
                    layout_chain_pos = player_info[i].chain_pos;
                    activity = RND_ACT_LAYOUT;
                    break;
                  }
                  else
                  {
                    if (joggword_info[cur_joggword].length == player_info[i].chain_pos)
                    {
                      int k = 0;

                      while(k < joggword_info[cur_joggword].length)
                      {
                        if (player_info[i].chain[k] != joggword_info[cur_joggword].chain[k])
                          break;

                        k++;
                      }

                      if (k == joggword_info[cur_joggword].length)
                      {
                        cur_joggword++;
                        found = 1;
                      }
                    }
                  }
                }
                else
                  found = dct_find(basic_dictionary,test_word);

                if (found)
                  colour = grid[player_info[i].chain[0]]->tile->init_colour;
                else
                  colour = 0x000000;

                while(--j>=0)
                {
                  int square = player_info[i].chain[j];

                  grid[square]->owner  = 0;
                  grid[square]->tile->colour = grid[square]->tile->init_colour;

                  if (found)
                  {
                    grid[square]->action = round_fade_action;

                    collect(grid[square]->collectable, &player_info[i], &word_score);

                    if (colour && (grid[square]->tile->init_colour != colour))
                      colour = 0x000000;

                    if (!(round_fade_action & ACT_RANDOM_DELAY))
                    {
                      obj_inst_setevent(grid[square]->tile, OBJ_EVENT_DEAD);
                      obj_inst_add_default(obj_inst_create(round_item[ITM_TILE_CLEAR],
                                                           grid[square]->x,
                                                           grid[square]->y,
                                                           GRID_Z,0,0,0,1,1,
                                                           player_info[i].colour));
                    }
                  }
                }

                /* ------------------------- */
                /* Check for first completer */
                /* ------------------------- */
                if ((round_bonuses & BON_FIRST_COMPLETER) &&
                     round_objective_count &&
                     player_info[i].objective_count >= round_objective_count)
                {
                  round_bonuses &= ~BON_FIRST_COMPLETER;

                  collect(round_fc_collectable, &player_info[i], &word_score);
                }

                /* --------------------- */
                /* Check for same colour */
                /* --------------------- */
                if ((round_bonuses & BON_SAME_COLOUR) && colour)
                  collect(round_sc_collectable, &player_info[i], &word_score);

                if (found)
                {
                  int item = ITM_NOTHING;

                  if (word_score > player_info[i].hi_word_score)
                    player_info[i].hi_word_score = word_score;

                  if (player_info[i].chain_pos > player_info[i].hi_word_length)
                    player_info[i].hi_word_length = player_info[i].chain_pos;

                  /* ----------------------------- */
                  /* Contribute to round objective */
                  /* ----------------------------- */
                  switch(round_objective_type & OBJ_TYPE_MASK)
                  {
                    case OBJ_MIN_SCORE:
                      player_info[i].objective_count += word_score;
                      break;
                    case OBJ_MIN_WORDS_STARTING_WITH:
                      if (test_word[0] == toupper(round_objective_value))
                        player_info[i].objective_count += 1;
                      break;
                    case OBJ_MIN_THREE_LETTER_WORDS:
                    case OBJ_MIN_FOUR_LETTER_WORDS:
                    case OBJ_MIN_FIVE_LETTER_WORDS:
                    case OBJ_MIN_SIX_LETTER_WORDS:
                      if (player_info[i].chain_pos >= (round_objective_type & OBJ_TYPE_MASK))
                        player_info[i].objective_count += 1;
                      break;
                    case OBJ_JOGGWORD:
                    default:
                      player_info[i].objective_count += 1;
                      break;
                  }

                  /* --------------------------------------- */
                  /* Work out which item to display (if any) */
                  /* --------------------------------------- */
                  switch(round_objective_type & OBJ_TYPE_MASK)
                  {
                    case OBJ_JOGGWORD:
                      break;
                    default:
                      if (item == ITM_NOTHING)
                      {
                        switch(player_info[i].chain_pos)
                        {
                          case 3:
                            item = ITM_3_LETTER_WORD;
                            break;
                          case 4:
                            item = ITM_4_LETTER_WORD;
                            break;
                          case 5:
                            item = ITM_5_LETTER_WORD;
                            break;
                          case 6:
                            item = ITM_6_LETTER_WORD;
                            break;
                          case 7:
                            item = ITM_7_LETTER_WORD;
                            break;
                          default:
                            item = ITM_BIG_WORD;
                            break;
                        }
                      }
                      break;
                  }

                  if (item != ITM_NOTHING && round_item[item])
                  {
                    obj_inst_add_default(obj_inst_create(round_item[item],
                                                         word_min_x+((word_max_x-word_min_x)/2),
                                                         word_min_y+((word_max_y-word_min_y)/2),
                                                         10,0,0,0,1,1,player_info[i].colour));
                    player_info[i].score += word_score;
                  }
                }
                else if (round_objective_type & OBJ_FLAG_INVALID_DEDUCT)
                {
                  /* ---------------------------- */
                  /* Detract from round objective */
                  /* ---------------------------- */
                  switch(round_objective_type & OBJ_TYPE_MASK)
                  {
                    case OBJ_MIN_SCORE:
                      player_info[i].objective_count -= word_score;
                    default:
                      player_info[i].objective_count -= 1;
                      break;
                  }
                }

                player_info[i].chain_pos = 0;
              }
            }
            else if (j>=MAX_HIGHLIGHT && ! dct_find(basic_dictionary,test_word))
            {
              /* ------------- */
              /* Force a CLEAR */
              /* ------------- */
              cont[player_info[i].cont].buttons |= CONT_X;
              cont[player_info[i].cont].prev_buttons &= ~CONT_X;
            }

            /* ---------------------- */
            /* RETIRE / LETTER CHANGE */
            /* ---------------------- */
            if (cont[player_info[i].cont].buttons & CONT_Y)
            {
              if (round_objective_type & OBJ_FLAG_LAYOUT)
              {
                j = player_info[i].square;

                grid[j]->letter = grid[j]->letter + 1;

                if (grid[j]->letter > 'Z')
                  grid[j]->letter = 'A';

                player_info[i].cur_delay = 10;
              }
              else if (!(cont[player_info[i].cont].prev_buttons & CONT_Y))
              {
                if ((round_objective_type & OBJ_TYPE_MASK) == OBJ_CLEARBOARD)
                {
                  player_info[i].type = RETIRED;
                  player_info[i].colour &= 0xFF7F7F7F;

                  /* ------------- */
                  /* Force a CLEAR */
                  /* ------------- */
                  cont[player_info[i].cont].buttons |= CONT_X;
                  cont[player_info[i].cont].prev_buttons &= ~CONT_X;
                }
              }
            }

            /* ----- */
            /* CLEAR */
            /* ----- */
            if (cont[player_info[i].cont].buttons & CONT_X && 
              !(cont[player_info[i].cont].prev_buttons & CONT_X))
            {
              int n = player_info[i].chain_pos-1;

              while(n>=0)
              {
                grid[player_info[i].chain[n]]->owner = 0;
                grid[player_info[i].chain[n]]->tile->colour = grid[player_info[i].chain[n]]->tile->init_colour;
                n--;
              }
  
              player_info[i].chain_pos = 0;
              player_info[i].word[0] = 0;
            }

            /* ----- */
            /* PAUSE */
            /* ----- */

            if (cont[player_info[i].cont].buttons & CONT_START &&
              !(cont[player_info[i].cont].prev_buttons & CONT_START))
            {
              start_time = curr_time;
              activity = RND_ACT_PAUSE;
            }
          }
          else
            player_info[i].cur_delay--;

          /* -------------------- */
          /* Check chain validity */
          /* -------------------- */
          if (player_info[i].chain_pos)
          {
            int n = player_info[i].chain_pos-1;

            while(n>=0)
            {
              /* ------------- */
              /* Invalid chain */
              /* ------------- */
              if (grid[player_info[i].chain[n--]]->letter == BLANK_LETTER)
              {
                /* word ruined by other player! */

                n = player_info[i].chain_pos-1;

                while(n>=0)
                {
                  grid[player_info[i].chain[n]]->owner = 0;
                  grid[player_info[i].chain[n]]->tile->colour = grid[player_info[i].chain[n]]->tile->init_colour;
                  n--;
                }

                player_info[i].chain_pos = 0;

                break;
              }


            }
          }
        }

        /* ------- */
        /* Shuffle */
        /* ------- */
        for(i=0;i<grid_wid;i++)
        {
          int j;

          int square = (grid_wid*grid_hgt)-i-1;

          for(j=0;j<grid_hgt;j++)
          {
            if (grid[square]->letter == BLANK_LETTER && grid[square]->tile)
            {
              obj_inst_t *tile = grid[square]->tile;

              int to_square   = square;
              int from_square = square-grid_wid;

              while(from_square >= 0 && grid[from_square]->tile)
              {
                int from_y = grid[from_square]->tile->y;

                grid[to_square]->letter      = grid[from_square]->letter;
                grid[to_square]->collectable = grid[from_square]->collectable;
                grid[to_square]->action      = grid[from_square]->action;
                grid[to_square]->orig_square = grid[from_square]->orig_square;

                grid[from_square]->tile->y = tile->y;
                grid[to_square]->tile = grid[from_square]->tile;
                tile->y = from_y;

                to_square -= grid_wid;
                from_square -= grid_wid;
              }

              grid[to_square]->tile = 0;
              grid[to_square]->letter = BLANK_LETTER;

              obj_inst_remove_default(tile,1);
            }

            square -= grid_wid;
          }
        }

        if (players_left == 0)
        {
          activity = RND_ACT_END;
        }
        else if (((round_objective_type & OBJ_TYPE_MASK) == OBJ_JOGGWORD) && 
            (cur_joggword == round_joggwords))
        {
          if (!(round_objective_type & OBJ_FLAG_LAYOUT))
            activity = RND_ACT_END;
        }
        else if (!(round_objective_type & OBJ_FLAG_INFINITE))
        {
          /* ------------------------- */
          /* Check for end of round... */
          /* ------------------------- */
          if (curr_time >= end_time)
          {
            curr_time = end_time;

            activity = RND_ACT_END;
          }
        }
        else
        {
          /* -------------------------------- */
          /* Count the number of squares left */
          /* -------------------------------- */
          tiles_left = 0;
          for (i=(grid_wid*grid_hgt)-1; i>=0; i--)
            if (grid[i]->letter != BLANK_LETTER)
              tiles_left++;

          /* ----------------------------------------------------------- */
          /* round_millis is overloaded with the number of squares which */
          /* when left on the screen represents the end of the round     */
          /* ----------------------------------------------------------- */
          if (tiles_left <= round_millis)
            activity = RND_ACT_END;
        }

        switch(activity)
        {
          case RND_ACT_PLAY:
            /* ------------------------- */
            /* Set clock segment colours */
            /* ------------------------- */
            j = ((curr_time-start_time) % 1000) / (1000/CLOCK_SEGMENTS);

            if (clock_info[j].colour != UTL_RAINBOW_PALETTE_SIZE-1)
            {
              clock_info[j].colour = UTL_RAINBOW_PALETTE_SIZE;
              j = 1;
            }
            else
              j = 0;

            for(i=0;i<CLOCK_SEGMENTS;i++)
              clock_info[i].colour = (clock_info[i].colour - j) % UTL_RAINBOW_PALETTE_SIZE;
            break;

          /* ------------------------------- */
          /* This is the end of the round... */
          /* ------------------------------- */
          case RND_ACT_END:
            /* ------------------------------------------ */
            /* Count the number and value of squares left */
            /* ------------------------------------------ */
            tiles_left = 0;
            tiles_left_value = 0;
            for (i=(grid_wid*grid_hgt)-1; i>=0; i--)
              if (grid[i]->letter != BLANK_LETTER)
              {
                tiles_left++;
                tiles_left_value += (letter_value[grid[i]->letter-'A'] * 10);
              }

            /* ---------------------------------------------- */
            /* Count the number of players (retired or alive) */
            /* ---------------------------------------------- */
            players_left = 0;
            for(i=0; i<players; i++)
              if (player_info[i].type < GOING)
                players_left++;
            break;
        }
      }
      break;

      /* Pause --------------------------------------------------------------- */
   
      case RND_ACT_PAUSE:
      {
        darken = 0.5;

        for(i=0;i < players;i++)
        {
          if (player_info[i].type == HUMAN)
          {
            if (cont[player_info[i].cont].buttons & CONT_START && 
              !(cont[player_info[i].cont].prev_buttons & CONT_START))
            {
              curr_time = get_millis();
              end_time += (curr_time-start_time);
              activity = RND_ACT_PLAY;
            }
            else if (cont[player_info[i].cont].buttons & CONT_X &&
                   !(cont[player_info[i].cont].prev_buttons & CONT_X))
              new_round = 0;
          }
        }

        fnt_write_simple(font,
                         text_string[TXT_PAUSED],
                         FNT_CENTRAL, FNT_CENTRAL,
                         10,
                         2,
                         1,
                         0xFFFFFF);
      }
      break;

      /* Round End ----------------------------------------------------------- */

      case RND_ACT_END:
      {
        int start_pressed = 0;

        darken = 0.5;

        for(i=0; i<players; i++)
        {
          if (player_info[i].type < GOING)
          {
            if (cont[player_info[i].cont].buttons & CONT_START &&
              !(cont[player_info[i].cont].prev_buttons & CONT_START))
            {
              start_pressed = 1;
              new_round = 0;
            }
          }
        }

        fnt_write_simple(font,
                         text_string[TXT_TIME_UP],
                         FNT_CENTRAL, 20,
                         10,
                         2,
                         1,
                         0xFFFFFF);

        fnt_write_simple(font,
                         text_string[TXT_OBJECTIVE],
                         FNT_CENTRAL, 100,
                         10,
                         1,
                         1,
                         0xFFFFFF);

        sprintf(temp_string,
                text_string[TXT_OBJ_MIN_SCORE+(round_objective_type & OBJ_TYPE_MASK)],
                round_objective_count, round_objective_value);
        fnt_write_simple(font,
                         temp_string,
                         FNT_CENTRAL, 140,
                         10,
                         1,
                         1,
                         0xFFFFFF);

        j = 200;

        for(i=0; i<players; i++)
        {
          if (player_info[i].type < GOING)
          {
            unsigned char *sorf = text_string[TXT_SUCCESS];
            
            /* -------------------------------------------- */
            /* Some rounds don't have per-player objectives */
            /* -------------------------------------------- */
            if (round_objective_count > 0)
            {
              if (player_info[i].objective_count < round_objective_count)
                sorf = text_string[TXT_FAILURE];

              sprintf(temp_string, text_string[TXT_OBJ_RESULT],
                      i+1,                  
                      sorf,                  
                      player_info[i].objective_count);

              fnt_write_simple(font,
                               temp_string,
                               FNT_CENTRAL, j,
                               10,
                               1,
                               1,
                               player_info[i].colour);

              j += 40;
            }

            if (start_pressed)
            {
              if ((round_objective_type & OBJ_TYPE_MASK) == OBJ_CLEARBOARD)
              {
                player_info[i].score = player_info[i].score - (tiles_left_value/players_left);

                if (player_info[i].score < 0)
                  player_info[i].score = 0;
              }

              if (sorf == text_string[TXT_FAILURE])
                player_info[i].type = GOING;
              else
                new_round = round + 1;
            }
          }
        }

        if (round_objective_count == 0)
        {
          switch((round_objective_type & OBJ_TYPE_MASK))
          {
            case OBJ_CLEARBOARD:
              sprintf(temp_string, text_string[TXT_OBJ_TILES_LEFT],
                      tiles_left,
                      tiles_left_value);

              fnt_write_simple(font,
                               temp_string,
                               FNT_CENTRAL, j,
                               10,
                               1,
                               1,
                               0xFF6535);

              j += 40;

              if (players_left > 1)
              {
                sprintf(temp_string, text_string[TXT_OBJ_CLEARBOARD_RESULTS],
                        tiles_left_value/players_left);

                fnt_write_simple(font,
                                 temp_string,
                                 FNT_CENTRAL, j,
                                 10,
                                 1,
                                 1,
                                 0xFF6535);

                j += 40;
              }
              break;

            case OBJ_JOGGWORD:
              break;
          }
        }

        sprintf(temp_string, text_string[TXT_START_TO], text_string[TXT_CONTINUE]);
        fnt_write_simple(font,
                         temp_string,
                         FNT_CENTRAL, 430,
                         10,
                         1,
                         1,
                         0xFFFFFF);
      }
      break;

      /* Layout display ------------------------------------------------------ */
   
      case RND_ACT_LAYOUT:
      {
        darken = 0.5;

        for(i=0;i < players;i++)
        {
          if (player_info[i].type == HUMAN)
          {
            if (cont[player_info[i].cont].buttons & CONT_START && 
              !(cont[player_info[i].cont].prev_buttons & CONT_START))
            {
              curr_time = get_millis();
              end_time += (curr_time-start_time);
              activity = RND_ACT_PLAY;
            }
          }
        }

        j = 32;

        for(i=0; i<layout_chain_pos; i++)
        {
          sprintf(temp_string, "square %d (orig: %d) %c",
                  layout_chain[i], grid[layout_chain[i]]->orig_square, grid[layout_chain[i]]->letter);
          fnt_write_simple(font,
                           temp_string,
                           48, j,
                           10,
                           0.5,
                           1,
                           0xFFFFFF);

          if (activity == RND_ACT_PLAY)
          {
            grid[layout_chain[i]]->owner = 0;
            grid[layout_chain[i]]->tile->colour = grid[layout_chain[i]]->tile->init_colour;
            grid[layout_chain[i]]->letter = BLANK_LETTER;
          }

          j+=16;
        }
      }
      break;


      /* Default ------------------------------------------------------------- */
   
      default:
      {
        darken = 0;
      }
      break;
    }

    #if defined(_WIN32)
    obj_inst_render_all(activity==RND_ACT_PLAY);
    #endif

    /* -------------- */
    /* Draw the clock */
    /* -------------- */
    
    sec = (end_time-curr_time)/1000;

    if (sec != prev_sec)
    {
      obj_inst_add_simple(round_item[ITM_SECOND_DOWN],
                          clock_info[j].x, clock_info[j].y, CLOCK_Z + 0.5);

      if (sec < 10)
        obj_inst_add_simple(round_item[ITM_10SECOND_DOWN],
                            clock_info[j].x, clock_info[j].y, CLOCK_Z + 0.5);
    }

    prev_sec = sec;

    if (!(round_objective_type & OBJ_FLAG_INFINITE))
    {
      sprintf(temp_string,"\7%c%d",60,(int)((curr_time-start_time) % 1000) / 100);
      fnt_write_simple(cfont,
                       temp_string,
                       532, 34,
                       CLOCK_Z - 0.5,
                       3.5,
                       1,
                       0x888888);
      sprintf(temp_string,"\7%c%02lu",32,sec);
    }
    else
      sprintf(temp_string,"!");

    for(i=0;i<CLOCK_SEGMENTS;i++)
    {
      til_render(round_tile_set, 7,
                 clock_info[i].x, clock_info[i].y,
                 CLOCK_Z - 0.5,
                 clock_info[i].rot,
                 1,
                 1,
                 utl_rainbow_palette[clock_info[i].colour]);
    }

    fnt_render(cfont,
               temp_string,
               560, 88,
               CLOCK_Z,
               0,
               time_scale,
               1,
               0xFFFFFF, 4, 1, 0);

    /* ------------- */
    /* Draw the grid */
    /* ------------- */
    for(i=(grid_wid*grid_hgt)-1; i>=0; i--)
    {
      int letter;
       
      switch(activity)
      {
        case RND_ACT_PAUSE:
          letter = (int)(RANDOM_NUMBER * 26);
          break;
        case RND_ACT_SHOW_OBJECTIVE:
          letter = grid[((grid_wid*grid_hgt)-1)-i]->letter - 'A';
          break;
        default:
          letter = grid[i]->letter - 'A';
          break;
      }

      /* ------- */
      /* Letters */
      /* ------- */
      if (grid[i]->letter != BLANK_LETTER)
      {
        if (grid[i]->action)
        {
          if (grid[i]->action & ACT_RANDOM_DELAY)
          {
            if ((RANDOM_NUMBER * 10) > 8)
            {
              grid[i]->action &= ~ACT_RANDOM_DELAY;
              obj_inst_setevent(grid[i]->tile, OBJ_EVENT_DEAD);
              obj_inst_add_simple(round_item[ITM_TILE_CLEAR],
                                  grid[i]->x,
                                  grid[i]->y,
                                  GRID_Z);
            }
          }
        }

        if (grid[i]->tile->scale <= 0 || grid[i]->tile->scale >= 3)
        {
          if (grid[i]->action & ACT_REPLACE)
            grid[i]->letter = 'A' + (RANDOM_NUMBER * 26);
          else if (!(grid[i]->action & ACT_LEAVE))
            grid[i]->letter = BLANK_LETTER;

          grid[i]->action = ACT_NOTHING;

          if (grid[i]->letter != BLANK_LETTER)
            obj_inst_setevent(grid[i]->tile, OBJ_EVENT_REVIVE);
        }
        else
        {
          if (grid[i]->collectable)
            til_render(round_tile_set,
                       (uint8)(40 + grid[i]->collectable - 'A'),
                       grid[i]->tile->x, grid[i]->tile->y,
                       grid[i]->tile->z+1,
                       grid[i]->tile->rot,
                       grid[i]->tile->scale,
                       1,
                       grid[i]->tile->colour & ~VID_BLACK_BORDER);

          til_render(letter_tile_set, 
                     (uint8)letter,
                     grid[i]->tile->x, grid[i]->tile->y,
                     grid[i]->tile->z + 2,
                     grid[i]->tile->rot,
                     grid[i]->tile->scale,
                     1,
                     grid[i]->owner?0x000000:0xFFFFFF);
        }
      }
    }

    /* ------------------------- */
    /* Draw the players / chains */
    /* ------------------------- */
    for(i=0; i<players; i++)
    {
      float amount_complete = 0;

      if (player_info[i].type < GONE)
      {
        int rot = 0;
        float scale = 1;

        if (!round_objective_count || player_info[i].objective_count >= round_objective_count)
          amount_complete = 1;
        else if (player_info[i].objective_count > 0)
          amount_complete = (float)player_info[i].objective_count/(float)round_objective_count;

        if (grid[player_info[i].square]->tile)
        {
          rot = grid[player_info[i].square]->tile->rot; 
          scale = grid[player_info[i].square]->tile->scale;
        }

        /* --------------------------------- */
        /* JIGGLE player marker if inactive! */
        /* --------------------------------- */
        if (player_info[i].type != RETIRED &&
            activity == RND_ACT_PLAY       &&
            ((player_info[i].inactivity/50)%5 == 4))
        {
          rot = (int)(RANDOM_NUMBER * 20)-10;
          scale += RANDOM_NUMBER;
        }

        til_render(round_tile_set,
                   (uint8)(2+i),
                   grid[player_info[i].square]->x, grid[player_info[i].square]->y,
                   6,
                   rot,
                   scale,
                   1,
                   player_info[i].colour);

        /* draw the x pointer */

        til_render(round_tile_set,
                   6,
                   player_info[i].x, round_player_arrow_y[i],
                   6,
                   round_player_arrow_y[i]<240?0:180,
                   1,
                   1,
                   player_info[i].colour);

        /* draw the y pointer */

        til_render(round_tile_set,
                   6, 
                   round_player_arrow_x[i], player_info[i].y,
                   6,
                   round_player_arrow_x[i]<320?270:90,
                   1,
                   1,
                   player_info[i].colour);

        /* draw the chain pointers */

        for(j=1; j<player_info[i].chain_pos; j++)
        {
          int x1  = grid[player_info[i].chain[j-1]]->x;
          int xd  = (grid[player_info[i].chain[j]]->x - x1)/2;
          int y1  = grid[player_info[i].chain[j-1]]->y;
          int yd  = (grid[player_info[i].chain[j]]->y - y1)/2;
          int rot = xd==0?(yd<0?180:0):(xd<0?(yd<0?135:(yd>0?45:90)):(yd<0?225:(yd>0?315:270)));

          til_render(round_tile_set,
                     6,
                     x1+xd, y1+yd,
                     4,
                     rot,
                     1,
                     1,
                     player_info[i].colour & 0xAAAAAA);
        }
      
        /* TEMPORARY SCORING STUFF */
        //vid_box(STATUS_AREA_X+67,
        //        STATUS_AREA_Y+(74*i)+(int)((1-amount_complete)*74),
        //        8,
        //        (int)(74*amount_complete),
        //        1,1,player_info[i].colour);
        
        if (!player_info[i].status_area)
          vid_box(view_x+557, view_y+194+(74*i), 142, 68, 0.5, 7, 0x000000);

        sprintf(temp_string,"%08d",player_info[i].score);
        fnt_write_simple(font,
                         temp_string,
                         494, 168+(74*i),
                         8,
                         0.75,
                         1,
                         player_info[i].colour);
        fnt_write_simple(font,
                         player_info[i].word,
                         494, 190+(74*i),
                         8,
                         0.5,
                         1,
                         0xDDDDDD);
        sprintf(temp_string, text_string[TXT_PERCENT_COMPLETE], (int)(amount_complete*100));
        fnt_write_simple(font,
                         temp_string,
                         494, 208+(74*i),
                         8,
                         0.5,
                         1,
                         player_info[i].colour);

        /* TEMPORARY SCORING STUFF */
      }
    }

    #if !defined(_WIN32)
    obj_inst_render_all(activity==RND_ACT_PLAY);
    #endif    

    /* ------------------------------------------ */
    /* If this is a Joggword round, show the clue */
    /* ------------------------------------------ */
    if ((round_objective_type & OBJ_TYPE_MASK) == OBJ_JOGGWORD)
    {
      if (joggword_info[cur_joggword].clue)
      {
        vid_box(view_x+560,view_y+84,152,120,0.5,7,0x000000);
        fnt_write(font,
                  joggword_info[cur_joggword].clue,
                  492, 32, 628,
                  8,
                  0,
                  0.5,
                  1,
                  0xFFFFFF,
                  0, 0,
                  4,
                  1,
                  0x000000);
      }
      else
        activity = RND_ACT_END;

    }

    if (darken != 0)
      vid_shadow(darken, 9, 0x000000);

    /* ------------------------------------------- */
    /* Additional rendering for the round activity */
    /* ------------------------------------------- */
    switch(activity)
    {
      case RND_ACT_PLAY:
        switch(prev_activity)
        {
          case RND_ACT_PLAY:
            snd_bg_poll();
            break;
          case RND_ACT_PAUSE:
            snd_bg_modify(BG_MODIFY_RESTART, round_background_music, 0);
            break;
          default:
            snd_bg_start(round_background_music);
            break;
        }
        break;

      case RND_ACT_PAUSE:
        if (prev_activity != activity)
          snd_bg_modify(BG_MODIFY_PAUSE, round_background_music, 0);
        break;

      default:
        if (prev_activity != activity)
          snd_bg_stop();
        break;
    }

    prev_activity = activity;

    vid_screen_end();
  }

  /* -------------------------------------------------------- */
  /* Make sure that the screen is reset (in case mid wobble)  */
  /* -------------------------------------------------------- */
  vid_view_set(VID_INIT_VIEW_X,
               VID_INIT_VIEW_Y,
               VID_INIT_VIEW_W,
               VID_INIT_VIEW_H,
               VID_INIT_VIEW_ROT);

  obj_inst_free_all();

  round_unload();
  
  return new_round;
}

/* start_game _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

main_state_t start_game(fnt_info_t *font, int style)
{
  int          player = 0;
  int          round = 1;
  char         roundname[50];
  int          hihiscore = 0;
  main_state_t new_state = MAIN_STATE_MENU;
  
  FILE *fp = 0;
  
  DBG_ENTRY(99, ("start_game(%p, %d [%s])", font, style, round_qualifier));
  
  sprintf(roundname,
          "%s/game/%s%s%03d.%s", 
          data_dir, RND_DEF_PREFIX, round_qualifier, round, RND_DEF_FILETYPE);
  
  DBG_PRINT(99, ("Loading '%s'", roundname));

  for (player=0;player<players;player++)
  {
    player_info[player].score = 0;
    player_info[player].hi_word_score = 0;
    player_info[player].hi_word_length = 0;
  }

  while(round && (fp = fopen(roundname, "rb")))
  {
    /* ---------------------------------------------------- */
    /* start_round returns the round to go to next, 0 means */
    /* game over... this means you can 'skip rounds' if you */
    /* want to by returning a value > round+1...            */
    /* ---------------------------------------------------- */
    round = start_round(fp, round, font);

    DBG_PRINT(99, ("new round %d", round)); 

    /* ---------------------------- */
    /* round completed successfully */
    /* ---------------------------- */
    if (round)
    {
      sprintf(roundname,
              "%s/game/%s%s%03d.%s",
              data_dir, RND_DEF_PREFIX, round_qualifier, round, RND_DEF_FILETYPE);
    }
  
    fclose(fp);
  }
  
  /* --------------------------------------------------- */
  /* If round > 0 here, it means that we couldn't find a */
  /* round file for the round reached... so the game has */
  /* been completed.                                     */
  /* --------------------------------------------------- */
  if (round)
  {
    /* BEAT ALL ROUNDS - SOME SORT OF 'WHOOPEE DO' */
  }
  else
  {
    /* GAME OVER */
  }
  
  /* ---------- */
  /* High Score */
  /* ---------- */
  fp = 0;

  sprintf(roundname, "%s/game/HI%s.%s", data_dir, round_qualifier, RND_DEF_FILETYPE);

  for(player=0; player<players; player++)
  {
    int i;

    for (i=0; i<HISCORES; i++)
    {
      if (player_info[player].score > hiscore[i].score)
      {
        int j;

        /* ------------------------------------------- */
        /* Animate the best highest hi score on return */
        /* ------------------------------------------- */
        if (player_info[player].score > hihiscore)
        {
          last_hiscore = i+1;
          hihiscore = player_info[player].score;
        }

        for(j=HISCORES-2; j>=i; j--)
          hiscore[j+1] = hiscore[j];

        hiscore[i].score = player_info[player].score;
        hiscore[i].hi_word_score = player_info[player].hi_word_score;
        hiscore[i].hi_word_length = player_info[player].hi_word_length;
        
        /* ------------------------- */
        /* Load the 'hi score' round */
        /* ------------------------- */
        if (fp == 0)
        {
          if ((fp = fopen(roundname,"rb")))
            round_load(fp,0,font);

          snd_bg_start(round_background_music);
        }

        hiscore_name(player, hiscore[i].initials, font);

        new_state = MAIN_STATE_HIGHSCORE;

        break;
      }
    }
  }

  if (fp)
  {
    round_unload();
    snd_bg_stop();
  }

  DBG_EXIT(99, ("start_game new_state=%d", new_state));
  
  return new_state;
}

/* game_load ________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void game_load(char *name)
{
  char gamename[50];

  FILE *fp = 0;

  DBG_ENTRY(99,("game_load(%s)",name));

  sprintf(gamename, "%s/game/%s.%s", data_dir, name, GAM_DEF_FILETYPE);

  DBG_PRINT(99, ("Loading '%s'", gamename));

  if ((fp = fopen(gamename, "rb")))
  {
    if (!text_string)
    {
      if ((text_string = (unsigned char **)utl_malloc(TXT_MAX*sizeof(unsigned char *))))
        memset(text_string,0x00,TXT_MAX*sizeof(unsigned char *));
      DBG_ELSE(99,("Unable to allocate text_string"));
    }

    if (!random_string)
    {
      if ((random_string = (unsigned char **)utl_malloc(RST_MAX*sizeof(unsigned char *))))
        memset(random_string,0x00,RST_MAX*sizeof(unsigned char *));
      DBG_ELSE(99,("Unable to allocate random_string"));
    }

    memset(&game_item, 0x00, sizeof(game_item));

    data_load(fp, "Loading");

    fclose(fp);
  }

  DBG_ELSE(99,("couldn't open '%s'", gamename));

  DBG_EXIT(99,("game_load"));

  return;
}

/* game_unload ______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void game_unload(void)
{
  int i;

  DBG_ENTRY(99,("game_unload()"));

  for(i=0; i<TXT_MAX; i++)
    utl_free(text_string[i]);
  utl_free(text_string);

  for(i=0; i<RST_MAX; i++)
    utl_free(random_string[i]);
  utl_free(random_string);

  til_set_unload(game_tile_set);

  til_set_unload(letter_tile_set);

  if (game_object_file)
  {
    obj_unload(game_object_file);  
    free(game_object_file);
    game_object_file = 0;
  }

  snd_bg_unload(game_background_music);
  game_background_music = -1;

  memset(&game_item, 0x00, sizeof(game_item));

  DBG_EXIT(99,("game_unload"));

  return;
}

/* main _____________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int main(int argc, char **argv) 
{
  unsigned char temp_string[100];
  int conts = 0;
  int i;

  /* Don't modify the screen mode just yet */
  vid_screen_init(0);

  obj_init(); /* incorporates til/img/ani/fnt/snd_all inits */

  conts = cnt_init((cnt_info_t *)&cont);
  
  fnt_load("FN000", &font);

  game_load(DEFAULT_GAME);

  game_loaded = 1;

  /* --------------------------------------------- */
  /* If there is at least one controller connected */
  /* --------------------------------------------- */
  if (conts)
  {
    int vidselect_row = 0;
    int menu_row = 0;
    int config_row = CONFIG_ROWS-1;
    int config_column = 0;

    int          title_frame = 0;
    int          logo_height = obj_getheight(game_item[ITM_LOGO]);
    main_state_t state = MAIN_STATE_VIDSELECT;
    main_state_t prev_state = MAIN_STATE_QUIT;

    players = conts;

    for (i=0; i<players; i++)
      player_info[i].cont = i;

    snd_bg_start(game_background_music);

    /* ------------------------------------------------ */
    /* state:                                           */
    /*  0 - Quit.                                       */
    /*  1 - Video Mode select                           */
    /*  2 - Menu.                                       */
    /*  3 - High scores.                                */
    /*  4 - Config.                                     */
    /*  5 - Game.                                       */
    /* ------------------------------------------------ */
    while(state != MAIN_STATE_QUIT && check_win32())
    {
      cnt_state_all((cnt_info_t *)&cont); 

      title_frame++;

      for(i=0;(i < conts) && (state == prev_state);i++)
      {
        if (state > MAIN_STATE_MENU &&
            state < MAIN_STATE_CONFIG &&
            !(title_frame % HIGHSCORE_FREQUENCY))
        {
          if (state == MAIN_STATE_MENU)
            state = MAIN_STATE_HIGHSCORE;
          else
            state = MAIN_STATE_MENU;
        }
        else if (cont[i].buttons & CONT_DPAD_DOWN && !(cont[i].prev_buttons & CONT_DPAD_DOWN))
        {
          if (state == MAIN_STATE_MENU && menu_row < MENU_ROWS-1)
            menu_row++;
          else if (state == MAIN_STATE_CONFIG && config_row < CONFIG_ROWS-1)
            config_row++;
          else if (state == MAIN_STATE_VIDSELECT && vidselect_row < VIDSELECT_ROWS-1)
          {
            vidselect_row++;
            vid_screen_init(vidselect_row+1);
          }
        }
        else if (cont[i].buttons & CONT_DPAD_UP && !(cont[i].prev_buttons & CONT_DPAD_UP))
        {
          if (state == MAIN_STATE_MENU && menu_row > 0)
            menu_row--;
          else if (state == MAIN_STATE_CONFIG && config_row > 0)
            config_row++;
          else if (state == MAIN_STATE_VIDSELECT && vidselect_row > 0)
          {
            vidselect_row--;
            vid_screen_init(vidselect_row+1);
          }
        }
        else if (cont[i].buttons & CONT_B && !(cont[i].prev_buttons & CONT_B))
        {
          if (state != MAIN_STATE_VIDSELECT)
            state = MAIN_STATE_MENU;
        }
        else if ((cont[i].buttons & CONT_START && !(cont[i].prev_buttons & CONT_START)) ||
                 (cont[i].buttons & CONT_A && !(cont[i].prev_buttons & CONT_A)))
        {
          switch(state)
          {
            case MAIN_STATE_MENU:
            {
              switch(menu_row)
              {
                case 0:
                  state = MAIN_STATE_GAME;
                  break;
                case 1:
                  state = MAIN_STATE_HIGHSCORE;
                  break;
                case 2:
                  state = MAIN_STATE_CONFIG;
                  break;
              }
            }
            break;

            case MAIN_STATE_VIDSELECT:
            case MAIN_STATE_HIGHSCORE:
            case MAIN_STATE_CONFIG:
              state = MAIN_STATE_MENU;
              break;
          }
        }
        else if ((cont[i].buttons & CONT_X) && (cont[i].buttons & CONT_Y)) /* TEMP */
          state = MAIN_STATE_QUIT;
      }
      
      if (state != prev_state)
      {
        obj_inst_free_all();

        switch (state)
        {
          case MAIN_STATE_VIDSELECT:
          case MAIN_STATE_MENU:
          case MAIN_STATE_HIGHSCORE:
          case MAIN_STATE_CONFIG:
            obj_inst_add_simple(game_item[ITM_LOGO],
                                view_mid_x, view_y + 16 + (logo_height/2), 14);
            obj_inst_add_simple(game_item[ITM_TITLE_BACKDROP_BOARD],
                                view_x + (512/2), view_mid_y, 2);
            obj_inst_add_simple(game_item[ITM_TITLE_BACKDROP_STATUS],
                                (view_x+view_w) - (128/2), view_mid_y, 2);

            break;
          default:
            break;
        }

        title_frame = 0;
      }

      vid_screen_begin_rgb(game_background_colour,1);

      switch(state)
      {
        /* ----------------------------------------------- */
        /* Vid Select / Menu / Highscore / Config          */
        /* ----------------------------------------------- */
        case MAIN_STATE_VIDSELECT:
        case MAIN_STATE_MENU:
        case MAIN_STATE_HIGHSCORE:
        case MAIN_STATE_CONFIG:
        {
          int x,y;

          vid_screen_continue();

          fnt_write_simple(&font,
                           text_string[TXT_DESCRIPTION],
                           FNT_CENTRAL, view_y + 20 + logo_height,
                           14,
                           0.5,
                           1,
                           0xFFFFFF);

          switch(state)
          {
            /* ---------- */
            /* Vid Select */
            /* ---------- */
            case MAIN_STATE_VIDSELECT:
              x = 0;
              y = view_y + 100 + logo_height;

              for(i=0; i < VIDSELECT_ROWS; i++)
              {
                int w,h;

                fnt_write(&font,
                          text_string[TXT_VIDSELECT_NTSC+i],
                          FNT_CENTRAL, y, 0,
                          14,
                          0,
                          0.75,
                          1,
                          i == vidselect_row ? (title_frame/5) | 0x01000000 : 0xFFFFFF,
                          &w, &h,
                          4,
                          1,
                          0x000000);

                if (w>x)
                  x=w;

                y += h + 10;
              }

              vid_box(view_mid_x,
                      view_y+90+logo_height + (y-(view_y+90+logo_height))/2,
                      x+20,
                      y-(view_y+90+logo_height),
                      0.5,
                      13,
                      0x000000);
              break;

            /* ---- */
            /* Menu */
            /* ---- */
            case MAIN_STATE_MENU:
             x = 0;
             y = view_y + 100 + logo_height;
      
             for(i=0; i < MENU_ROWS; i++)
             {
               int w,h;
      
               fnt_write(&font,
                         text_string[TXT_START_GAME+i],
                         FNT_CENTRAL, y, 0,
                         14,
                         0,
                         0.75,
                         1,
                         i == menu_row ? (title_frame/5) | 0x01000000 : 0xFFFFFF,
                         &w, &h,
                         4,
                         1,
                         0x000000);
      
               if (w>x)
                 x=w;
      
               y += h + 10;
             }
      
             vid_box(view_mid_x,
                     view_y+90+logo_height + (y-(view_y+90+logo_height))/2,
                     x+20,
                     y-(view_y+90+logo_height),
                     0.5,
                     13,
                     0x000000);
             break;
            /* ----------- */
            /* High Scores */
            /* ----------- */
            case MAIN_STATE_HIGHSCORE:
            {
              int w,h = 0;
              int yd = (int)(0.75 * (float)til_getheight(game_tile_set));
              int xd = (int)(0.75 * (float)til_getwidth(game_tile_set));

              y = view_y + 60 + logo_height;

              fnt_write(&font,
                        text_string[TXT_HIGH_SCORE_HEADING],
                        FNT_CENTRAL, y, 0,
                        14,
                        0,
                        0.75,
                        1,
                        (title_frame/5)|0x01000000,
                        &w, &h,
                        4,
                        1,
                        0x000000);

              y += h + 26;

              for (i=0; i<HISCORES; i++)
              {
                x = 184;

                for(w=0; w<3; w++)
                {
                  uint32 colour = hiscore_colours[(hiscore[i].score+w)%HISCORE_COLOURS];

                  if (i == last_hiscore-1)
                    colour = hiscore_colours[((curr_frame/10)+w)%HISCORE_COLOURS];

                  til_render(game_tile_set,
                             0,
                             x, y,
                             14,
                             0,
                             0.75,
                             1,
                             colour);

                  til_render(letter_tile_set, 
                             (uint8)(hiscore[i].initials[w]-'A'),
                             x, y,
                             15,
                             0,
                             0.75,
                             1,
                             0xFFFFFF);

                  x+=xd;
                }

                y -= 6;

                sprintf(temp_string, "%d",
                        hiscore[i].score);

                fnt_write(&font,
                          temp_string,
                          x, y, 0,
                          14,
                          0,
                          0.5,
                          1,
                          0xFFFFFF,
                          0, 0,
                          4,
                          1,
                          0x000000);

                x += 100;

                sprintf(temp_string, "%d",
                        hiscore[i].hi_word_score);

                fnt_write(&font,
                          temp_string,
                          x, y, 0,
                          14,
                          0,
                          0.5,
                          1,
                          0xFFFFFF,
                          0, 0,
                          4,
                          1,
                          0x000000);

                x += 100;

                sprintf(temp_string, "%d",
                        hiscore[i].hi_word_length);

                fnt_write(&font,
                          temp_string,
                          x, y, 0,
                          14,
                          0,
                          0.5,
                          1,
                          0xFFFFFF,
                          &w, &h,
                          4,
                          1,
                          0x000000);

                y += yd + 6;
              }

              vid_box(view_mid_x,
                      view_y+50+logo_height + (y-(view_y+50+logo_height))/2,
                      x,
                      y-(view_y+50+logo_height),
                      0.5,
                      13,
                      0x000000);

            }
            break;
            case MAIN_STATE_CONFIG:
              /* Configuration */
             break;
          }
      
          #if 0
          {
            //sprintf(temp_string, "ogh: %d item: %d type: %d (%d)", ogh, game_item[ITM_LOGO],obj_infos[game_item[ITM_LOGO]].type, 'I');
            //sprintf(temp_string, "rd: %f rdd: %f", (double)rot_delta,(double)rot_delta_delta);
            sprintf(temp_string, "sa count: %d", sa_c);
            fnt_write_simple(&font,
                             temp_string,
                             FNT_CENTRAL, FNT_CENTRAL,
                             20,
                             0.5,
                             1,
                             0xFFFFFF);
          }
          #endif
      
          fnt_write_simple(&font,
                           "Code: Warmtoe Music: SeGaFrEaK NL",
                           FNT_CENTRAL, 450,
                           10,
                           0.5,
                           1,
                           0xFFFFFF);
          
          for(x=0; x<16; x++)
          {
            y = sa_bars[x] * 2;

            vid_box(20+x*40,
                    view_mid_y,
                    36,
                    y,
                    0.8,
                    1,
                    utl_explosion_palette[UTL_EXPLOSION_PALETTE_SIZE-y]);
          }
          
          obj_inst_render_all(1);

          snd_bg_poll();

          vid_screen_end();

          prev_state = state;
        }
        break;
                
        /* ------------------------------------ */
        /* Game                                 */
        /* ------------------------------------ */
        case MAIN_STATE_GAME:
        {
          vid_screen_end();

          /* --------------------------------------------- */
          /* Temporary - this will go in the config screen */
          /* --------------------------------------------- */
          for(i=0;i<players;i++)
            player_info[i].type = HUMAN;

          snd_bg_stop();

          state = start_game(&font, 0);

          snd_bg_start(game_background_music);

          prev_state = MAIN_STATE_GAME;
        }
        break;
      }    
    }
  }
  else
  {
    vid_screen_begin_rgb(game_background_colour,0);
    
    fnt_write(&font, "No valid controllers connected", 1.0, 
              FNT_CENTRAL, FNT_CENTRAL, 0, 0, 2, 1, 0xFFFFFF, 0, 0,
              4, 1, 0x000000);
              
    vid_screen_end();
    
    thd_sleep(60000);
  }
  
  game_unload();

  fnt_unload(&font);

  obj_term(); /* incorporates snd_all/fnt/ani/img/til term */

  vid_screen_term();

  return 0;
}

#undef JOGGLE_C
