#define ANI_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <math.h>
#include <png/png.h>

#include "utils.h"
#include "object.h"
#include "video.h"
#include "anim.h"

ani_info_t anims[ANI_MAX_ANIMS];                 /* static array of all anims */

/* ani_init _________________________________________________________________ */
/*                                                                            */
/* Initialise anim support.                                                   */
/* __________________________________________________________________________ */

void ani_init(void)
{
  DBG_ENTRY(99,("ani_init()"));
  
  memset(anims,0x00,sizeof(anims));

  DBG_EXIT(99,("ani_init"));
  
  return;
}

/* ani_term _________________________________________________________________ */
/*                                                                            */
/* Terminate anim support.                                                    */
/* __________________________________________________________________________ */

void ani_term(void)
{
  DBG_ENTRY(99,("ani_term()"));
  
  DBG_EXIT(99,("ani_term"));
  
  return;
}

/* get_path_for_dir _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int get_path_for_dir(int dir)
{
  if (dir == -1)
    return 0;
  return (dir/45)+1;
}

/* get_path_for_word ________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int get_path_for_word(char *word)
{
  int dir = 0;

  if (strstr(word,"NORTHEAST"))
    dir = 2;
  else if (strstr(word,"SOUTHEAST"))
    dir = 4;
  else if (strstr(word,"SOUTHWEST"))
    dir = 6;
  else if (strstr(word,"NORTHWEST"))
    dir = 8;
  else if (strstr(word,"NORTH"))
    dir = 1;
  else if (strstr(word,"EAST"))
    dir = 3;
  else if (strstr(word,"SOUTH"))
    dir = 5;
  else if (strstr(word,"WEST"))
    dir = 7;

  return dir;
}

/* get_cmd_for_word ________________________________________________________ */
/*                                                                           */
/* _________________________________________________________________________ */

static int get_cmd_for_word(char *word)
{
  int cmd = 0;

  if (strstr(word,"VAR"))
    cmd = 1;

  return cmd;
}


/* get_transition ___________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int get_transition(int old_dir, int new_dir)
{
  switch(old_dir)
  {
    case 7:
    {
      switch(new_dir)
      {
         case 3:
           return ANI_PATH_MOVE_WEST_TO_MOVE_EAST;
           break;
      }
      break;
    }
    break;
    
    case 3:
    {
      switch(new_dir)
      {
        case 7:
          return ANI_PATH_MOVE_EAST_TO_MOVE_WEST;
          break;
      }
      break;
    }
    break;
  }
  
  return ANI_PATH_UNDEFINED;
}

/* ani_obj_create_str _______________________________________________________ */
/*                                                                            */
/* Create an animation given an input string from a file.                     */
/* __________________________________________________________________________ */

uint32 ani_obj_create_str(char *string)
{
  uint32 info = 0x00000000;
 
  int ani;
  
  char *t;
  char *e;
  
  DBG_ENTRY(99,("ani_obj_create_str(%s)",string));
   
  if ((t = strtok(string,UTL_WHITESPACE)))
  {
    ani=(uint8)strtol(t,&e,10);

    if ((t = strtok(0,UTL_WHITESPACE)))
      anims[ani].dirchange=(int32)strtol(t,&e,10);
    if ((t = strtok(0,UTL_WHITESPACE)))
      anims[ani].score=(int32)strtol(t,&e,10);
  
    if (1)
    {
      char   file[256];
    
      FILE  *fp;
      
      sprintf(file,"%s/game/%s%03d.%s",data_dir,ANI_DEF_PREFIX,ani,ANI_DEF_FILETYPE);
  
      DBG_PRINT(99,("Attempting load of %s file (%s)",ANI_DEF_FILETYPE, file));
  
      if ((fp = fopen(file, "rb")))
      {
        char line[2048];
        int32 last_command = -1;
      
        while (fgets(line,sizeof(line),fp))
        {
          int   commands = 0;
          int32 *path = 0;
          
          int path_num = -1;
        
          char *t = strtok(line, UTL_WHITESPACE);
        
          if (!t || *t == '#')
            continue;
          
          do
          {
            int32 command = -1;
  
            DBG_PRINT(99,("T='%s'",t));          
  
            /* Deal with continuation */
            if (!strcmp(t, "\\"))
            {
              if (fgets(line, sizeof(line), fp))
                t = strtok(line, UTL_WHITESPACE);
              else
                break;
            } 
            
            if (*t == '+' || *t == '-' || (*t>='0' && *t<='9'))
            {
              if (*(t+1)=='x')
                command = strtol(t,0,0);
              else
                command = strtol(t,0,10);
            }
            else
            {
              if (!strcmp(t,"SCORE"))
                command = OBJ_SPECIAL_SCORE;
              else     CHECK_PATH_NAME(command,t,TRACK_1);
              else     CHECK_PATH_NAME(command,t,TRACK_2);
              else     CHECK_PATH_NAME(command,t,TRACK_3);
              else CHECK_MOVEPATH_NAME(command,t,NORTHEAST);
              else CHECK_MOVEPATH_NAME(command,t,SOUTHEAST);
              else CHECK_MOVEPATH_NAME(command,t,SOUTHWEST);
              else CHECK_MOVEPATH_NAME(command,t,NORTHWEST);
              else CHECK_MOVEPATH_NAME(command,t,NORTH);
              else CHECK_MOVEPATH_NAME(command,t,EAST);
              else CHECK_MOVEPATH_NAME(command,t,SOUTH);
              else CHECK_MOVEPATH_NAME(command,t,WEST);
              else     CHECK_PATH_NAME(command,t,MOVE_EAST_TO_MOVE_WEST);
              else     CHECK_PATH_NAME(command,t,MOVE_WEST_TO_MOVE_EAST);
              else      CHECK_CMD_NAME(command,t,MOVEBACK);
              else     CHECK_PATH_NAME(command,t,MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_NORTHEAST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_SOUTHEAST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_SOUTHWEST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_NORTHWEST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_NORTH_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_EAST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_SOUTH_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_WEST_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP_TO_MOVE);
              else     CHECK_PATH_NAME(command,t,STOP);
              else     CHECK_PATH_NAME(command,t,SHOOT);
              else     CHECK_PATH_NAME(command,t,HIT);
              else     CHECK_PATH_NAME(command,t,DEAD);
              else     CHECK_PATH_NAME(command,t,BOUNDARY_1);
              else     CHECK_PATH_NAME(command,t,BOUNDARY_2);
              else     CHECK_PATH_NAME(command,t,BOUNDARY_3);
              else     CHECK_PATH_NAME(command,t,BOUNDARY_4);
              else     CHECK_PATH_NAME(command,t,BOUNDARY_5);
              else     CHECK_PATH_NAME(command,t,REVIVE);
              else      CHECK_CMD_NAME(command,t,RAND);
              else      CHECK_CMD_NAME(command,t,CALC);
              else      CHECK_CMD_NAME(command,t,SETTRACK);
              else      CHECK_CMD_NAME(command,t,NEXT);
              else      CHECK_CMD_NAME(command,t,OBJECT);
              else      CHECK_CMD_NAME(command,t,PAUSE);
              else      CHECK_CMD_NAME(command,t,COPY);
              else      CHECK_CMD_NAME(command,t,GOTO);
              else      CHECK_CMD_NAME(command,t,ADDINSTLOOP);
              else      CHECK_CMD_NAME(command,t,ADDINST);
              else      CHECK_CMD_NAME(command,t,REVERT);
              else      CHECK_CMD_NAME(command,t,DIE);
              else      CHECK_CMD_NAME(command,t,BOUNCE);
              else      CHECK_CMD_NAME(command,t,SETSPD);
              else      CHECK_CMD_NAME(command,t,MODSPD);
              else      CHECK_CMD_NAME(command,t,SETDIR);
              else      CHECK_CMD_NAME(command,t,MODDIR);
              else      CHECK_CMD_NAME(command,t,SETZ);
              else      CHECK_CMD_NAME(command,t,MODZ);
              else      CHECK_CMD_NAME(command,t,SETSCALE);
              else      CHECK_CMD_NAME(command,t,MODSCALE);
              else      CHECK_CMD_NAME(command,t,GETGOING);
              else      CHECK_CMD_NAME(command,t,SETROT);
              else      CHECK_CMD_NAME(command,t,MODROT);
              else      CHECK_CMD_NAME(command,t,SETEOL);
              else      CHECK_CMD_NAME(command,t,SETFADE);
              else      CHECK_CMD_NAME(command,t,MODFADE);
              else      CHECK_CMD_NAME(command,t,SETCOL);
              else      CHECK_CMD_NAME(command,t,MODCOL);
              else      CHECK_CMD_NAME(command,t,SETX);
              else      CHECK_CMD_NAME(command,t,MODX);
              else      CHECK_CMD_NAME(command,t,SETY);
              else      CHECK_CMD_NAME(command,t,MODY);
              else      CHECK_CMD_NAME(command,t,SETVIEWROT);
              else      CHECK_CMD_NAME(command,t,MODVIEWROT);
              else      CHECK_ARG_NAME(command,t,X);
              else      CHECK_ARG_NAME(command,t,Y);
              else      CHECK_ARG_NAME(command,t,EOL);
              else      CHECK_ARG_NAME(command,t,VAR);
              else       CHECK_OP_NAME(command,t,PLUS);
              else       CHECK_OP_NAME(command,t,MINUS);
              else       CHECK_OP_NAME(command,t,MULT);
              else       CHECK_OP_NAME(command,t,DIV);
              else       CHECK_OP_NAME(command,t,MOD);
              else       CHECK_OP_NAME(command,t,AND);
              else       CHECK_OP_NAME(command,t,OR);
              else  DBG_PRINT(0,("UNRECOGNISED: '%s' PATH=%d FILE='%s'",t, path_num, file));
            }
  
            //if (last_command == ANI_CMD_COPY)
            //{
            //  DBG_PRINT(0,("COPY (0x%08X) VALUE 0x%08X\n", last_command, command);
            //}

            if (path_num == -1)
            {
              path_num = command;
            }
            else
            {            
              if ((path = utl_realloc(path,sizeof(int32)*commands+1)))
                path[commands++] = command;
            }

            last_command = command;
          }
          while((t=strtok(0, UTL_WHITESPACE)));
          
          anims[ani].path[path_num] = path;
        }
        
        fclose(fp);
      }
      /* info is the address of the ani_info structure */
      info = (uint32)&anims[ani];
    }
  }

  DBG_EXIT(99,("ani_obj_create_str rc=0x%08X",(unsigned int)info));
  
  return info;
}

/* ani_obj_free _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void ani_obj_free(uint32 info)
{
  int n;
  ani_info_t *anim = (ani_info_t *)info;
  
  DBG_ENTRY(99,("ani_obj_free(0x%08X)",(unsigned int)info));
  
  for(n=0;n<ANI_MAX_PATHS;n++)
    utl_free(anim->path[n]);
    
  DBG_EXIT(99,("ani_obj_free"));
}

/* ani_inst_create __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint32 ani_inst_create(uint32 info)
{
  ani_inst_t *inst = (ani_inst_t *)utl_malloc(sizeof(ani_inst_t));
  
  DBG_ENTRY(99,("ani_inst_create(0x%08X)",(unsigned int)info));
  
  if (inst)
  {
    memset(inst, 0x00, sizeof(ani_inst_t));
    
    inst->anim = (ani_info_t *)info;
    inst->trackpos = inst->anim->path[ANI_PATH_TRACK_1];
  }
  
  DBG_EXIT(99,("ani_inst_create rc=%p",inst));
  
  return (uint32)inst;
}

/* ani_inst_reset ___________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void ani_inst_reset(uint32 info)
{
  ani_inst_t *inst = (ani_inst_t *)info;
  
  DBG_ENTRY(99,("ani_inst_create(0x%08X)",(unsigned int)info));
  
  if (inst)
  {
    inst->anipos = 0;
    inst->pause_count = inst->pause_until = 0;
  }
  
  DBG_EXIT(99,("ani_inst_reset"));
  
  return;
}

/* ani_inst_free ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void ani_inst_free(uint32 inst)
{
  DBG_ENTRY(99,("ani_inst_free(0x%08X)",(unsigned int)inst));
  
  utl_free((ani_inst_t *)inst);
  
  DBG_EXIT(99,("ani_inst_free"));
}

/* ani_get_obj ______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int ani_get_obj(uint32 inst)
{
  int obj = ((ani_inst_t *)inst)->obj;
  
  DBG_ENTRY(99,("ani_get_obj(0x%08X)",(unsigned int)inst));
  
  DBG_EXIT(99,("ani_get_obj rc=%d", obj));
  
  return(obj);
}

/* ani_setevent _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void ani_setevent(obj_inst_t *obj_inst, int event)
{
  int ani_dir = get_path_for_dir(obj_inst->dir);
  int expected_path = ANI_PATH_EVENTSTART+(event*9);

  ani_inst_t *inst = (ani_inst_t *)obj_inst->var;
  
  DBG_ENTRY(99,("ani_setevent(%p,%d)",obj_inst,event));

  if (inst->anim->path[expected_path+ani_dir])
    expected_path += ani_dir;
  else if (!(inst->anim->path[expected_path]))
    expected_path = ANI_MAX_PATHS;
  
  if (expected_path < ANI_MAX_PATHS)
  {
    DBG_PRINT(99,("expected_path=%d %p",expected_path, inst->anim->path[expected_path]));
      
    inst->anipos = inst->anim->path[expected_path];
    inst->pause_count = inst->pause_until = 0;

    inst->trackpos = 0;
    inst->track_pause_count = inst->track_pause_until = 0;

  }
  DBG_ELSE(99,("expected path not valid %d", expected_path));

  inst->old_speed = obj_inst->speed;
  inst->old_dir   = ani_dir;

  DBG_EXIT(99,("ani_setevent"));
}

/* ani_getattribute _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int32 ani_getattribute(uint32 inst_ref, int attribute)
{
  int32 rc = 0;
  ani_inst_t *inst = (ani_inst_t *)inst_ref;
  
  DBG_ENTRY(99,("ani_getattribute(%d,%d)",(int)inst_ref,(int)attribute));

  switch(attribute)
  {
    case OBJ_ATTR_DIRCHANGE:
      rc = inst->anim->dirchange;
      break;
    case OBJ_ATTR_SCORE:
      rc = inst->anim->score;
      break;
  }
  
  DBG_EXIT(99,("ani_getattribute rc=%d",(int)rc));

  return rc;  
}

/* ani_calc _________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int ani_calc(obj_inst_t *obj_inst, int arg1, int op, int arg2)
{
  ani_inst_t *inst = (ani_inst_t *)obj_inst->var;

  int *a = &arg1;

  int var = 0;

  DBG_ENTRY(99,("ani_calc(%p, %d, %d, %d)",obj_inst, arg1, op, arg2));

  while(1)
  {
    switch(*a)
    {
      case ANI_ARG_X:
        *a = obj_inst->x;
        break;
      case ANI_ARG_Y:
        *a = obj_inst->y;
        break;
      case ANI_ARG_EOL:
        *a = obj_inst->eol;
        break;
      case ANI_ARG_VAR:
        *a = inst->stored_var;
        break;
      default:
        break;
    }

    if (a == &arg1)
      a = &arg2;
    else
      break;
  }

  DBG_PRINT(99,("pre_calc %d, %d, %d",arg1, op, arg2));

  switch(op)
  {
    case ANI_OP_PLUS:
      var = arg1+arg2;
      break;
    case ANI_OP_MINUS:
      var = arg1-arg2;
      break;
    case ANI_OP_MULT:
      var = arg1*arg2;
      break;
    case ANI_OP_DIV:
      var = arg1/arg2;
      break;
    case ANI_OP_MOD:
      var = arg1%arg2;
      break;
    case ANI_OP_AND:
      var = (uint32)arg1&(uint32)arg2;
      break;
    case ANI_OP_OR:
      var = (uint32)arg1|(uint32)arg2;
      break;
  }

  DBG_EXIT(99,("ani_calc rc=%d (0x%08X)",var,var));

  return var;
}


/* ani_run __________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

static int ani_run(obj_inst_t *obj_inst, int runmode)
{
  int   rc = ANI_RUN_CONTINUE;

  ani_inst_t *inst = (ani_inst_t *)obj_inst->var;

  int32  var = inst->stored_var;
  int32  tmpval;
  int32  loop = 1;
  int32  command = -1;

  int32 *anipos;
  int32  pause_until;
  int32  pause_count;

  DBG_ENTRY(99,("ani_run(%p, %d)",obj_inst, runmode));

  if (runmode == ANI_RUNMODE_TRACK)
  {
    anipos = inst->trackpos;
    pause_until = inst->track_pause_until;
    pause_count = inst->track_pause_count;
  }
  else
  {
    anipos = inst->anipos;
    pause_until = inst->pause_until;
    pause_count = inst->pause_count;
  }

  do
  {   
    command = *(anipos++);

    DBG_PRINT(99,("COMMAND: %d",(int)command));

    switch(command)
    {
      case ANI_CMD_RAND:
        var = *(anipos++);
        var += RANDOM_NUMBER * (*(anipos++)-var);
        break;

      case ANI_CMD_CALC:
        inst->stored_var = var;
        var = ani_calc(obj_inst,*anipos,*(anipos+1),*(anipos+2));
        anipos+=3;
        break;

      case ANI_CMD_NEXT:
        rc = ANI_RUN_FINISH;
        break;

      case ANI_CMD_COPY:
        tmpval = *(anipos++);
        anipos = inst->anim->path[tmpval];
        break;

      case ANI_CMD_GOTO:
        tmpval = *(anipos++);
        anipos += tmpval;
        break;

      case ANI_CMD_SETTRACK:
        tmpval = *(anipos++);
        goto ANI_CMD_SETTRACK_FINISH;
        break;
      case ANI_CMD_SETTRACK_VAR:
        tmpval = var;
        ANI_CMD_SETTRACK_FINISH:
        if (tmpval)
          inst->trackpos = inst->anim->path[ANI_PATH_TRACK_1+(--tmpval)];
        else
          inst->trackpos = 0;
        if (runmode == ANI_RUNMODE_TRACK)
        {
          if (!(anipos = inst->trackpos))
            rc = ANI_RUN_FINISH;
        }
        break;

      case ANI_CMD_REVERT:
        pause_count = inst->prev_pause_count;
        pause_until = inst->prev_pause_until;
        anipos = inst->prev_anipos;
        rc = ANI_RUN_REVERTED;
        break;

      case ANI_CMD_GETGOING:
        obj_inst->speed = obj_inst->init_speed;
        rc = ANI_RUN_CHANGE_DIRECTION;
        break;

      case ANI_CMD_OBJECT:
        inst->obj = *(anipos++);
        break;
      case ANI_CMD_OBJECT_VAR:
        inst->obj = var;
        break;

      case ANI_CMD_PAUSE:
        pause_until = *(anipos++);
        goto ANI_CMD_PAUSE_FINISH;
        break;
      case ANI_CMD_PAUSE_VAR:
        pause_until = var;
        ANI_CMD_PAUSE_FINISH:
        pause_count = 0;
        rc = ANI_RUN_FINISH;
        break;

      case ANI_CMD_ADDINSTLOOP:
        loop = *(anipos++);
        tmpval = *(anipos++);
        goto ANI_CMD_ADDINST_FINISH;
        break;
      case ANI_CMD_ADDINSTLOOP_VAR:
        loop = *(anipos++);
        tmpval = var;
        goto ANI_CMD_ADDINST_FINISH;
        break;
      case ANI_CMD_ADDINST:
        loop = 1;
        tmpval = *(anipos++);
        goto ANI_CMD_ADDINST_FINISH;
        break;
      case ANI_CMD_ADDINST_VAR:
        loop = 1;
        tmpval = var;

        ANI_CMD_ADDINST_FINISH:
        {
          int32 *base_anipos = anipos;
          int obj=0, x, y, zd, rot, dir, speed, sint;
          uint32 colour;
          float z, scale, fade;

          while(loop-->0)
          {
            obj_inst_t *new_obj_inst = 0;

            anipos = base_anipos;

            // copy existing obj / deal with random         
            if (tmpval < 0)
            {
              if (tmpval == -1)
                obj = obj_inst->obj;
              else
              {
                y = -tmpval;

                x = RANDOM_NUMBER * y;

                while(y--)
                {
                  if (y==x)
                    obj = *anipos;

                  anipos++;
                }
              }
            }
            else
              obj = tmpval;

            x      = *(anipos++);
            y      = *(anipos++);
            zd     = *(anipos++);
            z      = obj_inst->z;
            rot    = *(anipos++);
            dir    = *(anipos++);
            speed  = *(anipos++);
            sint   = *(anipos++);
            fade   = obj_inst->fade;
            colour = (uint32)*(anipos++);

            if (z == -1.0)
              break;

            // copy existing / initial x
            switch(x)
            {
              case -1:
                x = obj_inst->x;
                break;
              case -2:
                x = obj_inst->init_x;
                break;
              case -3:
                x = view_x + (RANDOM_NUMBER * view_w);
                break;
            }
            // copy existing / initial y
            switch(y)
            {
              case -1:
                y = obj_inst->y;
                break;
              case -2:
                y = obj_inst->init_y;
                break;
              case -3:
                y = view_y + (RANDOM_NUMBER * view_h);
                break;
            }
            // copy existing / initial / random direction
            switch(dir)
            {
              case -1:
                dir = obj_inst->dir;
                break;
              case -2:
                dir = obj_inst->init_dir;
                break;
              case -3:
                dir = RANDOM_NUMBER * 360;
                break;
            }
            // copy existing / initial / random rotation
            switch(rot)
            {
              case -1:
                rot = obj_inst->rot;
                break;
              case -2:
                rot = obj_inst->init_rot;
                break;
              case -3:
                rot = RANDOM_NUMBER * 360;
                break;
              case -4:
                rot = obj_inst->dir;
                break;
            }  
            // copy existing / initial speed
            switch(speed)
            {
              case -1:
                speed = obj_inst->speed;
                break;
              case -2:
                speed = obj_inst->init_speed;
                break;
            }
            // copy existing / initial scale
            switch(sint)
            {
              case -1:
                scale = obj_inst->scale;
                break;
              case -2:
                scale = obj_inst->init_scale;
                break;
              default:
                scale = ANI_FLOAT_MULT * (float)sint;
                break;
            }
            // copy existing / initial z
            switch(zd)
            {
              case -1:
                z = obj_inst->z;
                break;
              case -2:
                z = obj_inst->init_z;
                break;
              default:
                z = obj_inst->z + (ANI_FLOAT_MULT * (float)zd);
                break;
            }
            // colour
            switch(colour)
            {
              case -1:
                colour = obj_inst->colour;
                break;
            }
            new_obj_inst = obj_inst_create(obj,
                                           (int32)x,
                                           (int32)y,
                                           z,
                                           (int32)rot,
                                           (int32)dir,
                                           (int32)speed,
                                           scale,
                                           fade,
                                           colour);

            if (new_obj_inst)
            {
              int class = OBJ_LIST_INCIDENTAL;

              switch(obj)
              {
                case OBJ_SPECIAL_SCORE:
                  new_obj_inst->var = inst->anim->score;
                  new_obj_inst->eol = 50;
                  break;

                default:
                  class = obj_infos[obj].class;
                  break;
              }

              obj_inst_animate(new_obj_inst);              
              obj_lists[class] = obj_inst_add(obj_lists[class],new_obj_inst);
            }
          }
        }
        break;

      case ANI_CMD_DIE:
        tmpval = *(anipos++);
        goto ANI_CMD_DIE_FINISH;
        break;
      case ANI_CMD_DIE_VAR:
        tmpval = var;
        ANI_CMD_DIE_FINISH:
        obj_inst->z = 0;
        obj_inst->eol = tmpval;
        pause_until = tmpval;
        pause_count = 0;
        rc = ANI_RUN_FINISH;
        break;

      case ANI_CMD_BOUNCE:
        obj_inst->dir = (obj_inst->dir+90)%360;
        rc = ANI_RUN_CHANGE_DIRECTION;
        break;

      case ANI_CMD_MOVEBACK:
        loop = *(anipos++);
        goto ANI_CMD_MOVEBACK_FINISH;
        break;
      case ANI_CMD_MOVEBACK_VAR:
        loop = var;
        ANI_CMD_MOVEBACK_FINISH:
        {
          #if !defined(OBJ_ACCURATE_MOVEMENT)
          float xd,yd;
          #endif
          float dir = (float)((180+obj_inst->dir)%360);

          loop *= obj_inst->speed;

          #if defined(OBJ_ACCURATE_MOVEMENT)
          obj_inst->real_x += fsin((float)dir * UTL_DEG2RAD)*(float)loop * 0.1;
          obj_inst->real_y -= fcos((float)dir * UTL_DEG2RAD)*(float)loop * 0.1;

          obj_inst->x = (int32)(obj_inst->real_x);
          obj_inst->y = (int32)(obj_inst->real_y);
          #else
          xd = fsin(dir * UTL_DEG2RAD)*(float)loop;
          yd = fcos(dir * UTL_DEG2RAD)*(float)loop;

          if (xd>0)
            xd += 0.5;
          else if (xd < 0)
            xd -= 0.5;

          if (yd>0)
            yd += 0.5;
          else if (yd < 0)
            yd -= 0.5;

          obj_inst->x += (int)xd;
          obj_inst->y -= (int)yd;
          #endif    
        }
        break;

      case ANI_CMD_SETSPD:
        inst->old_speed = obj_inst->speed = *(anipos++);
        break;
      case ANI_CMD_SETSPD_VAR:
        inst->old_speed = obj_inst->speed = var;
        break;
      case ANI_CMD_MODSPD:
        obj_inst->speed += *(anipos++);
        break;
      case ANI_CMD_MODSPD_VAR:
        obj_inst->speed += var;
        break;

      case ANI_CMD_SETDIR:
        obj_inst->dir = *(anipos++);
        break;
      case ANI_CMD_SETDIR_VAR:
        obj_inst->dir = var;
        break;
      case ANI_CMD_MODDIR:
        obj_inst->dir = *(anipos++);
        break;
      case ANI_CMD_MODDIR_VAR:
        obj_inst->dir = var;
        break;

      case ANI_CMD_SETZ:
        obj_inst->z = ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_SETZ_VAR:
        obj_inst->z = ((float)var*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODZ:
        obj_inst->z += ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODZ_VAR:
        obj_inst->z += ((float)var*ANI_FLOAT_MULT);
        break;
      
      case ANI_CMD_SETCOL:
        obj_inst->colour = (uint32)*(anipos++);
        break;
      case ANI_CMD_SETCOL_VAR:
        obj_inst->colour = (uint32)var;
        break;
      case ANI_CMD_MODCOL:
        obj_inst->colour += *(anipos++);
        break;
      case ANI_CMD_MODCOL_VAR:
        obj_inst->colour += var;
        break;

      case ANI_CMD_SETEOL:
        obj_inst->eol = *(anipos++);
        break;
      case ANI_CMD_SETEOL_VAR:
        obj_inst->eol = var;
        break;

      case ANI_CMD_SETROT:
        obj_inst->rot = *(anipos++);
        break;
      case ANI_CMD_SETROT_VAR:
        obj_inst->rot = var;
        break;
      case ANI_CMD_MODROT:
        obj_inst->rot += *(anipos++);
        break;
      case ANI_CMD_MODROT_VAR:
        obj_inst->rot += var;
        break;

      case ANI_CMD_SETSCALE:
        obj_inst->scale = ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_SETSCALE_VAR:
        obj_inst->scale = ((float)var*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODSCALE:
        obj_inst->scale += ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODSCALE_VAR:
        obj_inst->scale += ((float)var*ANI_FLOAT_MULT);
        break;

      case ANI_CMD_SETFADE:
        obj_inst->fade = ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_SETFADE_VAR:
        obj_inst->fade = ((float)var*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODFADE:
        obj_inst->fade += ((float)*(anipos++)*ANI_FLOAT_MULT);
        break;
      case ANI_CMD_MODFADE_VAR:
        obj_inst->fade += ((float)var*ANI_FLOAT_MULT);
        break;

      case ANI_CMD_SETX:
        obj_inst->x = *(anipos++);
        goto ANI_CMD_SETX_FINISH;
        break;
      case ANI_CMD_SETX_VAR:
        obj_inst->x = var;
        goto ANI_CMD_SETX_FINISH;
        break;
      case ANI_CMD_MODX:
        obj_inst->x += *(anipos++);
        goto ANI_CMD_SETX_FINISH;
        break;
      case ANI_CMD_MODX_VAR:
        obj_inst->x += var;
        ANI_CMD_SETX_FINISH:
        #if defined(OBJ_ACCURATE_MOVEMENT)
        obj_inst->real_x = (float)obj_inst->x;
        #endif
        break;

      case ANI_CMD_SETY:
        obj_inst->y = *(anipos++);
        goto ANI_CMD_SETY_FINISH;
        break;
      case ANI_CMD_SETY_VAR:
        obj_inst->y = var;
        goto ANI_CMD_SETY_FINISH;
        break;
      case ANI_CMD_MODY:
        obj_inst->y += *(anipos++);
        goto ANI_CMD_SETY_FINISH;
        break;
      case ANI_CMD_MODY_VAR:
        obj_inst->y += var;
        ANI_CMD_SETY_FINISH:
        #if defined(OBJ_ACCURATE_MOVEMENT)
        obj_inst->real_y = (float)obj_inst->y;
        #endif
        break;

      case ANI_CMD_SETVIEWROT:
        vid_view_set(view_x, view_y, view_w, view_h, *(anipos++));
        break;
      case ANI_CMD_SETVIEWROT_VAR:
        vid_view_set(view_x, view_y, view_w, view_h, var);
        break;
      case ANI_CMD_MODVIEWROT:
        vid_view_set(view_x, view_y, view_w, view_h, view_rot + *(anipos++));
        break;
      case ANI_CMD_MODVIEWROT_VAR:
        vid_view_set(view_x, view_y, view_w, view_h, view_rot + var);
        break;
    }
  }
  while (rc==ANI_RUN_CONTINUE);

  if (runmode == ANI_RUNMODE_TRACK)
  {
    inst->trackpos = anipos;
    inst->track_pause_until = pause_until;
    inst->track_pause_count = pause_count;
  }
  else
  {
    inst->anipos = anipos;
    inst->pause_until = pause_until;
    inst->pause_count = pause_count;
  }

  inst->stored_var = var;

  return rc;
}

/* ani_animate ______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int ani_animate(obj_inst_t *obj_inst)
{
  int rc = 0;
  
  int ani_dir;
  
  ani_inst_t *inst = (ani_inst_t *)obj_inst->var;
  
  DBG_ENTRY(99,("ani_animate(%p)",obj_inst));

  /* Don't animate again if we already did it this frame */
  if (inst->old_frame == curr_frame)
    return rc;

  if (inst->trackpos)
  {
    if (inst->track_pause_count == inst->track_pause_until)
      ani_run(obj_inst, ANI_RUNMODE_TRACK);
    else
      inst->track_pause_count++;
  }
  
CHANGE_DIRECTION:
  
  ani_dir = get_path_for_dir(obj_inst->dir);

  DBG_PRINT(99, ("anipos=%p old_dir=%d ani_dir=%d old_speed=%d speed=%d",
                 inst->anipos,
                 (int)inst->old_dir,
                 ani_dir,
                 (int)inst->old_speed,
                 (int)obj_inst->speed));

  if (inst->anipos == 0        ||
      inst->old_dir != ani_dir ||
      inst->old_speed != obj_inst->speed)
  {
    if (obj_inst->speed == 0)
    {
      inst->anipos = inst->anim->path[ANI_PATH_STOP + ani_dir];
      
      /* No direction specific static animation */
      if (!inst->anipos)
        inst->anipos = inst->anim->path[ANI_PATH_STOP];
    }
    /* Look for transitions */
    else
    {
      if (inst->old_speed == 0)
      {
        int old_path = inst->old_dir * 9;
        
        inst->anipos = inst->anim->path[ANI_PATH_STOP_TO_MOVE + old_path + ani_dir];

        if (!inst->anipos)
          inst->anipos = inst->anim->path[ANI_PATH_STOP_TO_MOVE + old_path];
          
        if (!inst->anipos)
          inst->anipos = inst->anim->path[ANI_PATH_STOP_TO_MOVE + ani_dir];
          
        if (!inst->anipos)
          inst->anipos = inst->anim->path[ANI_PATH_STOP_TO_MOVE];
      }
      else
      {
        inst->anipos = inst->anim->path[get_transition(inst->old_dir, ani_dir)];
      }
    }
    
    /* Need to use the basic direction animation */  
    if (!inst->anipos)
    {
      inst->anipos = inst->anim->path[ANI_PATH_MOVE + ani_dir];

      if (!inst->anipos)
        inst->anipos = inst->anim->path[ANI_PATH_MOVE];
    }
    
    if (!inst->anipos)
    {
      DBG_PRINT(0,("ani_dir=%d dir=%d\n",ani_dir,(int)obj_inst->dir));
      DBG_PRINT(0,("old_speed=%d speed=%d\n",(int)inst->old_speed,(int)obj_inst->speed));
      exit(0);
    }
    
    /* Cancel any waiting */      
    inst->pause_count = inst->pause_until;
  }
  
  inst->old_dir = ani_dir;
  inst->old_speed = obj_inst->speed;
  inst->old_frame = curr_frame;
  
REVERTED:
  
  if (inst->pause_count < inst->pause_until)
    inst->pause_count++;
  else if (inst->anipos)
  {
    int old_obj = inst->obj;
    int old_rot = obj_inst->rot;

    int run_rc = ani_run(obj_inst, ANI_RUNMODE_ANIM);
    
    switch(run_rc)
    {
      case ANI_RUN_REVERTED:
        goto REVERTED;
      case ANI_RUN_CHANGE_DIRECTION:
        goto CHANGE_DIRECTION;
    }

    if (old_rot != obj_inst->rot ||
        old_obj != inst->obj)
    {
      utl_free(obj_inst->mask);
      obj_inst->mask = 0;
    }
  }
  else
  {
    DBG_PRINT(0,("inst->anipos == null, need default ani"));
  }
  
  /* -------------------------------- */
  /* remember current state           */
  /* -------------------------------- */
  inst->prev_pause_count = inst->pause_count;
  inst->prev_pause_until = inst->pause_until;
  inst->prev_anipos = inst->anipos;
  
  DBG_EXIT(99,("ani_animate rc=%d", rc));
  
  return rc;
}

#undef ANI_C
