#define VID_C

#include <kos.h>
#include <dc/fmath.h>

#include "utils.h"
#include "video.h"

/* vid_screen_init __________________________________________________________ */
/*                                                                            */
/* Initialise the screen.                                                     */
/* __________________________________________________________________________ */

void vid_screen_init(int mode)
{
  pvr_init_params_t gui_params = 
  {
    { 
      PVR_BINSIZE_16,
      PVR_BINSIZE_0,
      PVR_BINSIZE_16,
      PVR_BINSIZE_0,
      PVR_BINSIZE_0
    },

    /* Vertex buffer size 512K */
    512*1024
  };

  DBG_ENTRY(99,("vid_screen_init()")); 

  switch(mode)
  {
    case 0:
    case 1:
      vid_set_mode(DM_640x480_NTSC_IL, PM_RGB565);
      vid_min_x = 300;
      vid_min_y = 30;
      vid_max_x = 920;
      vid_max_y = 255;
      break;
    case 2:
      vid_set_mode(DM_640x480_PAL_IL, PM_RGB565);
      vid_min_x = 300;
      vid_min_y = 76;
      vid_max_x = 920;
      vid_max_y = 284;
      break;
  }

  pvr_init(&gui_params);
  
  vid_playarea_set(VID_INIT_PLAYAREA_W, VID_INIT_PLAYAREA_H);
  
  vid_view_set(VID_INIT_VIEW_X,
                 VID_INIT_VIEW_Y,
                 VID_INIT_VIEW_W,
                 VID_INIT_VIEW_H,
                 VID_INIT_VIEW_ROT);

  vid_gun_screen = 0;             
  
  vid_x_delta = (float)vid_max_x-(float)vid_min_x;
  vid_y_delta = (float)vid_max_y-(float)vid_min_y;

  DBG_PRINT(0,("min %d,%d max %d,%d xd: %f yd: %f",
               vid_min_x, vid_min_y,
               vid_max_x, vid_max_y,
               vid_x_delta, vid_y_delta));

  DBG_EXIT(99,("vid_screen_init"));
}

/* vid_screen_begin_rgb _____________________________________________________ */
/*                                                                            */
/* Prime the screen for drawing.                                              */
/* __________________________________________________________________________ */

void vid_screen_begin_rgb(uint32 colour, int opaque)
{
  vid_screen_begin((float)((colour & 0xFF0000) >> 16) / 255.0,
                   (float)((colour & 0x00FF00) >> 8)  / 255.0,
                   (float)((colour & 0x0000FF)) / 255.0,
                   opaque);
}

/* vid_screen_begin _________________________________________________________ */
/*                                                                            */
/* Prime the screen for drawing.                                              */
/* __________________________________________________________________________ */

void vid_screen_begin(float red, float green, float blue, int opaque)
{ 
  DBG_ENTRY(99,("vid_screen_begin(%f,%f,%f)",(double)red,(double)green,(double)blue));
  
  VID_PERFORMANCE(0,0,0); 
  pvr_wait_ready(); 
  VID_PERFORMANCE(0,0,0);
  
  curr_frame++;

  if (vid_gun_screen)
    pvr_set_bg_color(1,1,1);
  else
    pvr_set_bg_color(red,green,blue);

  pvr_scene_begin();

  if (opaque)
    pvr_list_begin(PVR_LIST_OP_POLY);
  else
    pvr_list_begin(PVR_LIST_TR_POLY);

  DBG_EXIT(99,("vid_screen_begin"));
}

/* vid_screen_continue ______________________________________________________ */
/*                                                                            */
/* Continue if opaque polys are being used                                    */
/* __________________________________________________________________________ */

void vid_screen_continue(void)
{
  DBG_ENTRY(99,("vid_screen_continue()"));
  VID_PERFORMANCE(255,0,0);

  pvr_list_finish();
  pvr_list_begin(PVR_LIST_TR_POLY);

  DBG_EXIT(99,("vid_screen_continue"));
}

/* vid_screen_end ___________________________________________________________ */
/*                                                                            */
/* Finalise this screen.                                                      */
/* __________________________________________________________________________ */

void vid_screen_end(void)
{
  DBG_ENTRY(99,("vid_screen_end()"));
 
  pvr_list_finish();
  pvr_scene_finish();

  vid_gun_screen = 0;

  DBG_EXIT(99,("vid_screen_end"));
}

/* vid_screen_term __________________________________________________________ */
/*                                                                            */
/* Terminate the screen.                                                      */
/* __________________________________________________________________________ */

void vid_screen_term(void)
{
  DBG_ENTRY(99,("vid_screen_term()"));
  
  pvr_shutdown();
  
  DBG_EXIT(99,("vid_screen_term"));
}

/* vid_view_set  ____________________________________________________________ */
/*                                                                            */
/* Set up the view screen.                                                    */
/* __________________________________________________________________________ */

void vid_view_set(int x, int y, int w, int h, int rot)
{
  DBG_ENTRY(99,("vid_view_set(%d,%d,%d,%d,%d)",x,y,w,h,rot));
  
  view_w = w;
  view_h = h;
  view_half_w = w/2;
  view_half_h = h/2;

  view_scale = ((float)VID_SCREEN_W/(float)view_w);

  if (view_scale == 1)
    view_scale = ((float)VID_SCREEN_H/(float)view_h);
  
  w = w>>1;
  h = h>>1;

  if (x-w > 0)
  {
    if (x+w < vid_playarea_w)
      view_x = x-w;
    else
      view_x = vid_playarea_w-view_w;
  }
  else
    view_x = 0;
    
  if (y-h > 0)
  {
    if (y+h < vid_playarea_h)
      view_y = y-h;
    else
      view_y = vid_playarea_h-view_h;
  }
  else
    view_y = 0;
   
  view_mid_x  = (float)(view_x + w);
  view_mid_y  = (float)(view_y + h);
        
  view_rot = rot;
  view_rcos = fcos((float)view_rot * UTL_DEG2RAD);
  view_rsin = fsin((float)view_rot * UTL_DEG2RAD);
  
  DBG_EXIT(99,("vid_view_set"));
}

/* vid_playarea_set  ________________________________________________________ */
/*                                                                            */
/* Set up the play area.                                                      */
/* __________________________________________________________________________ */

void vid_playarea_set(int w, int h)
{
  DBG_ENTRY(99,("vid_playarea_set(%d,%d)",w,h));

  vid_playarea_w = w;
  vid_playarea_h = h;

  DBG_EXIT(99,("vid_playarea_set"));
}

/* vid_box __________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void vid_box(int x, int y, int w, int h, float fade, float z, uint32 colour)
{
  pvr_poly_hdr_t hdr;
	pvr_poly_cxt_t cxt;
  pvr_vertex_t vert;

  pvr_poly_cxt_col(&cxt, PVR_LIST_TR_POLY);    
  cxt.gen.shading = PVR_SHADE_FLAT;
  pvr_poly_compile(&hdr, &cxt);
  
  pvr_prim(&hdr, sizeof(hdr));
  
  vert.flags = PVR_CMD_VERTEX;
  vert.x = x;
  vert.y = y;
  vert.z = z;
  vert.u = vert.v = 0;
  vert.argb = (((uint8)( fade * 255 ) ) << 24 ) | colour;
  vert.oargb = 0;
  pvr_prim(&vert, sizeof(vert));
  
  vert.x += w;
  vert.u = 1;
  pvr_prim(&vert, sizeof(vert));
  
  vert.x = x;
  vert.y += h;
  vert.u = 0;
  vert.v = 1;
  pvr_prim(&vert, sizeof(vert));
  
  vert.flags = PVR_CMD_VERTEX_EOL;
  vert.x += w;
  vert.u = 1;
  pvr_prim(&vert, sizeof(vert));

  return;
}

/* vid_shadow _______________________________________________________________ */
/*                                                                            */
/* Cast a shadow over the screen                                              */
/* __________________________________________________________________________ */

void vid_shadow(float fade, float z, uint32 colour)
{
  DBG_ENTRY(99,("vid_shadow(%f,%f,0x%08X)",(double)fade,(double)z,(unsigned int)colour));
  
  vid_box(view_x, view_y, view_w, view_h, fade, z, colour);

  DBG_EXIT(99,("vid_shadow"));
}

#undef VID_C
