#define BTIM_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <math.h>
#include <string.h>

#include "object.h"
#include "utils.h"
#include "btime.h"
#include "video.h"
#include "cont.h"
#include "font.h"
#include "sound.h"
#include "anim.h"

#include "pal.h"
    
KOS_INIT_FLAGS(INIT_DEFAULT);

#if defined(USE_ROMDISK)
extern uint8 romdisk[];
KOS_INIT_ROMDISK(romdisk);
#endif

#if defined(_WIN32)
int round_to_start_on = START_ROUND;
#else
#define round_to_start_on START_ROUND
#endif

cnt_info_t cont[MAPLE_PORT_COUNT+1];

void check_compass(obj_inst_t  *inst,
                   obj_inst_t **north,
                   obj_inst_t **south,
                   obj_inst_t **east,
                   obj_inst_t **west)
{
  if (inst)
  {
    obj_inst_t  point_test;
   
    int ch = obj_inst_getheight(inst)/2;
    int cw = (obj_inst_getwidth(inst)/2)+1;

    point_test.z = 1.0f;        
    point_test.x = inst->x;
    point_test.y = inst->y + ch - 2;
    
    if (north)    
      *north = obj_inst_collide_fs(&point_test,
                                   obj_lists[OBJ_LIST_BOUNDARY_1], /* LADDERS */
                                   OBJ_COLL_YESNO | OBJ_COLL_POINT);
        
    point_test.y = inst->y + ch;         
        
    if (south)
      *south = obj_inst_collide_fs(&point_test,
                                   obj_lists[OBJ_LIST_BOUNDARY_1], /* LADDERS */
                                   OBJ_COLL_YESNO | OBJ_COLL_POINT);
        
    point_test.x = inst->x + cw;
    point_test.y = inst->y + ch - 2;

    if (east)
      *east = obj_inst_collide_fs(&point_test,
                                  obj_lists[OBJ_LIST_BOUNDARY_2], /* PLATFORMS */
                                  OBJ_COLL_YESNO | OBJ_COLL_POINT);
        
    point_test.x = inst->x - cw;
    
    if (west)    
      *west = obj_inst_collide_fs(&point_test,
                                  obj_lists[OBJ_LIST_BOUNDARY_2], /* PLATFORMS */
                                  OBJ_COLL_YESNO | OBJ_COLL_POINT);
  }
}

void round_load(FILE *fp)
{
  char line[1024];
    
  DBG_ENTRY(99,("round_load(%p)", fp));
 
  bg_music = DEF_BG_MUSIC;
  bg_colour = DEF_BG_COLOUR;

  while (fgets(line,sizeof(line),fp))
  {
    char *t = strtok(line, UTL_WHITESPACE);
   
    if (!t || *t == '#')
      continue;
    
    switch(*t)
    {
      case 'L':
        if ((t=strtok(0, UTL_WHITESPACE)))
          obj_inst_list_load(t);
        break;
      case 'T':
      {
        int y;

        if ((t=strtok(0, UTL_WHITESPACE)))
        {
          y = MAX_BURGERS*((int)strtol(t,0,0));
        
          while((t=strtok(0, UTL_WHITESPACE)))
            terminal_y[y++] = (int)strtol(t,0,0);
        }
      }
      break;
      case 'B':
        if ((t=strtok(0, UTL_WHITESPACE)))
          bottom_line = (int)strtol(t,0,0);
        break;
      case 'M':
        if ((t=strtok(0, UTL_WHITESPACE)))
          bg_music = snd_bg_load(t);
        break;
      case 'C':
        if ((t=strtok(0, UTL_WHITESPACE)))
          bg_colour = strtoul(t,0,16);
        break;

    }
  }
    
  memset(&p2b[0], 0x00, sizeof(p2b));
  memset(&bounce[0], 0x00, sizeof(bounce));

  DBG_EXIT(99,("round_load"));
}

void round_unload(void)
{
  int list;
  
  DBG_ENTRY(99,("round_unload()"));
 
  for(list=0; list<OBJ_MAX_LISTS; list++)
  {
    obj_inst_free(obj_lists[list]);
    obj_lists[list] = 0;
  }
  
  snd_bg_unload(bg_music);
  
  DBG_EXIT(99,("round_unload"));
}

#define CUT_SCENE_FRAMES 196

void round_cut_scene(FILE *fp, int round, fnt_info_t *font)
{
  pvr_poly_hdr_t hdr;
	pvr_poly_cxt_t cxt;
  pvr_vertex_t vert;

  int cut_scene_frames = CUT_SCENE_FRAMES;
  char roundName[20];
  int outer_index = 0;
  float rotation = 0;
  float rot_delta = (RANDOM_NUMBER*5)-2.5;
  float rot_delta_delta = (RANDOM_NUMBER*0.03)-0.015;
  
  DBG_ENTRY(99,("round_cut_scene(%p,%d,%p)", fp, round, font));
  
  /* -------------------------- */
  /* This could become a thread */
  /* -------------------------- */
  round_load(fp);

  #if !defined(_WIN32)
  if (round > 1)
  {
    #if defined(_WIN32)
    {
      pvr_ptr_t fake_tex = pvr_mem_malloc(1);
  
      if (fake_tex)
      {
        uint8 *x = (uint8 *)fake_tex;
        x[0] = 0xFF; /*BA*/
        x[1] = 0xFF; /*GR*/
  
        DBG_PRINT(99,("fake_tex = %p\n", fake_tex));
  
        pvr_poly_cxt_txr(&cxt,
                         PVR_LIST_OP_POLY,
                         PVR_TXRFMT_ARGB4444,
                         1,1,
                         fake_tex,
                         PVR_FILTER_BILINEAR);
      }
    }
    #else
    pvr_poly_cxt_col(&cxt, PVR_LIST_OP_POLY);
    #endif
    
    cxt.gen.shading = PVR_SHADE_FLAT;
    pvr_poly_compile(&hdr, &cxt);
  
    sprintf(roundName,"ROUND %d", round);
  
    /* ----------------------- */
    /* Play the LEVEL UP sound */
    /* ----------------------- */
    obj_render(115, 320, 240, 0);
    
    while(cut_scene_frames-- && check_win32())
    {
      float z  = 1.0;
      float hwd = 320;
      float hht = 240;
      int   index = outer_index++;
      
      vid_screen_begin(0,0,0,1);
      
      rotation = 0;
      
      while(hht>0)
      {
        int col = 0xFF000000;
        
        float rcos = fcos(rotation * UTL_DEG2RAD);
        float rsin = fsin(rotation * UTL_DEG2RAD);
      
        float rcn = rcos*-hwd;     /* pre-calculate values */
        float rcp = rcos*hwd;
        float rco = rcos*-hht;
        float rcq = rcos*hht;
        float rsn = rsin*-hwd;
        float rsp = rsin*hwd;
        float rso = rsin*-hht;
        float rsq = rsin*hht;
        
        if (index > 46 && index < CUT_SCENE_FRAMES-48)
          col |= utl_rainbow_palette[(index-46)%16];
        
        index++;
        
        rotation += rot_delta;
     
        pvr_prim(&hdr, sizeof(hdr));
  
        vert.flags = PVR_CMD_VERTEX;
        vert.x = 320+(rcn+rsq);
        vert.y = 240-(rcq-rsn);
        vert.z = z;
        vert.u = vert.v = 0;
        vert.argb = col;
        vert.oargb = 0;
        pvr_prim(&vert, sizeof(vert));
  
        vert.x = 320+(rcp+rsq);
        vert.y = 240-(rcq-rsp);
        vert.u = 1;
        pvr_prim(&vert, sizeof(vert));
  
        vert.x = 320+(rcn+rso);
        vert.y = 240-(rco-rsn);
        vert.u = 0;
        vert.v = 1;
        pvr_prim(&vert, sizeof(vert));
  
        vert.flags = PVR_CMD_VERTEX_EOL;
        vert.x = 320+(rcp+rso);
        vert.y = 240-(rco-rsp);
        vert.u = 1;
        pvr_prim(&vert, sizeof(vert));
        
        hwd-=5;
        hht-=5;
        z+=0.1;
      }
  
      rot_delta += rot_delta_delta;
  
      vid_screen_continue();
  
      fnt_render(font, roundName,
                 320, 240, 11, 0,
                 1,  1, 0x000000,
                 4, 1, 0xFFFFFF);
      
      vid_screen_end();
    }
  }
  #endif
  
  DBG_EXIT(99,("round_cut_scene"));
}

/* start_round ______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int start_round(FILE *fp, int round, fnt_info_t *font)
{
  int i;
  int frames_this_round = 0;
  int new_round = round;
  int alive = playing;
  int round_activity = 0;
  int prev_round_activity = -5;

  int prev_at_max = 0;
  int total_parts = 0;

  obj_inst_t  point_test;
  
  obj_inst_t *curr;
  
  obj_inst_t *north = 0;
  obj_inst_t *south = 0;
  obj_inst_t *east = 0;
  obj_inst_t *west = 0;

  obj_inst_t *target = 0;

  round_cut_scene(fp, round, font);

  point_test.z = 1.0f;

  /* ------------------------------------------------------- */
  /* Count total number of burger ingredients this round has */
  /* ------------------------------------------------------- */
  for(i=OBJ_LIST_BURGER_START;i<=OBJ_LIST_BURGER_END;i++)
    if (obj_lists[i])
      total_parts++;

  /* --------------------------------------------------- */
  /* Calculate number of parts for number of ingredients */
  /* --------------------------------------------------- */
  total_parts = total_parts<<4;

  while(new_round == round && check_win32())
  {
    int         at_max = 0; /* number of burger parts at terminal location */
    obj_inst_t *coll   = 0;
    obj_inst_t *tmp    = 0;
    int         list;
    int         player;
    int         dir_change;

    cnt_state((cnt_info_t *)&cont); 

    vid_screen_begin_rgb(bg_colour, 0);

    /* --------------- */
    /* Player movement */
    /* --------------- */

    VID_PERFORMANCE(255,0,0);

    switch(round_activity)
    {
      /* ----------- */
      /* Normal play */
      /* ----------- */
      case 0:
      {
        for(list=OBJ_LIST_PLAYER_START;list<OBJ_LIST_PLAYER_START+playing;list++)
        {
          player = list-OBJ_LIST_PLAYER_START;

          if (obj_lists[list])
          {
            int new_dir = -1;
            int new_speed = 10;
            int bblist = 0;
    
            curr = obj_lists[list];
    
            /* -------------------------------- */
            /* Check for vicinity of Generators */
            /* -------------------------------- */
            coll = obj_lists[OBJ_LIST_GENERATORS];

            while(coll)
            {
              if (curr->x < coll->x-32 ||
                  curr->x > coll->x+32 ||
                  curr->y < coll->y-32 ||
                  curr->y > coll->y+32)
                coll->z = 0;
              else
                coll->z = -1;

              coll = coll->next;              
            }

            /* ------------------------------------------- */
            /* Ensure that we count down any current mode  */
            /* ------------------------------------------- */
            if (mode_countdown[player] > 0)
              mode_countdown[player]--;

            /* -------------------------------------- */
            /* If we are currently accepting input... */
            /* -------------------------------------- */
            if (curr->mode != OBJ_MODE_NO_INPUT)
            {
              if ((curr->mode != OBJ_MODE_STANDARD) && (mode_countdown[player] <= 0))
              {
                curr->mode = OBJ_MODE_STANDARD;
                obj_inst_setevent(curr, OBJ_EVENT_MODECHANGE);
              }

              /* -------------------------------- */
              /* Check for collision with baddies */
              /* -------------------------------- */
              if ((coll = obj_inst_collide(curr,
                                           obj_lists[OBJ_LIST_BADDIES],
                                           OBJ_COLL_YESNO)) && coll->speed)
              {
                /* ---------------------------------------------------------- */
                /* If not in bonus mode, player dies, otherwise... coll dies! */
                /* ---------------------------------------------------------- */
                if ((bonus[player] & BONUS_ALL) != BONUS_ALL)
                {
                  old_player[player] = curr;
                
                  obj_lists[list] = obj_inst_remove(obj_lists[list],
                                                    curr,
                                                    0);
                  
                  obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                curr);
                                                              
                  obj_inst_setevent(curr, OBJ_EVENT_DEAD);
                }
                else
                {
                  score[player] += obj_inst_getattribute(coll, OBJ_ATTR_SCORE);

                  /* Move to the incidental list */

                  obj_lists[OBJ_LIST_BADDIES] = obj_inst_remove(obj_lists[OBJ_LIST_BADDIES],
                                                                coll,
                                                                0);

                  obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                coll);

                  obj_inst_setevent(coll, OBJ_EVENT_DEAD);
                }
              }
              /* ------------------------------------- */
              /* Check for collision with baddie shots */
              /* ------------------------------------- */
              else if ((coll = obj_inst_collide(curr,
                                                obj_lists[OBJ_LIST_BADDIES_SHOTS],
                                                OBJ_COLL_YESNO)))
              {
                dir_change = obj_inst_getattribute(coll, OBJ_ATTR_DIRCHANGE);

                switch(dir_change)
                {
                  case OBJ_EVENT_ALARMED:
                    coll->eol = obj_inst_getattribute(coll, OBJ_ATTR_SCORE);
                    target = coll;
                    break;
                }

                if (dir_change != 0)
                  obj_inst_setevent(curr, dir_change);

                /* Move to the incidental list */

                obj_lists[OBJ_LIST_BADDIES_SHOTS] = obj_inst_remove(obj_lists[OBJ_LIST_BADDIES_SHOTS],
                                                                    coll,
                                                                    0);

                obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                              coll);

                obj_inst_setevent(coll, OBJ_EVENT_DEAD);
              }
              else
              {
                int buttons = cont[player].buttons;
                
                if (buttons & CONT_A)
                {
                  if (curr->dir == 90 || curr->dir == 270)
                  {
                    switch(curr->mode)
                    {
                      default:
                        if ((cont[player].prev_buttons & CONT_A) || pepper[player] <= 0)
                          goto NO_SHOOT;
                        else if (!(bonus[player] & INFINITE_PEPPER))
                          pepper[player]--;
                        break;
                      case OBJ_MODE_HAND_GUN:
                        if (cont[player].prev_buttons & CONT_A)
                          goto NO_SHOOT;
                        break;
                      case OBJ_MODE_MACHINE_GUN:
                        break;
                    }

                    obj_inst_setevent(curr, OBJ_EVENT_SHOOT);

                    NO_SHOOT:;
                  }
                }

                if ((buttons & CONT_START) &&
                    !(cont[player].prev_buttons & CONT_START))
                {
                  DBG_PRINT(99,("X=%d Y=%d", curr->x, curr->y));
                  round_activity = -2;
                }

                check_compass(curr, &north, &south, &east, &west);
                
                /* ------------------ */
                /* Check for movement */
                /* ------------------ */
                if (east && (buttons & CONT_DPAD_RIGHT))
                  new_dir = 90;
                else if (west && (buttons & CONT_DPAD_LEFT))
                  new_dir = 270;
                else if (north && (buttons & CONT_DPAD_UP))
                  new_dir = 0;
                else if (south && (buttons & CONT_DPAD_DOWN))
                  new_dir = 180;
                
                if (new_dir != curr->dir)
                {
                  switch(new_dir)
                  {
                    case 90:
                      curr->y = east->y - 26;
                      break;
                    case 270:
                      curr->y = west->y - 26;
                      break;
                    case 0:
                      curr->x = north->x;
                      break;
                    case 180:
                      curr->x = south->x;
                      break;
                    default:
                      new_speed = 0;
                  }
                }

                /* --------------------------------------- */
                /* Are we colliding with any burger bits ? */
                /* --------------------------------------- */
                if (new_speed && (new_dir == 90 || new_dir == 270))
                {
                  for(bblist=OBJ_LIST_BURGER_START;bblist<=OBJ_LIST_BURGER_END;bblist++)
                  {
                    if (obj_lists[bblist] && 
                        obj_lists[bblist]->y > curr->y &&
                        obj_lists[bblist]->y < terminal_y[bblist/MAX_BURGER_BITS] &&
                        !bounce[bblist-OBJ_LIST_BURGER_START])
                    {
                      int         start_num   = 8;
                      obj_inst_t *coll        = 0;
                      obj_inst_t *start_point = obj_lists[bblist];
                    
                      while(start_point && start_num-- > 0)
                        start_point = start_point->next;
    
                      if ((coll = obj_inst_collide_fs(curr,
                                                      start_point,
                                                      OBJ_COLL_YESNO)))
                      { 
                        int x = coll->x;
    
                        coll = obj_lists[bblist];
    
                        while(coll)
                        {
                          if (coll->x == x)
                            coll->y++;
                          coll = coll->next;
                        }
    
                        // make a note in the player-2-burger array
                        p2b[bblist-OBJ_LIST_BURGER_START] = (list+1) - OBJ_LIST_PLAYER_START;
    
                        bblist = OBJ_LIST_BURGER_END + 1;
                      }
                    }
                  }
                }
    
                /* --------------------------------------- */
                /* Are we colliding with any collectables? */
                /* --------------------------------------- */
                if ((coll = obj_inst_collide_fs(curr,
                                                obj_lists[OBJ_LIST_COLLECTABLES],
                                                OBJ_COLL_YESNO)))
                {
                  int coll_score = obj_inst_getattribute(coll, OBJ_ATTR_SCORE);

                  switch(coll_score)
                  {
                    case -BONUS_B:
                    case -BONUS_O:
                    case -BONUS_N:
                    case -BONUS_U:
                    case -BONUS_S:
                    case -INFINITE_PEPPER:
                    case -CHILLI_PEPPER:
                    case -DISTANT_PEPPER:
                      bonus[player] |= -coll_score;
                      break;
                    case -EXTRA_PEPPER:
                      pepper[player]++;
                      break;
                    case -EXTRA_LIFE:
                      lives[player]++;
                      break;
                    case -HAND_GUN:
                      curr->mode = OBJ_MODE_HAND_GUN;
                      obj_inst_setevent(curr, OBJ_EVENT_MODECHANGE);
                      mode_countdown[player] = obj_inst_getattribute(coll,
                                                                     OBJ_ATTR_DIRCHANGE);
                      break;
                    case -MACHINE_GUN:
                      curr->mode = OBJ_MODE_MACHINE_GUN;
                      obj_inst_setevent(curr, OBJ_EVENT_MODECHANGE);
                      mode_countdown[player] = obj_inst_getattribute(coll,
                                                                     OBJ_ATTR_DIRCHANGE);
                      break;
                    default:
                      score[player] += coll_score;
                      break;
                  }
                

                  /* Move to the incidental list */

                  obj_lists[OBJ_LIST_COLLECTABLES] = obj_inst_remove(obj_lists[OBJ_LIST_COLLECTABLES],
                                                                     coll,
                                                                     0);

                  obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                coll);

                  obj_inst_setevent(coll, OBJ_EVENT_DEAD);
                }
              
                /* ----------------- */
                /* Set the new speed */
                /* ----------------- */
                if ((new_dir != -1 && new_dir != curr->dir) ||
                    new_speed != curr->speed)
                {
                  while(curr)
                  {
                    if (new_dir != -1)
                      curr->dir = new_dir;
                    curr->speed = new_speed;
                    curr = curr->next;
                  }
                }                
              }
            }
          }
        }

        /* ---------------- */
        /* Sort out baddies */
        /* ---------------- */
        curr = obj_lists[OBJ_LIST_BADDIES];
        
        while(curr)
        {
          tmp = 0;
          dir_change = obj_inst_getattribute(curr, OBJ_ATTR_DIRCHANGE);
    
          if (curr->speed)
          {
            for(list=OBJ_LIST_SHOT_START;list<OBJ_LIST_SHOT_START+playing;list++)
            {
              if ((coll = obj_inst_collide_fs(curr,
                                             obj_lists[list],
                                             OBJ_COLL_YESNO)))
              {
                tmp = curr->next;

                dir_change = obj_inst_getattribute(coll, OBJ_ATTR_DIRCHANGE);

                score[list-OBJ_LIST_SHOT_START] += obj_inst_getattribute(curr, OBJ_ATTR_SCORE);

                if (dir_change != OBJ_EVENT_PEPPERED)
                {
                  /* Move to the incidental list */

                  obj_lists[OBJ_LIST_BADDIES] = obj_inst_remove(obj_lists[OBJ_LIST_BADDIES],
                                                                curr,
                                                                0);

                  obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                curr);

                  obj_inst_setevent(coll, OBJ_EVENT_DEAD);
                }

                obj_inst_setevent(curr, dir_change);

                curr = tmp;

                goto NEXT_BADDIE;
              }
            }

            check_compass(curr, &north, &south, &east, &west);
    
            /* ---------------------------------------------------------- */
            /* If direction change is odd, can change direction any time  */
            /* if even, can only do it at a 'crossroads'.                 */
            /* Don't make them change direction if player has 'B-O-N-U-S' */
            /* ---------------------------------------------------------- */
            if (target == 0)
            {
              if ((RANDOM_NUMBER * 1000) > dir_change)
              {
                if ((dir_change % 2) ||
                    ((north || south) && (west || east)))
                {
                  int follow_player = (int)(RANDOM_NUMBER * (playing-1));

                  if ((bonus[follow_player] & BONUS_ALL) != BONUS_ALL)
                    tmp = obj_lists[OBJ_LIST_PLAYER_START + follow_player];
                }
              }  
            }
            else
            {
              if (target->eol == 0)
                target = 0;
              else if ((north || south) && (west || east))
                tmp = target;
            }

            if (tmp && tmp->z)
            {
              int ch = obj_inst_getheight(curr)/2;
              int cw = obj_inst_getwidth(curr)/2;
                
              int tx, ty, cx, cy, dx, dy;
                
              cx = curr->x + cw;
              cy = curr->y + ch;
              tx = tmp->x + obj_inst_getwidth(tmp)/2;
              ty = tmp->y + obj_inst_getheight(tmp)/2;
                
              dx = cx-tx;
              dy = cy-ty;

              if (target && abs(dx) >= abs(dy) &&
                  ((dx <= 0 && east) || (dx > 0 && west)))
                north = south = 0;

              if (north && ty < cy)
              {
                curr->dir = 0;
                curr->x = north->x;
              }
              else if (south && ty > cy)
              {
                curr->dir = 180;
                curr->x = south->x;
              }
              else if (west && tx < cx)
              {
                curr->dir = 270;
                curr->y = west->y - ch-10;
              }
              else if (east && tx > cx)
              {
                curr->dir = 90;
                curr->y = east->y - ch-10;
              }
            }
              
            switch(curr->dir)
            {
              case 0:
                if (north)
                  break;
              case 90:
                if (!east)
                  curr->dir = 270;
                else
                  curr->dir = 90;
                break;
              case 180:
                if (south)
                  break;
              case 270:
                if (!west)
                  curr->dir = 90;
                else
                  curr->dir = 270;
                break;
            }
          }

          curr = curr->next;

          NEXT_BADDIE:;
        }
        
        VID_PERFORMANCE(255,255,0);
        
        for(list = OBJ_LIST_BURGER_START; list <= OBJ_LIST_BURGER_END; list++)
        {
          int bit = 0;
          
          int max_y = terminal_y[list-OBJ_LIST_BURGER_START];
          
          int touching_drop = 0;
          
          /* -------------------------------------------- */
          /* Are any burger bits now falling or stopping? */
          /* -------------------------------------------- */
          curr = obj_lists[list];
    
          if (!curr)
            continue;
    
          if (bounce[list-OBJ_LIST_BURGER_START])
            bounce[list-OBJ_LIST_BURGER_START]--;
          else
          {
            while(curr)
            {
              int         x;
              obj_inst_t *coll;
              
              if (curr->y >= max_y)
              {
                if (curr->speed != 0)
                {
                  x = curr->x; 
                  
                  coll = obj_lists[list];
      
                  while(coll)
                  {
                    if (coll->x == x)
                      coll->speed = 0;
                    coll = coll->next;
                  }
                }
                
                at_max++;
                touching_drop++; /* FAKE STOPPING */
              }
              else if (bit > 7) 
              {
                if (obj_inst_collide_fs(curr, 
                                        obj_lists[OBJ_LIST_BOUNDARY_4], /* STOPLINES */
                                        OBJ_COLL_YESNO))
                {
                  x = curr->x;
                  
                  coll = obj_lists[list];
                  
                  while(coll)
                  {
                    if (coll->x == x)
                      coll->speed = 0;
                    coll = coll->next;
                  }
                }
      
                if (!touching_drop && 
                    obj_inst_collide_fs(curr,
                                        obj_lists[OBJ_LIST_BOUNDARY_3], /* DROPLINES */
                                        OBJ_COLL_YESNO))
                  touching_drop++;
      
                if (curr->speed && curr->y < max_y)
                {
                  int plist = OBJ_LIST_PLAYER_START;
                  
                  int start_num = bit;
                  int check_list = obj_lists[list+4] ? list+4 : list+8;
                  obj_inst_t *start_point = obj_lists[check_list];
                  obj_inst_t *list_used = start_point;
                  
                  /* find the corresponding top of the part below... */
                  while(start_point && start_num-- > 0)
                    start_point = start_point->next;
      
                  if ((coll = obj_inst_collide_fs(curr,
                                                  start_point,
                                                  OBJ_COLL_YESNO | OBJ_COLL_CHECKONE)))
                  {
                    int dy = (curr->speed)/10;
                    x = coll->x;
                    
                    coll = list_used;
                    
                    while(coll)
                    {
                      if (coll->x == x)
                        coll->y += dy;
                      coll = coll->next;
                    }
      
                    // Pass on the value in the p2b list
                    p2b[check_list-OBJ_LIST_BURGER_START] = p2b[(list-OBJ_LIST_BURGER_START)];
                  }
                
                  if (!coll)
                  {
                    // hitting a bad guy...
                    if ((coll = obj_inst_collide_fs(curr,
                                                    obj_lists[OBJ_LIST_BADDIES],
                                                    OBJ_COLL_YESNO)))
                    {
                      if (p2b[list-OBJ_LIST_BURGER_START])
                        score[p2b[list-OBJ_LIST_BURGER_START]-1] += obj_inst_getattribute(coll, OBJ_ATTR_SCORE);
                    
                      bounce[list-OBJ_LIST_BURGER_START] = START_BOUNCE;
      
                      /* Move to the incidental list */
                    
                      obj_lists[OBJ_LIST_BADDIES] = obj_inst_remove(obj_lists[OBJ_LIST_BADDIES],
                                                                    coll,
                                                                    0);
                    
                      obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                    coll);
                                                                  
                      obj_inst_setevent(coll, OBJ_EVENT_CRUSHED);
                    }
                  
                    // hitting a player...
                    for(plist=OBJ_LIST_PLAYER_START;plist<=OBJ_LIST_PLAYER_END;plist++)
                    {
                      if (obj_lists[plist] != curr &&
                          (coll = obj_inst_collide_fs(curr,
                                                      obj_lists[plist],
                                                      OBJ_COLL_YESNO)))
                      {
                        if (p2b[list-OBJ_LIST_BURGER_START])
                          score[p2b[list-OBJ_LIST_BURGER_START]-1] += obj_inst_getattribute(coll, OBJ_ATTR_SCORE);
      
                        bounce[list-OBJ_LIST_BURGER_START] = START_BOUNCE;
    
                        old_player[plist-OBJ_LIST_PLAYER_START] = coll;
                  
                        obj_lists[plist] = obj_inst_remove(obj_lists[plist],
                                                           coll,
                                                           0);
                    
                        obj_lists[OBJ_LIST_INCIDENTAL] = obj_inst_add(obj_lists[OBJ_LIST_INCIDENTAL],
                                                                    coll);
                                                                
                        obj_inst_setevent(coll, OBJ_EVENT_CRUSHED);
                      }
                    }
                  }
                }
              }
              
              curr = curr->next;
      
              bit++;
            }
          }
    
          curr = obj_lists[list];  
          
          
          if (!touching_drop)
          {
            int a = 0;
            
            /* ------------ */
            /* Play a floop */
            /* ------------ */
            if (!curr->speed)
            {
              obj_inst_add_default(obj_inst_create(FLOOP_NOISE,
                                                   curr->x,
                                                   curr->y,
                                                   curr->z,
                                                   0,0,0,0,0,0));
            }

            while(curr)
            {
              int yd[3] = {2,4,6};
              int sd    = 20;
    
              int b = a++%8;
              
              if (bounce[list-OBJ_LIST_BURGER_START])
              {
                curr->speed = 0;
    
                sd = 10;
    
                switch(bounce[list-OBJ_LIST_BURGER_START])
                {
                  case START_BOUNCE:      
                    curr->z += 0.1;        
                  case (START_BOUNCE/4)*3:
                    curr->dir = 0;
                    yd[0]=-1;
                    yd[1]=-2;
                    yd[2]=-3;
                    break;
                  case START_BOUNCE/4:
                  case START_BOUNCE/2:
                    curr->dir = 180;
                    yd[0]=1;
                    yd[1]=2;
                    yd[2]=3;
                    break;
                  case 1:
                    curr->z -= 0.1;
                  default:
                    yd[0]=yd[1]=yd[2]=0;
                    break;
                }
              }
              else
              {
                curr->y += curr->y % 2;
              }
    
              if (!curr->speed)
              {
                if (b==0 || b==7)
                  curr->y += yd[0];
                else if (b==1 || b==6)
                  curr->y += yd[1];
                else
                  curr->y += yd[2];
    
                curr->speed =  sd;
              }
              
              curr = curr->next;
            }
          }
    
          // This burger bit has just hit the bottom...
          if (at_max>>4 > prev_at_max)
          {
            // Attribute the score...
            if (p2b[list-OBJ_LIST_BURGER_START])
              score[p2b[list-OBJ_LIST_BURGER_START]-1] += 250;
    
            prev_at_max = at_max>>4;
          }
        }
      }
      break; /* round_activity == 0 */

      /* --------- */
      /* GAME OVER */
      /* --------- */
      case -1:
        new_round = 0;
        break;

      /* ----- */
      /* PAUSE */
      /* ----- */
      case -2:
        //vid_view_set(view_mid_x, view_mid_y, view_w, view_h, (view_rot+1)%360);
        
        for(list=OBJ_LIST_PLAYER_START;list<=OBJ_LIST_PLAYER_END;list++)
        {
          int player = list-OBJ_LIST_PLAYER_START;

          if (obj_lists[list])
          {
            if (((cont[player].buttons & CONT_A) &&
                !(cont[player].prev_buttons & CONT_A)))
              obj_inst_list_save("TEMP");

            if (((cont[player].buttons & CONT_START) &&
                !(cont[player].prev_buttons & CONT_START)))
              round_activity = 0;
          }
        }
        break;

      /* ------------ */
      /* ROUND CHANGE */
      /* ------------ */
      default: 
        new_round = round_activity;
        break;
    }

    //DBG_SET_LEVEL(0);
  
    VID_PERFORMANCE(0,255,0);
    
    /* ----------------- */
    /* Render everything */
    /* ----------------- */
    for(list=0; list<OBJ_MAX_LISTS; list++)
    {
      curr = obj_lists[list];
        
      while(curr)
      {
        obj_inst_t *next = curr->next;

        if (round_activity == 0)
          obj_inst_animate(curr);

        if (obj_inst_render(curr,round_activity==0) == -1)
        {
          int player;
          
          for(player=0; player<playing; player++)
          {
            if (curr == old_player[player])
            {
              old_player[player] = 0;
              
              if (!--lives[player])
                alive--;
                
              break;
            }
          }
          
          obj_lists[list] = obj_inst_remove(obj_lists[list],
                                            curr,
                                            1);
        }
          
        curr = next;
      }
    }
    
    VID_PERFORMANCE(0,0,255);
    
    fnt_render(font, "1UP",
               180, 27, 2, 0,
               0.70, 1, 0xFF0000,
               1, 1, 0xE00000);
              
    fnt_render(font, "HI-SCORE",
               320, 27, 2, 0,
               0.70, 1, 0xFF0000,
               1, 1, 0xE00000);
              
    fnt_render(font, "PEPPER",
               490, 27, 2, 0,
               0.70, 1, 0x0FDD0F,
               1, 1, 0x00C000);

    /* TEMP */
    {
      char scoreSize = (char)(16.0 * view_scale);
      char scoreText[20];
      
      if (showscore[0] < score[0])
        showscore[0] = UTL_MIN(showscore[0] + 10, score[0]);
      
      sprintf(scoreText, "\7%c% 6d", scoreSize, showscore[0]);
      fnt_render(font, scoreText,
                 160, 47, 2, 0,
                 0.70, 1, 0xFFFFDF,
                 1, 4, 0x0000000);

      sprintf(scoreText,"\7%c%c%c%c%c%c", scoreSize,
              bonus[0] & BONUS_B ? 'B':' ',
              bonus[0] & BONUS_O ? 'O':' ',
              bonus[0] & BONUS_N ? 'N':' ',
              bonus[0] & BONUS_U ? 'U':' ',
              bonus[0] & BONUS_S ? 'S':' ');
      fnt_render(font, scoreText,
                 180, 71, 2, 0,
                 0.70, 1, 0x0FDF0F,
                 3, 1, 0x0000000);
                
      sprintf(scoreText, "\7%c%d", scoreSize, UTL_MAX(score[0],hiscores[0]));
      fnt_render(font, scoreText,
                 320, 47, 2, 0,
                 0.70, 1, 0xFFFFDF,
                 1, 4, 0x000000);

      if (!(bonus[0] & INFINITE_PEPPER))
        sprintf(scoreText, "\7%c% 3d", scoreSize, pepper[0]);
      else
        sprintf(scoreText, "\7%c?", scoreSize);

      fnt_render(font, scoreText,
                 500, 47, 2, 0,
                 0.70, 1, 0xFFFFDF,
                 1, 4, 0x00000000);


      for(i=0; i<lives[0]; i++)
        obj_render(6, 32 /*68*/, 462-(i*20), 2+i);
    }
    /* TEMP */

    /* ------------------------------------------- */
    /* Additional rendering for the round activity */
    /* ------------------------------------------- */
    switch(round_activity)
    {
      /* ---- */
      /* PLAY */
      /* ---- */
      case 0:
        switch(prev_round_activity)
        {
          case 0:
            snd_bg_poll();
            break;
          case -2:
            snd_bg_modify(BG_MODIFY_RESTART, bg_music, 0);
            break;
          default:
            snd_bg_modify(BG_MODIFY_SWITCH, bg_music, 0);
            break;
        }
        break;


      /* ----- */
      /* PAUSE */
      /* ----- */
      case -2:
        if (prev_round_activity != round_activity)
          snd_bg_modify(BG_MODIFY_PAUSE, bg_music, 0);

        fnt_render(font, "PAUSED",
                   320, 240, 2, 0,
                   1, 1, 0xFFFFFF,
                   4, 1, 0x000000);
        break;

      default:
        if (prev_round_activity != round_activity)
          snd_bg_stop();
        break;
    }

    vid_screen_end();
    
    //Make sure music gets updated
    //if (frames_this_round == 0)
    //  snd_bg_modify(BG_MODIFY_SWITCH, bg_music, 0);
    //else if (frames_this_round % 2 == 0)
    //  snd_bg_poll();
    
    frames_this_round++;    

    prev_round_activity = round_activity;

    /* -------------------- */
    /* all players are dead */
    /* -------------------- */
    if (!alive)
    {
      round_activity = -1;
    }
    
    /* ----------------------------------------- */
    /* all burger parts are at max... next round */
    /* ----------------------------------------- */
    if (at_max == total_parts)
    {
      int player;

      for(player=0; player<1; player++)
      {
        // If B-O-N-U-S was in play, switch it off
        if ((bonus[player] & BONUS_ALL) == BONUS_ALL)
          bonus[player] &= ~BONUS_ALL;
      
        // Switch off all other specials
        bonus[player] &= BONUS_ALL;
      }

      round_activity = new_round+1;
    }

    //DBG_SET_LEVEL(0);
  }

  // Stop any music
  snd_bg_stop();

  round_unload();
  
  return new_round;
}

 

/* start_game _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int start_game(fnt_info_t *font, int style)
{
  int new_state = 1;
  int i;

  int  round = round_to_start_on;
  char roundname[50];
  
  FILE *fp = 0;
  
  DBG_ENTRY(99, ("start_game(%p, %d [%s])", font, style, game_style[style]));
  
  sprintf(roundname,
          "%s/game/%s%s%03d.%s", 
          data_dir, RND_DEF_PREFIX, game_style[style], round, RND_DEF_FILETYPE);
  
  DBG_PRINT(99, ("Loading '%s'", roundname));

  obj_load(game_style[style]);

  #if 0
  {
    int n;
    uint8 *mask;

    for(n=0;n<OBJ_INITIAL_OBJS;n++)
    {
      obj_inst_t *o = obj_inst_create(n,320,240,1,0,0,0,1,1,0xFFFFFF);

      if ((mask = obj_inst_getmask(o)))
      {
        DBG_PRINT(0,("OBJECT: %d", n));
        DBG_MASK(0, mask, obj_inst_getwidth(o), obj_inst_getheight(o));
      }
    }

    //if ((mask = obj_inst_getmask(obj_inst_create(75,320,240,1,0,0,0,1,1,0xFFFFFF))))
    //  DBG_MASK(0, mask, 32, 48);

    exit(99);
  }
  #endif

  for(i=0; i<4; i++)
  {
    lives[i] = INITIAL_LIVES;
    pepper[i] = INITIAL_PEPPER;
    showscore[i] = score[i] = 0;
    bonus[i] = 0;
    mode_countdown[i] = 0;
  }

  while(round && (fp = fopen(roundname, "rb")))
  {
    /* ---------------------------------------------------- */
    /* start_round returns the round to go to next, 0 means */
    /* game over... this means you can 'skip rounds' if you */
    /* want to by returning a value > round+1...            */
    /* ---------------------------------------------------- */
    round = start_round(fp, round, font);

    DBG_PRINT(99, ("new round %d", round)); 

    /* ---------------------------- */
    /* round completed successfully */
    /* ---------------------------- */
    if (round)
    {
      sprintf(roundname,
              "%s/game/%s%s%03d.%s",
              data_dir, RND_DEF_PREFIX, game_style[style], round, RND_DEF_FILETYPE);
    }
  
    fclose(fp);
  }
  
  obj_unload(game_style[style]);
  
  /* --------------------------------------------------- */
  /* If round > 0 here, it means that we couldn't find a */
  /* round file for the round reached... so the game has */
  /* been completed.                                     */
  /* --------------------------------------------------- */
  if (round)
  {
    /* BEAT ALL ROUNDS - SOME SORT OF 'WHOOPEE DO' */
  }
  else
  {
    /* GAME OVER */
  }

  /* HI-SCORE STUFF - set new_state=2 to go straight to high score table... */

  DBG_EXIT(99, ("start_game new_state=%d", new_state));
  
  return new_state;
}
/* main _____________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int main(int argc, char **argv) 
{
  int conts = 0;
  int i;
  fnt_info_t font;
  
  conts = cnt_init((cnt_info_t *)&cont);
  
  int dc_region, ct;
  
  dc_region = flashrom_get_region();
  ct = vid_check_cable();
 
  /* Prompt the user for whether to run in PAL50 or PAL60 if the flashrom says
  the Dreamcast is European and a VGA Box is not hooked up. */
  if(dc_region == FLASHROM_REGION_EUROPE && ct != CT_VGA) {
		if(pal_menu()) {
			vid_screen_init(0);
		}
		else {
			vid_screen_init(2);
		}
  }
  else vid_screen_init(0);

  obj_init();

  fnt_load("FN000",&font);

  #if defined(_WIN32)
  if (argc > 1)
    round_to_start_on = (int)strtol(argv[1], 0, 10);
  #endif
  
  /* --------------------------------------------- */
  /* If there is at least one controller connected */
  /* --------------------------------------------- */
  if (conts)
  {
    int state = 1;
    int prev_state = 4;
    int title_frame = 0;
    
    int title_music = snd_bg_load("BtimeTTL.mp3");

    obj_load("TITLES");

    /* ------------------------------------------------ */
    /* state:                                           */
    /*  0 - Quit.                                       */
    /*  1 - Titles.                                     */
    /*  2 - High scores.                                */
    /*  3 - Config.                                     */
    /*  4 - Game.                                       */
    /* ------------------------------------------------ */
    while(state && check_win32())
    {
      int list;

      cnt_state_all((cnt_info_t *)&cont); 

      title_frame++;

      for(i=0;i<conts;i++)
      {
        if ((cont[i].buttons & CONT_START) && !(cont[i].prev_buttons & CONT_START))
        {
          switch(state)
          {
            case 1:
              state = 4;
              title_frame = 0;
              break;
            case 2:
              state = 1;
              title_frame = 0;
              break;
          }
        }
        else if (state == 1)
        {
          if (!(title_frame % HIGHSCORE_FREQUENCY))
          {
            state = 2;
            title_frame = 0;
          }
          else
          {  
            #if 0
            if (cont[i].buttons & CONT_A) /* TEMP */
              state = 0;
            #endif
              
            /* Handle selection changes etc */
          }
        }
        else if (state == 2)
        {
          if (!(title_frame % HIGHSCORE_FREQUENCY))
          {
            state = 1;
            title_frame = 0;
          }
        }
      }
      
      if (state != prev_state)
      {
        round_unload();

        switch (state)
        {
          case 1:
          case 2:
            obj_inst_list_load("TITLES");
            break;
        }
      }

      vid_screen_begin(0,0,0,0);

      /* ----------------- */
      /* Render everything */
      /* ----------------- */
      for(list=0; list<OBJ_MAX_LISTS; list++)
      {
        obj_inst_t *curr = obj_lists[list];
        
        while(curr)
        {
          obj_inst_t *next = curr->next;

          obj_inst_animate(curr);
          
          if (obj_inst_render(curr, 1) == -1)
          {
            obj_lists[list] = obj_inst_remove(obj_lists[list],
                                              curr,
                                              1);
          }
          
          curr = next;
        }
      }
      
      switch(state)
      {
        /* ------------------------------------ */
        /* Titles / Highscore / Config          */
        /* ------------------------------------ */
        case 1:
        case 2:
        case 3:
        {
          if (prev_state == 4){
            snd_bg_modify(BG_MODIFY_SWITCH, title_music, 0);
        }
          else {
            snd_bg_poll();
		}

          switch(state)
          {
            case 1:
             break;
            case 2:
             break;
            case 3:
              fnt_render(&font, "Configuration...",
                        320, 52, 2, 0,
                        2, 1, 0xFFFFFF,
                        1, 0, 0);
             break;
          }
          
          fnt_render(&font,"PRESS START!",
                    136, 432, 2, 0,
                    1, 1, 0x1FFFFFF,
                    4, 1, utl_rainbow_palette[16-(title_frame%16)]);

          fnt_render(&font, "Code: Warmtoe   Graphics: Krut   Music: DaMadFiddler",
                    320, 464, 2, 0,
                    0.5, 1, 0x000000,
                    1, 0, 0);

          vid_screen_end();

          prev_state = state;
        }
        break;
                
        /* ------------------------------------ */
        /* Game                                 */
        /* ------------------------------------ */
        case 4:
        {
          vid_screen_end();

          snd_bg_stop();

          state = start_game(&font, 0);

          prev_state = 4;
        }
        break;
      }    
    }

    obj_unload("TITLES");

    snd_bg_unload(title_music);
  }
  else
  {
    vid_screen_begin(0,0,0,0);
    
    fnt_render(&font, "No valid controllers connected",
              320, 320, 2, 0,
              1, 1,  0xFFFFFF,
              4, 1, 0x000000);
              
    vid_screen_end();
    
    thd_sleep(60000);
  }
  
  fnt_unload(&font);

  obj_term();
  
  vid_screen_term();

  return 0;
}

#undef BTIM_C
