#define OBJ_C

#include <kos.h>
#include <stdlib.h>
#include <dc/fmath.h>
#include <math.h>
#include <string.h>

#include "utils.h"
#include "object.h"
#include "video.h"
#include "tiles.h"
#include "images.h"
#include "anim.h"
#include "font.h"
#include "sound.h"

int objs = 0;                                 /* total available object slots */
int high_obj = -1;                  /* obj highwater mark (used for def objs) */
int last_def_obj = -1;                /* last default object (obj_init phase) */

/* obj_init _________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_init(void)
{
  DBG_ENTRY(99,("obj_init()"));

  if ((obj_infos = utl_malloc(sizeof(obj_info_t)*OBJ_INITIAL_OBJS)))
  {
    objs = OBJ_INITIAL_OBJS;
    high_obj = -1;
  }
  DBG_ELSE(0,("unable to allocate obj_infos (%d)",OBJ_INITIAL_OBJS));
  
  /* clear all object lists */
  
  memset(obj_lists,0,sizeof(obj_lists));
  
  til_init();
  img_init();
  ani_init();
  fnt_init();
  snd_all_init();
  
  /* Load 'default' objects -> an ini file which then gives an obj_load
   * filename */
   
  last_def_obj = high_obj;
  
  DBG_EXIT(99,("obj_init ldo=%d",last_def_obj));
  
  return;
}

/* obj_term _________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_term(void)
{
  DBG_ENTRY(99,("obj_term()"));

  /* clear up the default objects */
  last_def_obj = -1;
  
  obj_unload(0);

  utl_free(obj_infos);

  obj_infos = 0;
  objs = 0;
  high_obj = -1;

  snd_all_term();
  fnt_term();
  ani_term();
  img_term();
  til_term();
  
  DBG_EXIT(99,("obj_term"));
  
  return;
}

/* obj_load _________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_load(char *obj_file)
{
  int    rc = 0;
  
  FILE  *fp;
  char   filename[256];

  DBG_ENTRY(99,("obj_load(%s)",obj_file));

  sprintf(filename,"%s/game/%s.%s",data_dir,obj_file,OBJ_EXTENSION);

  DBG_PRINT(99,("opening %s",filename));
  
  if ((fp=fopen(filename,"rb")))
  {
    char line[1024];
    
    DBG_INT(dbg_lines,0);
    DBG_INT(dbg_objs,0);

    while (fgets(line,sizeof(line),fp))
    {
      int    obj_num;
      int    obj_class = 0;
      int8   obj_hits_needed = -1;

      uint32 flags = 0x00000000;
      
      char *t = strtok(line, UTL_WHITESPACE);
      char *e;
      
      DBG_INC(dbg_lines);

      if (!t || *t == '#')
      {
        continue;
      }
        
      obj_num = strtol(t,&e,10);
      
      if (e != t) 
      {
        if ((t = strtok(0,UTL_WHITESPACE))) obj_class = strtol(t,&e,10);
       
        if ((t = strtok(0, UTL_WHITESPACE))) obj_hits_needed = (int8)strtol(t,&e,10);

        if ((t = strtok(0,UTL_WHITESPACE)))
        {
          int wid = 0;
          int hgt = 0;
          uint8 *mask = 0;
          char obj_type = *t;
          uint32 var    = 0;
              
          t += strlen(t)+1;    
              
          switch(obj_type)
          {
            case OBJ_TILE:
              var = til_obj_create_str(t);

              if (var)
              {
                wid = til_getwidth(var);
                hgt = til_getheight(var);
                mask = til_getmask(var, 0);
              }
              break;

            case OBJ_IMAGE:
              var = img_obj_create_str(t);

              if (var)
              {
                wid = img_getwidth_obj(var);
                hgt = img_getheight_obj(var);
              }
              break;
            
            case OBJ_ANIM:
              var = ani_obj_create_str(t);
              break;
            
            case OBJ_SFX:
              var = snd_fx_obj_create_str(t);
              break;

            case OBJ_MUSIC:
              var = snd_bg_obj_create_str(t);
              break;

            case OBJ_FONT:
              var = fnt_obj_create_str(t);
              break;

            case OBJ_COPY:
              if ((t = strtok(0,UTL_WHITESPACE)))
              {
                int copy_num = strtol(t,&e,10);
                
                obj_type = obj_infos[copy_num].type;
                var      = obj_infos[copy_num].var;
                flags   |= OBJ_INFO_FLAG_COPY;
              }
              break;
              
            default:
              DBG_PRINT(99, ("%04d: unrecognised type '%c'",dbg_lines,obj_type));
              break;
          }
          
          if (var)
          {
            obj_set(obj_num,obj_class,obj_type,obj_hits_needed,var,flags,wid,hgt,mask);
            DBG_INC(dbg_objs);
          }
          DBG_ELSE(0,("%04d: '%c' create err",dbg_lines,obj_type));
        }
        DBG_ELSE(0, ("%04d: nothing after obj_num.",dbg_lines));
      }
      DBG_ELSE(0,("%04d: no obj_num '%s'",dbg_lines,t));
    }
    
    DBG_PRINT(99, ("%04d: created %04d objects",dbg_lines,dbg_objs));
    
    fclose(fp); 
  }
  DBG_ELSE(0,("open failed with %d (%s)",errno,strerror(errno)));
    
  DBG_EXIT(99,("obj_load rc=%d",rc));
  
  return rc;
}

/* obj_unload _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_unload(char *obj_file)
{
  DBG_ENTRY(99,("obj_unload(%s)", (obj_file?obj_file:"null")));
  
  if (obj_infos)
  {
    int i = -1;
    FILE  *fp = 0;
    char   filename[256];

    if (obj_file)
    {
      sprintf(filename,"%s/game/%s.%s",data_dir,obj_file,OBJ_EXTENSION);

      DBG_PRINT(99,("opening %s",filename));

      fp=fopen(filename,"rb");
      
      DBG_PRINT(99,("fp=%p", fp));
    }

    if ((obj_file && fp) || (!obj_file))
    {
      do
      {
        if (obj_file)
        {
          char line[1024];

          i = objs;

          while(fgets(line,sizeof(line),fp))
          {
            char *t = strtok(line, UTL_WHITESPACE);

            if (!t || *t == '#')
            {
              continue;
            }

            i = strtol(t,0,10);

            break;
          }
        }
        else
         i++;

        DBG_PRINT(99,("i=%d",i));

        if (i==objs)
          break;

        /* ------------------------ */
        /* If this is not a copy... */
        /* ------------------------ */     
        if (!obj_infos[i].flags & OBJ_INFO_FLAG_COPY)
        {
          switch(obj_infos[i].type)
          {
            case OBJ_TILE:
              til_obj_free(obj_infos[i].var);
              break;
        
            case OBJ_IMAGE:
              img_obj_free(obj_infos[i].var);
              break;

            case OBJ_ANIM:
              ani_obj_free(obj_infos[i].var);
              break;
          
            case OBJ_SFX:
              snd_fx_obj_free(obj_infos[i].var);
              break;

            case OBJ_MUSIC:
              snd_bg_obj_free(obj_infos[i].var);
              break;

            case OBJ_FONT:
              fnt_obj_free(obj_infos[i].var);
              break;
          }
        }

        utl_free(obj_infos[i].mask);

        obj_infos[i].type = OBJ_NONE;
        obj_infos[i].var = 0;
        obj_infos[i].wid = 0;
        obj_infos[i].hgt = 0;
        obj_infos[i].mask = 0;
      }
      while(i < objs);

      if (obj_file)
      {
        fclose(fp);
      }
    }
  }
  
  DBG_EXIT(99,("obj_unload"));
}

/* obj_set __________________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_set(int num, 
             int class,
             uint8 type, 
             int8 hits_needed,
             uint32 var,
             uint32 flags,
             int wid,
             int hgt,
             uint8 *mask)
{
  DBG_ENTRY(99,("obj_set(%d,%d,%d,%d,%d,0x%08X,%d,%p,%d)",
                num,class,(int)type,(int)hits_needed,(int)var,flags,wid,hgt,mask));
  
  if (num >= objs)
  {
    obj_info_t *tmp = utl_realloc(obj_infos,sizeof(obj_info_t)*num+1);

    if (tmp)
    {
      memset(tmp+objs,0x00,(num-objs+1)*sizeof(obj_info_t));
      
      obj_infos = tmp;
      objs = num+1;
    }
    else
    {
      DBG_PRINT(0,("unable to realloc obj_infos (%d -> %d)",objs,num+1));
      return;
    }
  }
  
  if (obj_infos)
  {
    obj_infos[num].class = class;
    obj_infos[num].type = type;
    obj_infos[num].var = var;
    obj_infos[num].flags = flags;
    obj_infos[num].wid = wid;
    obj_infos[num].hgt = hgt;
    obj_infos[num].mask = mask;
    obj_infos[num].hits_needed = hits_needed;

    if (num > high_obj)
      high_obj = num;
  }
  DBG_ELSE(0,("obj_infos is null, object %d not added",num));
  
  DBG_EXIT(99,("obj_set"));
}

/* obj_getwidth _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_getwidth(int obj)
{
  int rc = obj_infos[obj].wid;
  
  DBG_ENTRY(99,("obj_getwidth(%d)",obj));

  if (!rc)
  {
    switch(obj_infos[obj].type)
    {
      case OBJ_TILE:
        rc = til_getwidth(obj_infos[obj].var);
        break;
      
      case OBJ_IMAGE:
        rc = img_getwidth_obj(obj_infos[obj].var);      
        break;
    }

    obj_infos[obj].wid = rc;
  }
  
  DBG_EXIT(99,("obj_getwidth rc=%d",rc));
  
  return rc;
}

/* obj_getheight ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_getheight(int obj)
{
  int rc = obj_infos[obj].hgt;
  
  DBG_ENTRY(99,("obj_getheight(%d)",obj));

  if (!rc)
  {
    switch(obj_infos[obj].type)
    {
      case OBJ_TILE:
        rc = til_getheight(obj_infos[obj].var);
        break;
      
      case OBJ_IMAGE:
        rc = img_getheight_obj(obj_infos[obj].var);
        break;
    }

    obj_infos[obj].hgt = rc;
  }

  DBG_EXIT(99,("obj_getheight rc=%d",rc));
  
  return rc;
}

/* obj_render _______________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_render(int obj, int x, int y, float z)
{
  switch(obj_infos[obj].type)
  {
    case OBJ_SFX:
      snd_fx_render(obj_infos[obj].var,
                    obj_infos[obj].class,
                    x,
                    y);
      break;
    case OBJ_MUSIC:
      snd_bg_modify(BG_MODIFY_SWITCH, obj_infos[obj].var, 0);
      snd_bg_start();
      break;
    case OBJ_TILE:
      til_render_obj(obj_infos[obj].var,
                     0,
                     x,
                     y,
                     z,
                     0, 1, 1, 0xFFFFFF);
      break;
    case OBJ_FONT:
      fnt_render_obj(obj_infos[obj].var,
                     0,
                     x,
                     y,
                     z,
                     0, 1, 1, 0xFFFFFF);
      break;
  }
}

/* obj_inst_render __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_inst_render(obj_inst_t *inst, int animate)
{
  int obj = inst->obj;
  int rc = 0;
  
  DBG_ENTRY(99,("obj_inst_render(%p)",inst));

  if (obj == OBJ_SPECIAL_SCORE)
  {
    rc = fnt_render_obj(FNT_MAX_OBJECTS,
                        inst->var,
                        inst->x,
                        inst->y,
                        inst->z,
                        inst->rot,
                        inst->scale<0?-inst->scale:inst->scale,
                        inst->fade,
                        inst->colour);
  }
  else
  {
    if (obj_infos[obj].type == OBJ_ANIM)
      obj = ani_get_obj(inst->var);
  
    if (obj != -1 && inst->z > 0)
    {
      switch(obj_infos[obj].type)
      {
        case OBJ_TILE:
          rc = til_render_obj(obj_infos[obj].var,
                              inst->var,
                              inst->x,
                              inst->y,
                              inst->z,
                              inst->rot,
                              inst->scale<0?-inst->scale:inst->scale,
                              inst->fade,
                              inst->colour);
          break;
        
        case OBJ_IMAGE:
          rc = img_render_obj(obj_infos[obj].var,
                              inst->var,
                              inst->x,
                              inst->y,
                              inst->z,
                              inst->rot,
                              inst->scale<0?-inst->scale:inst->scale,
                              inst->fade,
                              inst->colour);
          break;
          
        case OBJ_SFX:
          rc = snd_fx_render(obj_infos[obj].var,
                             obj_infos[obj].class,
                             inst->x,
                             inst->y);
          /* ------------------------- */                   
          /* Once played, it must die! */
          /* ------------------------- */
          inst->eol = 1;
          break;
  
        case OBJ_FONT:
          rc = fnt_render_obj(obj_infos[obj].var,
                              inst->var,
                              inst->x,
                              inst->y,
                              inst->z,
                              inst->rot,
                              inst->scale<0?-inst->scale:inst->scale,
                              inst->fade,
                              inst->colour);
          break;
      }

      if (animate)
      {
        /* -------------------- */
        /* Auto colour increase */
        /* -------------------- */
        if (inst->colour & VID_COLOUR_INC)
        {
          uint32 prev_colour = inst->colour;

          inst->colour = ((prev_colour & VID_RGB_MASK) + 1) % VID_RGB_MASK;
          inst->colour |= prev_colour & ~VID_RGB_MASK;
        }

        /* -------------------- */
        /* Auto scale reduction */
        /* -------------------- */
        if (inst->scale < -0.1 && inst->eol > 0)
        {
          if (inst->eol < 200 && (inst->eol % 20 == 0))
            inst->scale += 0.025;
        }
      }
    }
  }

  /* modify end of life if required */
  if (animate && inst->eol > 0 && --inst->eol == 0)
    rc = -1;
    
  DBG_EXIT(99,("obj_inst_render rc=%d",rc));
  
  return rc;
}

/* obj_inst_render_all ______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_render_all(int animate)
{
  int list;

  for(list=0; list<OBJ_MAX_LISTS; list++)
  {
    obj_inst_t *curr = obj_lists[list];
        
    while(curr)
    {
      obj_inst_t *next = curr->next;

      if (animate)
        obj_inst_animate(curr);

      if (obj_inst_render(curr, animate) == -1)
      {
        DBG_PRINT(99,("EOL %p",curr));

        obj_lists[list] = obj_inst_remove(obj_lists[list], curr, 1);

        DBG_PRINT(99,("value=%p",obj_lists[list]));
      }

      curr = next;
    }
  }

  return;
}

/* obj_inst_move ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_move(obj_inst_t *inst, int reverse)
{
  DBG_ENTRY(99,("obj_inst_move(%p)",inst));
  
  if (inst->speed)
  {
    #if !defined(OBJ_ACCURATE_MOVEMENT)
    float xd,yd;
    #endif
    float dir = (float)inst->dir;
    
    if (reverse)
      dir = (float)((180+inst->dir)%360);
    
    #if defined(OBJ_ACCURATE_MOVEMENT)
    inst->real_x += fsin((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    inst->real_y -= fcos((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    
    /* simple x and y for the rendering step */
    inst->x = (int32)(inst->real_x);
    inst->y = (int32)(inst->real_y);
    #else
    xd = fsin(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    yd = fcos(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    
    if (xd>0)
      xd += 0.5;
    else if (xd < 0)
      xd -= 0.5;
      
    if (yd>0)
      yd += 0.5;
    else if (yd < 0)
      yd -= 0.5;
   
    inst->x += (int)xd;
    inst->y -= (int)yd;
    #endif    
  }
  
  DBG_EXIT(99,("obj_inst_animate"));
}

/* obj_inst_animate _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_inst_animate(obj_inst_t *inst)
{
  int rc = 0;
  
  DBG_ENTRY(99,("obj_inst_animate(%p)",inst));
  
  obj_inst_move(inst, 0);
  
  switch(obj_infos[inst->obj].type)
  {
    case OBJ_ANIM:
      rc = ani_animate(inst);
      break;
  }
  
  DBG_EXIT(99,("obj_inst_animate rc=%d",rc));
  
  return rc;
}

/* obj_inst_reset ___________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_reset(obj_inst_t *inst, int position)
{
  DBG_ENTRY(99,("obj_inst_reset(%p)",inst));
  
  if (position)
  {
    inst->x = inst->init_x;
    inst->y = inst->init_y;
    #ifdef OBJ_ACCURATE_MOVEMENT
    inst->real_x = (float)inst->x;
    inst->real_y = (float)inst->y;
    #endif
    inst->dir = inst->init_dir;
    inst->rot = inst->init_rot;
    inst->speed = inst->init_speed;
  }

  inst->z = inst->init_z;
  inst->hits_needed = obj_infos[inst->obj].hits_needed;

  switch(obj_infos[inst->obj].type)
  {
    case OBJ_ANIM:
      ani_inst_reset(inst->var);
      break;
  }
  
  DBG_EXIT(99,("obj_inst_reset"));
}

/* obj_inst_setevent ________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_setevent(obj_inst_t *inst, int event)
{
  DBG_ENTRY(99,("obj_inst_setevent(%p,%d)",inst,event));
  
  /* -------------------------------- */
  /* Turn a 'HIT' into 'DEAD' if used */
  /* -------------------------------- */
  if (event == OBJ_EVENT_HIT && inst->hits_needed != -1)
  {
    if (inst->hits_needed == 1)
      event = OBJ_EVENT_DEAD;

    inst->hits_needed--;
  }

  switch(obj_infos[inst->obj].type)
  {
    case OBJ_ANIM:
      ani_setevent(inst,event);
      break;
    default:
      /* Kill other objects */
      if (event == OBJ_EVENT_DEAD)
        inst->eol = 1;
      break;
  }
  
  DBG_EXIT(99,("obj_inst_setevent"));
  
  return;
}

/* obj_inst_getattribute ____________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int32 obj_inst_getattribute(obj_inst_t *inst, int attribute)
{
  int32 rc = 0;
  
  DBG_ENTRY(99,("obj_inst_getattribute(%p,%d)",inst,attribute));
  
  switch(obj_infos[inst->obj].type)
  {
    case OBJ_ANIM:
      rc = ani_getattribute(inst->var, attribute);
      break;
  }
  
  DBG_EXIT(99,("obj_inst_getattribute rc=%d",(int)rc));
  
  return rc;
}


/* obj_inst_getwidth  _______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_inst_getwidth(obj_inst_t *inst)
{
  int rc = obj_infos[inst->obj].wid;
  
  DBG_ENTRY(99,("obj_inst_getwidth(%p)",inst));

  if (!rc)
  {
    int obj = inst->obj;

    if (obj_infos[obj].type == OBJ_ANIM)
      obj = ani_get_obj(inst->var);
    
    if (obj != -1)
      rc = obj_infos[obj].wid;
  }

  DBG_EXIT(99,("obj_inst_getwidth rc=%d",rc));
  
  return rc;
}

/* obj_inst_getheight _______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

int obj_inst_getheight(obj_inst_t *inst)
{
  int rc = obj_infos[inst->obj].hgt;

  DBG_ENTRY(99,("obj_inst_getheight(%p)",inst));

  if (!rc)
  {
    int obj = inst->obj;

    if (obj_infos[obj].type == OBJ_ANIM)
      obj = ani_get_obj(inst->var);

    if (obj != -1)
      rc = obj_infos[obj].hgt;
  }

  DBG_EXIT(99,("obj_inst_getheight rc=%d",rc));
  
  return rc;
}

/* obj_inst_getmask _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

uint8 *obj_inst_getmask(obj_inst_t *inst)
{
  int obj = inst->obj;
  uint8 *rc = 0;
  
  DBG_ENTRY(99,("obj_inst_getmask(%p)",inst));
  
  if (obj_infos[obj].type == OBJ_ANIM)
    obj = ani_get_obj(inst->var);

  if (obj != -1)
  {
    if (inst->rot)
    {
      if (!inst->mask)
      {
        switch(obj_infos[obj].type)
        {
          case OBJ_TILE:
            inst->mask = til_getmask(obj_infos[obj].var, inst->rot);
            break;
        }
      }

      rc = inst->mask;
    } 
    else
      rc = obj_infos[obj].mask;
  }
  
  DBG_EXIT(99,("obj_inst_getmask rc=%p",rc));
  
  return rc;
}  

/* obj_inst_collide_fs ______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_collide_fs(obj_inst_t *inst, obj_inst_t *list, uint32 coll_mode)
{
  const int point_mode = (coll_mode & OBJ_COLL_POINT)?1:0;
  int x = inst->x;
  int y = inst->y;
  uint8 *mask = 0;
  int wid = 0;
  int hgt = 0;
  
  obj_inst_t *check_inst = list;
  obj_inst_t *g_hit = 0;
  
  int obj_collwid = 1;
  int obj_collhgt = 1;
  
  obj_inst_t *list_next = 0;

  DBG_ENTRY(99,("obj_inst_collide_fs(%p,%p,%d)",inst,list,(int)coll_mode));  
  
  if (coll_mode & OBJ_COLL_POINT)
    coll_mode &= ~OBJ_COLL_POINT;
  else
  {  
    obj_collwid = (int)obj_inst_getwidth(inst);
    obj_collhgt = (int)obj_inst_getheight(inst);
  
    if (!obj_collwid || !obj_collhgt)
      return 0;
    
    if (inst->rot % 90)
    {
      obj_collwid = fsqrt((obj_collwid*obj_collwid)<<1);
      obj_collhgt = fsqrt((obj_collhgt*obj_collhgt)<<1);;
    }
    else if (inst->rot % 180)
    {
      int tmp = obj_collwid;
    
      obj_collwid = obj_collhgt;
      obj_collhgt = tmp;
    }
  }
  
  if (check_inst)
  {
    wid = obj_inst_getwidth(check_inst);
    hgt = obj_inst_getheight(check_inst);

    if (coll_mode & OBJ_COLL_CHECKONE)
    {
      coll_mode &= ~OBJ_COLL_CHECKONE;
      list_next = list->next;
      list->next = 0;
    }

    if (coll_mode & OBJ_COLL_LOOKAHEAD)
    {
      float xd,yd;
      float dir = (float)inst->dir;

      coll_mode &= ~OBJ_COLL_LOOKAHEAD;

      #if defined(OBJ_ACCURATE_MOVEMENT)
      xd = inst->real_x + fsin((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
      yd = inst->real_y - fcos((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;

      x = (int32)xd;
      y = (int32)yd;
      #else
      xd = fsin(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
      yd = fcos(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;

      if (xd>0)
        xd += 0.5;
      else if (xd < 0)
        xd -= 0.5;

      if (yd>0)
        yd += 0.5;
      else if (yd < 0)
        yd -= 0.5;

      x = inst->x + (int)xd;
      y = inst->y - (int)yd;
    #endif    
    }
  }
  
  while (check_inst)
  {
    int x1 = x-(obj_collwid>>1);
    int y1 = y-(obj_collhgt>>1);
    int x2 = x1 + obj_collwid;
    int y2 = y1 + obj_collhgt;
    int a1;
    int a2;

    int cobj_collwid = wid;
    int cobj_collhgt = hgt;

    uint8 *check_mask = 0;

    /* -------------------------------------------------- */    
    /* Don't check for self-collision!                    */
    /* And don't check objects whose z-order is 0 or less */
    /* -------------------------------------------------- */
    if (check_inst == inst ||
        check_inst->z <= 0 ||
        inst->z <= 0)
    {
      check_inst = check_inst->next;
      continue;
    }
    
    if (check_inst->rot % 90)
    {
      cobj_collwid = fsqrt((cobj_collwid*cobj_collwid)<<1);
      cobj_collhgt = fsqrt((cobj_collhgt*cobj_collhgt)<<1);
    }
    else if (check_inst->rot % 180)
    {
      int tmp = cobj_collwid;
        
      cobj_collwid = cobj_collhgt;
      cobj_collhgt = tmp;
    }
      
    a1 = check_inst->x-(cobj_collwid>>1);
    a2 = a1 + cobj_collwid;

    DBG_PRINT(99,("x1: %d y1: %d x2: %d y2: %d",x1,y1,x2,y2));
    DBG_PRINT(99,("a1: %d b1: xx a2: %d b2: xx (%p)",a1,a2,check_inst));

    if ((x1>=a1 && x1<=a2) ||   /* First Edge in  */
        (x2>=a1 && x2<=a2) ||   /* Second Edge in */
        (x1<a1 && x2>a2))       /* Enclosed       */
    {
      int b1 = check_inst->y-(cobj_collhgt>>1);
      int b2 = b1 + cobj_collhgt;

      DBG_PRINT(99,("PASSED X TEST"));
      DBG_PRINT(99,("x1: %d y1: %d x2: %d y2: %d",x1,y1,x2,y2));
      DBG_PRINT(99,("a1: %d b1: %d a2: %d b2: %d (%p)",a1,b1,a2,b2,check_inst));
        
      if ((y1>=b1 && y1<=b2) || /* First Edge in  */
          (y2>=b1 && y2<=b2) || /* Second edge in */
          (y1<b1 && y2>b2))     /* Enclosed       */
      {
        DBG_PRINT(99,("PASSED Y TEST"));
                  
        if (coll_mode == OBJ_COLL_BBONLY)
        {
          g_hit = check_inst;
          goto EXIT;
        }
        else   /* PIXEL PERFECT collision detection */
        {
          int dx = UTL_MIN(x1,a1);
          int dy = UTL_MIN(y1,b1);

          /* convert to 0 based co-ordinates */

          x1 -= dx; x2 -= dx; a1 -= dx; a2 -= dx;
          y1 -= dy; y2 -= dy; b1 -= dy; b2 -= dy;

          DBG_IF(99,(x1<0||x2<0||a1<0||a2<0||y1<0||y2<0||b1<0||b2<0),
                   ("BASIC FAILURE 1"));
          DBG_IF(99,(x2<a1||a2<x1||y2<b1||b2<y1),("BASIC FAILURE 2"));

          {
            int x = UTL_MAX(x1,a1);
            int y = UTL_MAX(y1,b1);
            int a = UTL_MIN(x2,a2);
            int b = UTL_MIN(y2,b2);

            DBG_PRINT(99,("%d,%d -> %d,%d",x,y,a,b));
            DBG_PRINT(99,("%d,%d",x-a1, y-b1));
              
            if (point_mode || 
                mask       ||
                ((mask = obj_inst_getmask(inst))))
            {
              if (check_mask ||
                  ((check_mask = obj_inst_getmask(check_inst))))
              {
                const int sp1 = ((y-y1)*obj_collwid)+(x-x1);
                const int sp2 = ((y-b1)*cobj_collwid)+(x-a1);
                const int cb  = a-x;                  /* pixels per line to check */
                int cl  = b-y;                  /* lines to check           */

                uint8 fake_point = 0xFF;
                uint8 register *cm1 = (point_mode?&fake_point:mask + sp1);
                uint8 register *cm2 = check_mask + sp2;

                DBG_MASK(99, check_mask, cobj_collwid, cobj_collhgt);

                DBG_PRINT(99,("cl=%d cb=%d %d %d sp2=%d", cl, cb, *cm1, *cm2, sp2));

                while(cl)
                {
                  int ccb = 0;

                  while(ccb < cb)
                  {
                    #if 0
                    cb_count++;
                    #endif

                    if (*cm1 & *cm2)  /* collision NB: BITWISE AND */
                    {
                      g_hit = check_inst;
                      goto EXIT;
                    }

                    cm1+=(!point_mode);
                    cm2++;
                    ccb++;
                  }

                  cm1 += (point_mode?0:(obj_collwid-cb));
                  cm2 += (cobj_collwid-cb);
                  cl--;
                }
              }
            }
          }
        } /* PIXEL PERFECT collision detection */             
      }
    }

    check_inst = check_inst->next;
  }
 
  EXIT:

  if (list_next)
    list->next = list_next;

  DBG_EXIT(99,("obj_inst_collide_fs rc=%p",g_hit));

  return g_hit;
}

/* obj_inst_collide _________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_collide(obj_inst_t *inst, obj_inst_t *list, uint32 coll_mode)
{
  int x = inst->x;
  int y = inst->y;

  const int point_mode = (coll_mode & OBJ_COLL_POINT)?1:0;
  
  uint8 *mask = 0;

  obj_inst_t *check_inst = list;
  obj_inst_t *g_hit = 0;
  
  int obj_collwid = 1;
  int obj_collhgt = 1;

  obj_inst_t *list_next = 0;
  
  DBG_ENTRY(99,("obj_inst_collide(%p,%p,%d)",inst,list,(int)coll_mode));  
  
  if (coll_mode & OBJ_COLL_POINT)
    coll_mode &= ~OBJ_COLL_POINT;
  else
  {  
    obj_collwid = (int)obj_inst_getwidth(inst);
    obj_collhgt = (int)obj_inst_getheight(inst);
  
    if (!obj_collwid || !obj_collhgt)
    {
      return 0;
    }
    
    if (inst->rot % 90)
    {
      obj_collwid = fsqrt((obj_collwid*obj_collwid)<<1);
      obj_collhgt = fsqrt((obj_collhgt*obj_collhgt)<<1);;
    }
    else if (inst->rot % 180)
    {
      int tmp = obj_collwid;
    
      obj_collwid = obj_collhgt;
      obj_collhgt = tmp;
    }
  }
  
  if (list && coll_mode & OBJ_COLL_CHECKONE)
  {
    coll_mode &= ~OBJ_COLL_CHECKONE;
    list_next = list->next;
    list->next = 0;
  }

  if (coll_mode & OBJ_COLL_LOOKAHEAD)
  {
    float xd,yd;
    float dir = (float)inst->dir;
    
    coll_mode &= ~OBJ_COLL_LOOKAHEAD;

    #if defined(OBJ_ACCURATE_MOVEMENT)
    xd = inst->real_x + fsin((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    yd = inst->real_y - fcos((float)dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    
    x = (int32)xd;
    y = (int32)yd;
    #else
    xd = fsin(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    yd = fcos(dir * UTL_DEG2RAD)*(float)inst->speed * 0.1;
    
    if (xd>0)
      xd += 0.5;
    else if (xd < 0)
      xd -= 0.5;
      
    if (yd>0)
      yd += 0.5;
    else if (yd < 0)
      yd -= 0.5;
   
    x = inst->x + (int)xd;
    y = inst->y - (int)yd;
    #endif    
  }

  while (check_inst)
  {
    int x1 = x-(obj_collwid/2);
    int y1 = y-(obj_collhgt/2);
    int x2 = x1 + obj_collwid;
    int y2 = y1 + obj_collhgt;
    int a1;
    int a2;

    int cobj_collwid = (int)obj_inst_getwidth(check_inst);
    int cobj_collhgt = (int)obj_inst_getheight(check_inst);

    uint8 *check_mask = 0;

    /* -------------------------------------------------- */    
    /* Don't check for self-collision!                    */
    /* And don't check objects whose z-order is 0 or less */
    /* -------------------------------------------------- */
    if (check_inst == inst ||
        check_inst->z <= 0 ||
        inst->z <= 0)
    {
      check_inst = check_inst->next;
      continue;
    }
    
    if (cobj_collwid && cobj_collhgt)
    {
      if (check_inst->rot % 90)
      {
        cobj_collwid = fsqrt((cobj_collwid*cobj_collwid)<<1);
        cobj_collhgt = fsqrt((cobj_collhgt*cobj_collhgt)<<1);
      }
      else if (check_inst->rot % 180)
      {
        int tmp = cobj_collwid;
        
        cobj_collwid = cobj_collhgt;
        cobj_collhgt = tmp;
      }
      
      a1 = check_inst->x-(cobj_collwid/2);
      a2 = a1 + cobj_collwid;

      DBG_PRINT(99,("x1: %d y1: %d x2: %d y2: %d",x1,y1,x2,y2));
      DBG_PRINT(99,("a1: %d b1: xx a2: %d b2: xx (%p)",a1,a2,check_inst));

      if ((x1>=a1 && x1<=a2) ||   /* First Edge in  */
          (x2>=a1 && x2<=a2) ||   /* Second Edge in */
          (x1<a1 && x2>a2))       /* Enclosed       */
      {
        int b1 = check_inst->y-(cobj_collhgt/2);
        int b2 = b1 + cobj_collhgt;

        DBG_PRINT(99,("PASSED X TEST"));
        DBG_PRINT(99,("x1: %d y1: %d x2: %d y2: %d",x1,y1,x2,y2));
        DBG_PRINT(99,("a1: %d b1: %d a2: %d b2: %d (%p)",a1,b1,a2,b2,check_inst));
        
        if ((y1>=b1 && y1<=b2) || /* First Edge in  */
            (y2>=b1 && y2<=b2) || /* Second edge in */
            (y1<b1 && y2>b2))     /* Enclosed       */
        {
          obj_inst_t *hit = NULL;
         
          int xmin = obj_collwid;
          int xmax = 0;
          int ymin = obj_collhgt;
          int ymax = 0;
          int area = 0;
          
          DBG_PRINT(99,("PASSED Y TEST"));
                  
          if (coll_mode == OBJ_COLL_BBONLY)
          {
            hit = check_inst;
          }
          else   /* PIXEL PERFECT collision detection */
          {
            int dx = UTL_MIN(x1,a1);
            int dy = UTL_MIN(y1,b1);

            /* convert to 0 based co-ordinates */

            x1 -= dx; x2 -= dx; a1 -= dx; a2 -= dx;
            y1 -= dy; y2 -= dy; b1 -= dy; b2 -= dy;

            DBG_IF(99,(x1<0||x2<0||a1<0||a2<0||y1<0||y2<0||b1<0||b2<0),
                     ("BASIC FAILURE 1"));
            DBG_IF(99,(x2<a1||a2<x1||y2<b1||b2<y1),("BASIC FAILURE 2"));

            {
              int x = UTL_MAX(x1,a1);
              int y = UTL_MAX(y1,b1);
              int a = UTL_MIN(x2,a2);
              int b = UTL_MIN(y2,b2);

              DBG_PRINT(99,("%d,%d -> %d,%d",x,y,a,b));
              DBG_PRINT(99,("%d,%d",x-a1, y-b1));
              
              if (point_mode || 
                  mask       ||
                  ((mask = obj_inst_getmask(inst))))
              {
                if (check_mask ||
                    ((check_mask = obj_inst_getmask(check_inst))))
                {
                  const int sp1 = ((y-y1)*obj_collwid)+(x-x1);
                  const int sp2 = ((y-b1)*cobj_collwid)+(x-a1);
                  const int cb  = a-x;                  /* pixels per line to check */
                  int cl  = b-y;                  /* lines to check           */

                  uint8 fake_point = 0xFF;
                  uint8 *cm1 = (point_mode?&fake_point:mask + sp1);
                  uint8 *cm2 = check_mask + sp2;

                  int xpos = (sp2 % cobj_collwid)+cb;
                  int ypos = sp2 / cobj_collwid;

                  DBG_MASK(99, check_mask, 32, 32);

                  DBG_PRINT(99,("cl=%d cb=%d %d %d sp2=%d", cl, cb, *cm1, *cm2, sp2));

                   while(cl)
                   {
                     int ccb = 0;

                     xpos -= cb;

                     while(ccb < cb)
                     {
                       #if 0
                       cb_count++;
                       #endif

                       if (*cm1 & *cm2)  /* collision NB: BITWISE AND */
                       {
                         hit = check_inst;

                         /* If only interested in a hit - stop */
                         if (coll_mode == OBJ_COLL_YESNO)
                           goto FOUND_HIT;

                         area++;

                         xmin = UTL_MIN(xmin,xpos);
                         xmax = UTL_MAX(xmax,xpos);
                         ymin = UTL_MIN(ymin,ypos);
                         ymax = UTL_MAX(ymax,ypos);
                       }

                       cm1+=(!point_mode);
                       cm2++;
                       xpos++;
                       ccb++;
                     }

                     cm1 += (point_mode?0:(obj_collwid-cb));
                     cm2 += (cobj_collwid-cb);
                     ypos++;
                     cl--;
                   }
                 }
               }
             }
          } /* PIXEL PERFECT collision detection */
             
          FOUND_HIT:
             
          if (hit)
          {           
            /* -------------------------------------------------------- */
            /* Requester only interested in existense of a collision... */
            /* -------------------------------------------------------- */
            if (coll_mode != OBJ_COLL_FULL)
            {
              g_hit = hit; /* NOT A LIST!!! */
              break;
            }
            else {
              obj_inst_t *newhit = utl_malloc(sizeof(obj_inst_t));
               
              /* allocate our 'hit' */
              if (newhit) {
                g_hit = obj_inst_add(g_hit,newhit);
                 
                /* refer to the real instance hit */
                g_hit->coll = hit;
                
                /* copy select information to our hit record */
                g_hit->rot = hit->rot;
                g_hit->speed = hit->speed;
                g_hit->obj = hit->obj;
                 
                /* COLLISION INFORMATION */ 
                g_hit->x = hit->x-(cobj_collwid/2)+xmin+((xmax-xmin)/2);
                g_hit->y = hit->y-(cobj_collhgt/2)+ymin+((ymax-ymin)/2);
                
                g_hit->var = area;
              }
            }
          }
        }
      }
    }

    check_inst = check_inst->next;
  }
 
  if (list_next)
    list->next = list_next;

  DBG_EXIT(99,("obj_inst_collide rc=%p",g_hit));

  return g_hit;
}

/* obj_inst_create __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_create(int num,
                            int32 x,
                            int32 y,
                            float z,
                            int32 rot,
                            int32 dir,
                            int32 speed,
                            float scale,
                            float fade,
                            uint32 colour)
{
  obj_inst_t *inst = 0;                          
  
  DBG_ENTRY(99,("obj_inst_create(%d)",num));
  
  DBG_PRINT(99,("o=%d x=%d y=%d z=%f r=%d d=%d s=%d s=%f f=%f c=0x%08X",
               num, (int)x, (int)y, (double)z, (int)rot, (int)dir,
               (int)speed, (double)scale, (double)fade, (int)colour));

  if ((inst = utl_malloc(sizeof(obj_inst_t))))
  {
    uint32 var = 0;
   
    if (num < OBJ_SPECIAL_START)
    {
      switch(obj_infos[num].type)
      {
        case OBJ_TILE:
          var = til_inst_create(obj_infos[num].var);
          break;
        
        case OBJ_IMAGE:
          var = img_inst_create(obj_infos[num].var);
          break;
        
        case OBJ_ANIM:
          var = ani_inst_create(obj_infos[num].var);
          break;
        
        case OBJ_SFX:
          var = snd_fx_inst_create(obj_infos[num].var);
          break;

        case OBJ_FONT:
          var = fnt_inst_create(obj_infos[num].var);
          break;
      }
    }

    inst->obj = num;
    inst->var = var;    
    inst->hits_needed = obj_infos[num].hits_needed;
    inst->init_x = inst->x = x;
    inst->init_y = inst->y = y;
    #ifdef OBJ_ACCURATE_MOVEMENT
    inst->real_x = (float)x;
    inst->real_y = (float)y;
    #endif
    inst->init_z = inst->z = z;
    inst->init_rot = inst->rot = rot;
    inst->init_dir = inst->dir = dir;
    inst->init_speed = inst->speed = speed;
    inst->init_scale = inst->scale = scale;
    inst->init_fade = inst->fade = fade;
    inst->init_colour = inst->colour = colour;
    inst->eol = 0;
  }
  DBG_ELSE(0,("failed to allocate obj_inst_t"));
  
  DBG_EXIT(99,("obj_inst_create rc=%p",inst));
  
  return inst;
}

/* obj_inst_add _____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_add(obj_inst_t *list, obj_inst_t *inst)
{
  DBG_ENTRY(99,("obj_inst_add(%p,%p)",list,inst));
  
  if (inst)
  {
    inst->prev = 0;
    inst->next = list;
    
    if (list)
    {
      list->prev = inst;
    }
  }
  else
  {
    inst = list;
  }
  
  DBG_EXIT(99,("obj_inst_add rc=%p",inst));
  
  return inst;
}

/* obj_inst_add_default _____________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_add_default(obj_inst_t *inst)
{
  int class = obj_infos[inst->obj].class;

  DBG_ENTRY(99,("obj_inst_add_default(%p)",inst));
  
  DBG_PRINT(99,("class=%d oldvalue=%p",class, obj_lists[class]));

  obj_lists[class] = obj_inst_add(obj_lists[class],inst);

  DBG_PRINT(99,("class=%d newvalue=%p",class, obj_lists[class]));

  DBG_EXIT(99,("obj_inst_add_default"));
  
  return;
}

/* obj_inst_add_simple ______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_add_simple(int num, int32 x, int32 y, float z)
{
  obj_inst_t *obj_inst = 0;
  
  
  if (num)
  {
    obj_inst = obj_inst_create(num,x,y,z,0,0,0,1,1,0xFFFFFF);
    obj_inst_add_default(obj_inst);
  }
    
  return obj_inst;
}

/* obj_inst_remove __________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

obj_inst_t *obj_inst_remove(obj_inst_t *list, obj_inst_t *inst, int free)
{
  DBG_ENTRY(99,("obj_inst_remove(%p,%p,%d)",list,inst,free));
  
  DBG_PRINT(99,("prev=%p next=%p",inst->prev,inst->next));

  DBG_IF(99,(list==inst),("****LIST==INST****"));

  if (inst->prev)
  {
    inst->prev->next = inst->next;
  }
  else
  {
    list = inst->next;
  }

  if (inst->next)
  {
    inst->next->prev = inst->prev;
  }
    
  inst->prev = inst->next = 0;
  
  if (free)
    obj_inst_free(inst);
  
  DBG_EXIT(99,("obj_inst_remove rc=%p",list));
  
  return list;
}

/* obj_inst_remove_default __________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_remove_default(obj_inst_t *inst, int free)
{
  int class = obj_infos[inst->obj].class;

  DBG_ENTRY(99,("obj_inst_remove_default(%p,%d)",inst,free));
  
  obj_lists[class] = obj_inst_remove(obj_lists[class],inst,free);

  DBG_EXIT(99,("obj_inst_remove_default"));
  
  return;
}

/* obj_inst_list_load _______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_list_load(char *file)
{
  FILE *fp;
  char filename[256];

  DBG_ENTRY(99,("obj_inst_list_load(%s)",file));
  
  sprintf(filename,"%s/game/%s.%s",data_dir,file,OBJ_INST_LIST_EXTENSION);
  
  DBG_PRINT(99,("opening %s",filename));
  
  if ((fp=fopen(filename,"rb")))
  {
    char line[1024];
    int line_no = 0;

    while (fgets(line,sizeof(line),fp))
    {
      char *t = strtok(line, UTL_WHITESPACE);

      if (!t || *t == '#')
        continue;

      switch (line_no)
      {
        default:
        {
          uint32 absolute = 0;
          
          int    rep = 1;
          
          uint32 obj=0;
          int32  x=0,y=0,rot=0,dir=0,speed=0,class=0;
          float  z=0.0f,scale=1.0f,fade=1.0f;
          uint32 x_delta=0, y_delta=0,colour=0xFFFFFF;
          
          switch (*t)
          {
            case 'S':   /* absolute SINGLE */
              absolute = 1;
            case 's':   /* grid based SINGLE */
              break;
            case 'X':   /* absolute REPEATING X */
              absolute = 1;
            case 'x':   /* grid based REPEATING X */
              x_delta=1;
              if ((t=strtok(0,UTL_WHITESPACE))) rep = (int)strtol(t,0,10);
              break;
            case 'Y':   /* absolute REPEATING Y */
              absolute = 1;
            case 'y':   /* grid based REPEATING Y */
              y_delta=1;
              if ((t=strtok(0,UTL_WHITESPACE))) rep = (int)strtol(t,0,10);
              break;
            case 'D':   /* absolute REPEATING DIAGONAL */
              absolute = 1;
            case 'd':   /* grid based REPEATING DIAGONAL */
              x_delta=1;
              y_delta=1;
              if ((t=strtok(0,UTL_WHITESPACE))) rep = (int)strtol(t,0,10);
              break;
            default:    /* unrecognized... */
              rep = 0;
              break;
          }
          
          if (rep)
          {
            uint32 width=0; 
            uint32 height=0;
            
            if ((t=strtok(0,UTL_WHITESPACE))) obj =    (uint32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) x =      (int32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) y =      (int32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) z =      (float)strtol(t,0,10)*0.1f;
            if ((t=strtok(0,UTL_WHITESPACE))) rot =    (int32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) dir =    (int32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) speed =  (int32)strtol(t,0,10);
            if ((t=strtok(0,UTL_WHITESPACE))) scale =  (float)strtod(t,0);
            if ((t=strtok(0,UTL_WHITESPACE))) fade  =  (float)strtod(t,0);
            if ((t=strtok(0,UTL_WHITESPACE))) colour = (int32)strtol(t,0,16);
            if ((t=strtok(0,UTL_WHITESPACE))) class =  (int32)strtol(t,0,10);

            width = obj_infos[obj].wid;
            height = obj_infos[obj].hgt;
            
            DBG_PRINT(99,("width=%d height=%d",(int)width,(int)height));
            
            if (!absolute)
            {
              x *= width;
              y *= height;
            }

            x += width/2;
            y += height/2;
                        
            x_delta = x_delta * width;
            y_delta = y_delta * height;

            /* class not explicitly specified, use info */
            if (class == -1)
            {
              class = obj_infos[obj].class;
            }
            
            while(rep--)
            {
              obj_lists[class] = obj_inst_add(obj_lists[class],
                                              obj_inst_create(obj,
                                                              x,
                                                              y,
                                                              z,
                                                              rot,
                                                              dir,
                                                              speed,
                                                              scale,
                                                              fade,
                                                              colour));

              x+=x_delta;
              y+=y_delta;
            }
          }
        }
        break;
      }

      line_no++;
    }

    fclose(fp);
  }
  
  DBG_EXIT(99,("obj_inst_list_load"));
}

/* obj_inst_list_save _______________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_list_save(char *file)
{
  FILE *fp;
  char filename[256];
 
  DBG_ENTRY(99,("obj_inst_list_save(%s)",file));
  
  sprintf(filename,"%s/temp/%s.%s",data_dir,file,OBJ_INST_LIST_EXTENSION);
  
  DBG_PRINT(99,("opening %s",filename));
  
  if ((fp=fopen(filename,"wb")))
  {
    int  y = 0;

    fprintf(fp, "# ===================================================\n");
    fprintf(fp, "# Rp On  X   Y   Zd  Rot Dir Spd Scl Fde Colour   Cls\n");
    fprintf(fp, "# ---------------------------------------------------\n");

    while(y<vid_playarea_h)
    {
      int x = 0;

      while(x<vid_playarea_w)
      {
        int list = 0;

        while(list<OBJ_MAX_LISTS)
        {
          obj_inst_t *inst = obj_lists[list];

          while(inst)
          {
            if (inst->x == x && inst->y == y)
            {
              int hwid = obj_infos[inst->obj].wid/2;
              int hhgt = obj_infos[inst->obj].hgt/2;

              fprintf(fp, "S   ");
              fprintf(fp, " %03d",  inst->obj);
              fprintf(fp, " %03d",  (int)inst->x-hwid);
              fprintf(fp, " %03d",  (int)inst->y-hhgt);
              fprintf(fp, " %03d",  (int)(inst->z/0.1f));
              fprintf(fp, " %03d",  (int)inst->rot);
              fprintf(fp, " %03d",  (int)inst->dir);
              fprintf(fp, " %03d",  (int)inst->speed);
              fprintf(fp, " %2.1f", (double)inst->scale);
              fprintf(fp, " %2.1f", (double)inst->fade);
              fprintf(fp, " 0x%08X",(unsigned int)inst->colour);
              fprintf(fp, " %03d", list);
              fprintf(fp,"\n");
            }

            inst = inst->next;
          }

          list++;
        }

        x++;
      }

      y++;
    }

    fclose(fp);
  }

  DBG_EXIT(99,("obj_inst_list_save"));
}

/* obj_inst_free ____________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_free(obj_inst_t *inst)
{
  obj_inst_t *curr = inst;
  
  DBG_ENTRY(99,("obj_inst_free(%p)",inst));
  
  if (inst && inst->prev)
  {
    inst->prev->next = 0;
  }
  
  while(curr)
  {
    obj_inst_t *next = curr->next;
   
    utl_free(curr->mask);
   
    DBG_PRINT(99,("TYPE: %d",obj_infos[curr->obj].type));
   
    switch(obj_infos[curr->obj].type)
    {
      case OBJ_TILE:
        til_inst_free(curr->var);
        break;
      
      case OBJ_IMAGE:
        img_inst_free(curr->var);
        break;
        
      case OBJ_ANIM:
        ani_inst_free(curr->var);
        break;
        
      case OBJ_SFX:
        snd_fx_inst_free(curr->var);
        break;

      case OBJ_FONT:
        fnt_inst_free(curr->var);
        break;
    }
    
    utl_free(curr);
    
    curr = next;
  }
  
  DBG_EXIT(99,("obj_inst_free")); 
  
  return;
}

/* obj_inst_free_all ________________________________________________________ */
/*                                                                            */
/* __________________________________________________________________________ */

void obj_inst_free_all(void)
{
  int list;

  for(list=0; list<OBJ_MAX_LISTS; list++)
  {
    obj_inst_free(obj_lists[list]);
        
    obj_lists[list] = 0;
  }

  return;
}

#undef OBJ_C
