#include <kos.h>
#include <stdio.h>
#include <stdlib.h>

#include "Z80/Z80.h"
#include "phoenix.h"
#include "memory.h"
#include "video.h"
#include "roms.h"	/* Include Roms, converted with bin2c.exe */
#include "input.h"
#include "sound.h"
#include "gui.h"

/* ---------------------------------------------------------------------------
   Allocate the relevant portions of memory
 ----------------------------------------------------------------------------*/

int AllocateMem(void)
{
	word count;

	bfont_draw_str(vram_s+40*640+20, 640, 0, "Allocating 32K Memory");

	RAM = malloc(0x8000);                       /* allocate 32k memory for "RAM" */
	CharA = malloc(0x1000);                     /* and 4K for gfx characters */
	CharB = malloc(0x1000);                     /* for 4K more gfx data */
	VRAM0 = malloc(0x1000);                     /* and for video bank0 */
  	VRAM1 = malloc(0x1000);                     /* and for video bank1 */

	if (!RAM || !CharA || !CharB || !VRAM0 || !VRAM1)
	{             				    /* if those didn't work */
		bfont_draw_str(vram_s+60*640+20, 640, 0, "Failed!");
						    /* mention the problem */
		return(1);                          /* and get outta here */
	}

	bfont_draw_str(vram_s+60*640+20, 640, 0, "Done");

	bfont_draw_str(vram_s+80*640+20, 640, 0, "Clearing 32 KB Memory");

	for (count=0x0000;count<0x8000;count++)
		RAM[count]=0;                       /* fill memory with 0's */

	for (count=0x0000;count<0x1000;count++)
	{
	    	VRAM0[count]=0;                     /* also fill both banks */
	    	VRAM1[count]=0;                     /* of VRAM with 0's */
  	}

	bfont_draw_str(vram_s+100*640+20, 640, 0, "Done");

	clearbm();				    /* Clear background bitmaps */

	return (0);
}

/* ---------------------------------------------------------------------------
   ResetMachine resets the emulator (Press Y on control pad)
 ----------------------------------------------------------------------------*/

void ResetMachine(void)
{
  	word count;
  	int oldgame = game;

  	CPURunning=0;
  	if (game == 1) spu_disable();

  	for (count=0x4000;count<0x8000;count++)
    		RAM[count]=0;                       /* delete contents of RAM */


 	for (count=0x0000;count<0x1000;count++)
 	{
 	    	VRAM0[count]=0;                     /* also fill both banks */
 	    	VRAM1[count]=0;                     /* of VRAM with 0's */
   	}


  	clearbm();				    /* Clear Bitmaps */

	clearscr(320, 240);
	vid_set_mode(DM_640x480, PM_RGB565);
	phoenix_gui();				    /* Load GUI */
	clearscr(640, 480);

	if (game != oldgame)
		read_roms();
	else
		RAM[0x7800]=Dips;

	VRAM = 0;				    /* Set VRAM bank */

	vid_set_mode(DM_320x240, PM_RGB565);

	if (game == 1) sound_setup();

	pallete = 1;				    /* Reset Pallete */
  	ResetZ80(&R);                               /* Reset processor */
  	Z80(R);
}

/* ---------------------------------------------------------------------------
   M_WRMEM handles Writing to "Memory", as required by the "CPU"
 ----------------------------------------------------------------------------*/

void M_WRMEM(word A, byte V)
{
	if (A>=0x4000 && A<0x4400)
	{
		if (A>=0x4000 && A<0x4340)     	    /* writing to video A? */
			DirtyCharA[A-0x4000]=1;     /* keep track of changes */

    		if (VRAM) VRAM1[A-0x4000]=V;        /* handle video RAM banking */
    		else VRAM0[A-0x4000]=V;             /* on the memory-write end */
	}
	else if (A>=0x4800 && A<0x4C00)
	{
		if (A>=0x4800 && A<0x4B40)     	    /* for both planes */
			DirtyCharB[A-0x4800]=1;     /* of video memory */
		if (VRAM) VRAM1[A-0x4000]=V;        /* handle video RAM banking */
    		else VRAM0[A-0x4000]=V;             /* on the memory-write end */
	}
	else if (A>=0x5800 && A<0x5C00) RAM[0x5800]=V;   /* mirror scroll register */
	else if (A>=0x5000 && A<0x5400 && (RAM[0x5000]&1)!=(V&1))
	{ 					    /* if changing this bit... */
    		int count;                          /* (will need counter var) */
    		if (V&1) VRAM=1;                    /* set VRAM bank */
    		else VRAM=0;                        /* appropriately */
    		for (count=0;count<832;count++)
    		{         			    /* mark every byte of */
      			DirtyCharA[count]=1;        /* screen RAM "dirty"*/
      			DirtyCharB[count]=1;        /* to force full re-draw */
    		}
    		RAM[0x5000]=(RAM[0x5000]&~1)+(V&1); /* write to base addr */
  	}

	if (A>=0x5000 && A<0x5400 && (RAM[0x5000]&2)!=(V&2))
	{ 					    /* if changing video reg */
		if (V&2)
		{                                   /* raised bit 1? */
			pallete = 2;                /* select alternate pallete */
	    	}
	    	else
	    	{                                   /* must've unset bit 1... */
	      		pallete = 1;                /* select primary pallete */
	    	}

	    	RAM[0x5000]=(RAM[0x5000]&~2)+(V&2); /* store byte at "base" addr */
	}

    	if ((A>=0x6000 && A<=0x63ff && RAM[A]!=V) && game == 1)
    	{
        	RAM[A] = V;
        	if (V == 143)
			sound_effect(2);	    /* Explosion */

		if (! (RAM[0x7000] & 0x10))
		{
			if (V == 79 || (V > 101 && V < 107))
				sound_effect(4);    /* Laser */
		}

        	if (V == 12)
			sound_effect(1);	    /* Eeeep */

        	if (V == 80)
			sound_effect(0);	    /* Blow (ooeer) */
    	}

    	if ((A>=0x6800 && A<=0x6bff && RAM[A] != V) && game == 1)
    	{
        	RAM[A] = V;
        	if (V == 12)
			sound_effect(5);	    /* Shield */

        	if (V == 2)
			sound_effect(3);	    /* Hit */
    	}

	if (A<0x4000||A>=0x8000) return;            /* don't write on ROMs... */
	RAM[A]=V;				    /* OK--put byte in memory */

	return;
}

/* ---------------------------------------------------------------------------
   M_RDMEM handles Fetching Data Stored in "Memory" for the "CPU"
 ----------------------------------------------------------------------------*/

byte M_RDMEM(word A)
{
	if (A<0x4000) return RAM[A];                /* should be most common case */

  	if (A>=0x4000 && A<0x4400)
  	{           				    /* this deal here handles */
    		if (VRAM) return(VRAM1[A-0x4000]);  /* banking the video RAM */
    		else return(VRAM0[A-0x4000]);       /* from the mem-read end */
  	}

  	if (A>=0x4800 && A<0x4C00)
  	{           				    /* this deal here handles */
    		if (VRAM) return(VRAM1[A-0x4000]);  /* banking the video RAM */
    		else return(VRAM0[A-0x4000]);       /* from the mem-read end */
  	}

	if (A>=0x7800 && A<0x7C00)
	{                			    /* reading dipswitches? */
		if(RAM[0x7800]&0x80)
		{                  		    /* if sync bit of switch high */
	      		RAM[0x7800]&=~0x80;         /* clear the bit */
	      		return (RAM[0x7800]|0x80);  /* return value with bit 7 high */
	    	}                           	    /* I don't completely understand the above */
						    /* but it works... I think Phoenix is very */
						    /* unusual in this respect, anyway */
	    	return RAM[0x7800];            	    /* return value from memory */
	}

	else if (A>=0x7000 && A<0x7400)
	{           				    /* if reading input area... */
	    UserInput();                            /* ..get some input... */
	    return RAM[0x7000];                     /*...and return this byte */
	}                                           /* (this achieves "mirroring") */
	  return RAM[A];                            /* return the value from RAM */
}

/* ---------------------------------------------------------------------------
  read_roms reads the game ROMs from the arrays
 ----------------------------------------------------------------------------*/
void read_roms(void)
{
	bfont_draw_str(vram_s+120*640+20, 640, 0,"Reading ROMs");

	switch(game)
	{
		case 1: /* Phoenix */
		load_rom(ic45, RAM, 0x0000, 0x0800);
		load_rom(ic46, RAM, 0x0800, 0x1000);
		load_rom(ic47, RAM, 0x1000, 0x1800);
		load_rom(ic48, RAM, 0x1800, 0x2000);
		load_rom(ic49, RAM, 0x2000, 0x2800);
		load_rom(ic50, RAM, 0x2800, 0x3000);
		load_rom(ic51, RAM, 0x3000, 0x3800);
		load_rom(ic52, RAM, 0x3800, 0x4000);
		load_rom(ic39, CharA, 0x0000, 0x0800);
		load_rom(ic40, CharA, 0x0800, 0x1000);
		load_rom(ic23, CharB, 0x0000, 0x0800);
		load_rom(ic24, CharB, 0x0800, 0x1000);
		RAM[0x7800]=Dips; 		    /* Set Dipswitches */
		break;

		case 2: /* Pleiads */
		load_rom(pleiadsprog, RAM, 0x0000, 0x4000);
		load_rom(pleiadsgfx2, CharA, 0x0000, 0x1000);
		load_rom(pleiadsgfx1, CharB, 0x0000, 0x1000);
		RAM[0x7800]=Dips; 		   /* Set Dipswitches */
		break;

		default:
		break;
	};

	bfont_draw_str(vram_s+140*640+20, 640, 0,"Done");
}

/* ---------------------------------------------------------------------------
  load_rom copies an array into memory
 ----------------------------------------------------------------------------*/

void load_rom(unsigned char *from, byte *to, word start, word end)
{
	word counter;
	word counter2 = 0x0000;

	for (counter=start; counter < end; counter++)
	{
		to[counter] = from[counter2];
		counter2++;
	}

	return;
}

/* ---------------------------------------------------------------------------
   The following "do-nothing" functions must exist to satisfy the Z80 emulator
 ----------------------------------------------------------------------------*/

byte DoIn(byte Port) {return(0);}
void DoOut(byte Port, byte value) {}
void Patch(reg *R) {}
word Interrupt(reg *R){return(0);}