/* tictactoe.c

   Tic-Tac-Toe C file

   Simple implementation of TicTacToe for Sega Dreamcast

   This program uses the awesome KallistiOS and is designed for 1 or 2
   players.  If you're really bored you can also set 2 computers to 
   play against each other.  Obviously, these games end quickly and
   aren't much fun....

   Thanks to crt0 for the Bresenham implementation and other general
   help.

   DirtySanchez, 02/2002
   
   ---------------------------------------------------------------
   Update for using KOS 2.X :)
   
   Indiket, 08/2013

*/

#include <kos.h>
#include <oggvorbis/sndoggvorbis.h>
#include "line.h"
#include "tictactoe.h"
#include "pal.h" // 50/60Hz Selector

#define RGB565(r, g, b) ((r >> 3) << 11)| ((g >> 2) << 5)| ((b >> 3) << 0)

KOS_INIT_FLAGS(INIT_DEFAULT); 
extern uint8 romdisk[];
KOS_INIT_ROMDISK(romdisk);

/* Main Variables */

char matrix[9];              /* Create Board Array */
maple_device_t *mcont;                 /* Create Controller Array */
cont_state_t *cond;               /* Controller Data Struct */
player player1, player2;        /* Players */

/* Main Program Code */

int main(int argc, char **argv)
{
   char done;			/* Game finished variable */
   int playagain;		/* Play again? */

    /*Init video with PM_RGB565 mode */
	int dc_region, ct;
	  
	dc_region = flashrom_get_region();
	ct = vid_check_cable();
	 
	/* Prompt the user for whether to run in PAL50 or PAL60 if the flashrom says
	the Dreamcast is European and a VGA Box is not hooked up. */
	if(dc_region == FLASHROM_REGION_EUROPE && ct != CT_VGA) {
		if(pal_menu()) {
			vid_set_mode(DM_640x480_NTSC_IL, PM_RGB565);
		}
		else {
			vid_set_mode(DM_640x480_PAL_IL, PM_RGB565);
		}
	}
	else vid_set_mode(DM_640x480_NTSC_IL, PM_RGB565);
   
   init_controllers();		
   disp_startup();		/* Display startup info... */

   sndoggvorbis_init();
   sndoggvorbis_start("/rd/music.ogg",1);

   do {
      vid_clear(0,0,0);
      get_players();		/* Select players (human, comp difficulty) */
      vid_clear(0,0,0);
      playagain = 0;		
      done = ' ';
      init_matrix();		/* Init the board */
      timer_spin_sleep(1000);	/* Wait for 1 second... */

      do {
         disp_matrix();		/* Display Board */

	 #ifdef DEBUG
         printf("Bitrate: %s\n",sndoggvorbis_getbitrate());
		 printf("Player 1's turn\n\n");
     #endif
         get_move(player1);	/* Get player1's move */
         disp_matrix();
         done = check();	/* Check if player1 wins or draw */
         if (done != ' ') break;	/* End game? */

	 #ifdef DEBUG
	 printf("Player 2's turn\n\n");
     #endif

         get_move(player2);	/* Get player2's move */
         done = check();	/* Check if player2 wins or draw */
      } while (done == ' ');

      disp_matrix();
      timer_spin_sleep(2000);
      vid_clear(0,0,0);
      if (done == 'X') 
      {
         bfont_draw_str(vram_s + 220*640+250, 640, 0, "Player 1 Wins!!!");
      }
      else if (done == 'O')
      {
         bfont_draw_str(vram_s + 220*640+250, 640, 0, "Player 2 Wins!!!");
      }
      else if (done == 'D')
      {
         bfont_draw_str(vram_s + 220*640+250, 640, 0, "It's a Draw!!!");
      }
     
      bfont_draw_str(vram_s + 268*640+150, 640, 0, "Do you want to play again?");
      bfont_draw_str(vram_s + 292*640+250, 640, 0, "A - Yes, B - No");
      
      while(1) {
         vid_waitvbl();
      
	     cond = (cont_state_t *)maple_dev_status(mcont);
         if (!cond)
         {
	    #ifdef DEBUG
            printf("Error getting controller data\n");
	    #endif
         }
         if (cond->buttons & CONT_A)
         {
            playagain = 1;
            break;
         }
         else if (cond->buttons & CONT_B)
         {
            break;
         }
      }
   } while (playagain == 1);

   sndoggvorbis_stop();
   vid_waitvbl();

   return 0;
}

void init_controllers(void)
{
   //KOS 2.0: Use new Maple functions, and only 1 Controller.
   mcont = maple_enum_type(0, MAPLE_FUNC_CONTROLLER);

   if (!mcont)
   {
      #ifdef DEBUG
      printf("No Controllers Attached!\n");
      #endif
   }
   return;
}

void get_players(void)
{
   int x, y;

   player1.position = PLAYER1;
   player1.difficulty = HUMAN;
   player2.position = PLAYER2;
   player2.difficulty = COMPUTEREASY;
   vid_clear(0,0,0);

   while (1) 
   {
      x = 10;
      y = 34;
      bfont_draw_str(vram_s + y*640+x, 640, 0, "Press A to change player 1");
      y += 24;
      bfont_draw_str(vram_s + y*640+x, 640, 0, "Press B to change player 2");
      y += 48;
      if (player1.difficulty == HUMAN)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 1: Human");
      }
      else if (player1.difficulty == COMPUTEREASY)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 1: Computer-Easy");
      }
      else if (player1.difficulty == COMPUTERMED)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 1: Computer-Medium");
      }
      else if (player1.difficulty == COMPUTERHARD)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 1: Computer-Hard");
      }
      y += 24;
      if (player2.difficulty == HUMAN)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 2: Human");
      }
      else if (player2.difficulty == COMPUTEREASY)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 2: Computer-Easy");
      }
      else if (player2.difficulty == COMPUTERMED)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 2: Computer-Medium");
      }
      else if (player2.difficulty == COMPUTERHARD)
      {
          bfont_draw_str(vram_s + y*640+x, 640, 0, "Player 2: Computer-Hard");
      }   
      timer_spin_sleep(500);
      while(1)
      {  
         vid_waitvbl();
		 cond = (cont_state_t *)maple_dev_status(mcont); //KOS2-Reading Controller
         if (!cond)
         {
	    #ifdef DEBUG
            printf("Error Reading Controller");
	    #endif
         }
         else
         {
            if (cond->buttons & CONT_A)
            {
               player1.difficulty = (player1.difficulty + 1) % 4;
               break;
            }
            if (cond->buttons & CONT_B)
            {
               player2.difficulty = (player2.difficulty + 1) % 4;
               break;
            }
            if (cond->buttons & CONT_START)
            {
               return;
            }
         }
      }
      vid_clear(0,0,0);
   }
}

void init_matrix(void)
{
   int i, x, y;

   for (i = 0; i < 9; i++)
      matrix[i] = ' ';
   
   for (y = 0; y < 480; y++)
   {
      vram_s[y*640+211] = 255;
      vram_s[y*640+212] = 255;
      vram_s[y*640+213] = 255;
      vram_s[y*640+214] = 255;
      vram_s[y*640+215] = 255;
      vram_s[y*640+426] = 255;
      vram_s[y*640+427] = 255;
      vram_s[y*640+428] = 255;
      vram_s[y*640+429] = 255;
      vram_s[y*640+430] = 255;
      if (y % 160 == 0)
      {
         for (x = 0; x < 640; x++)
         {
            vram_s[(y-2)*640+x] = 255;
            vram_s[(y-1)*640+x] = 255;
            vram_s[(y)*640+x] = 255;
            vram_s[(y+1)*640+x] = 255;
            vram_s[(y+2)*640+x] = 255;
         }
      }   
   }
}

void get_move(player playern)
{
   if (playern.difficulty == HUMAN)
      get_player_move(playern.position);
   else 
      get_computer_move(playern);
}

void get_player_move(int playerpos)
{
   int i, x = 0, y = 0;

   do {
      get_controller_move(playerpos, &x, &y);
      timer_spin_sleep(500);
      i = y*3 + x;
      if (matrix[i] != ' ')
      {
	 #ifdef DEBUG
         printf("Invalid move, please try again!\n");
	 #endif
      }
      else
      {
         if (playerpos == PLAYER1)
            matrix[i] = 'X';
         else if (playerpos == PLAYER2)
            matrix[i] = 'O';
      }
   } while (matrix[i] == ' ');
}

void get_computer_move(player playern)
{
   int pickedsquare;

   if (playern.difficulty == COMPUTEREASY)
   {
      pickedsquare = pickRandom();
   } else if (playern.difficulty == COMPUTERMED)
   {
      pickedsquare = makeWin(playern.position);

      if (-1 == pickedsquare)
         pickedsquare = makeBlock(playern.position);

      if (-1 == pickedsquare)
         pickedsquare = pickRandom();
      
   } else if (playern.difficulty == COMPUTERHARD)
   {
      pickedsquare = makeWin(playern.position);

      if (-1 == pickedsquare)
         pickedsquare = makeBlock(playern.position);

      if ((-1 == pickedsquare) && (' ' == matrix[4]))
         pickedsquare = 4;

      if (-1 == pickedsquare)
         pickedsquare = pickCorner();

      if (-1 == pickedsquare)
         pickedsquare = pickRandom();
      
   }

   timer_spin_sleep(500);
   if (playern.position == PLAYER1)
      matrix[pickedsquare] = 'X';
   else if (playern.position == PLAYER2)
      matrix[pickedsquare] = 'O';
}

int pickCorner()
{
   if (' ' == matrix[0])
      return 0;
   else if (' ' == matrix[2])
      return 2;
   else if (' ' == matrix[6])
      return 6;
   else if (' ' == matrix[8])
      return 8;
   else
      return -1;
}

int pickRandom()
{
   int i;

   for (i = 0; i < 9; i++)
   {
      if (matrix[i] == ' ') break;
   }
   return i;
}
      
int makeWin(int position)
{
   if (PLAYER1 == position)
      return EmptySquareInLineWith("X");
   else if (PLAYER2 == position)
      return EmptySquareInLineWith("O");
}

int makeBlock(int position)
{
   if (PLAYER1 == position)
      return EmptySquareInLineWith("O");
   else if (PLAYER2 == position)
      return EmptySquareInLineWith("X");
}

int EmptySquareInLineWith(char player)
{
   int i, sum;
   int grid[9];
   
   for (i = 0; i < 9; i++)
   {
      if ('O' == matrix[i])
         grid[i] = -1;
      else if ('X' == matrix[i])
         grid[i] = 1;
      else
         grid[i] = 0;
   }

   if ('O' == player)
      sum = -2;
   else
      sum = 2;

   /* Top Row Check */

   if (grid[0] + grid[1] + grid[2] == sum)
   {
      if (0 == grid[0])
         return 0;
      else if (0 == grid[1])
         return 1;
      else
         return 2;
   }

   /* Middle Row Check */

   if (grid[3] + grid[4] + grid[5] == sum)
   {
      if (0 == grid[3])
         return 3;
      else if (0 == grid[4])
         return 4;
      else
         return 5;
   }

   /* Bottom Row Check */

   if (grid[6] + grid[7] + grid[8] == sum)
   {
      if (0 == grid[6])
         return 6;
      else if (0 == grid[7])
         return 7;
      else
         return 8;
   }

   /* Left Column Check */

   if (grid[0] + grid[3] + grid[6] == sum)
   {
      if (0 == grid[0])
         return 0;
      else if (0 == grid[3])
         return 3;
      else
         return 6;
   }

   /* Middle Column Check */
   if (grid[1] + grid[4] + grid[7] == sum)
   {
      if (0 == grid[1])
         return 1;
      else if (0 == grid[4])
         return 4;
      else
         return 7;
   }

   /* Right Column Check */

   if (grid[2] + grid[5] + grid[8] == sum)
   {
      if (0 == grid[2])
         return 2;
      else if (0 == grid[5])
         return 5;
      else
         return 8;
   }

   /* diagonal upper-left to lower-right check */

   if (grid[0] + grid[4] + grid[8] == sum)
   {
      if (0 == grid[0])
         return 0;
      else if (0 == grid[4])
         return 4;
      else
         return 8;
   }

   /* diagonal lower-left to upper-right check */

   if (grid[6] + grid[4] + grid[2] == sum)
   {
      if (0 == grid[6])
         return 6;
      else if (0 == grid[4])
         return 4;
      else
         return 2;
   }

   return -1;
}

void disp_matrix(void)
{
   int t, y;

   y = 75;
   
   for (t = 0; t < 9; t = t + 3)
   {
      bfont_draw(vram_s + y*640+105, 640, 0, matrix[t]);
      bfont_draw(vram_s + y*640+315, 640, 0, matrix[t+1]);
      bfont_draw(vram_s + y*640+525, 640, 0, matrix[t+2]);
      y += 150;
   }
   #ifdef DEBUG   
   int i;
   for (i = 0; i < 3; i++)
   {
      printf("%c | %c | %c\n", matrix[i*3], matrix[i*3+1], matrix[i*3+2]);
      if (i < 2)
         printf("---------\n");
      else
         printf("\n\n");
   }
   #endif
}

char check(void)
{
   int i, count;

   for (i = 0; i < 3; i++)
   {
      if ((matrix[i*3] == matrix[i*3+1]) &&
          (matrix[i*3+1] == matrix[i*3+2])
         )
      {
         if (matrix[i*3] == 'X' || matrix[i*3] == 'O')
         {
            line_fast(0,84+(i*150),639,84+(i*150),255,255,255);
            line_fast(0,85+(i*150),639,85+(i*150),255,255,255);
            line_fast(0,86+(i*150),639,86+(i*150),255,255,255);
            return matrix[i*3];
         }
      }
  
      if ((matrix[i] == matrix[3+i]) &&
          (matrix[i] == matrix[6+i])
         )
      {
         if (matrix[i] == 'X' || matrix[i] == 'O')
         {
            line_fast(108+i*210,0,109+i*210,479,255,255,255);
            line_fast(109+i*210,0,110+i*210,479,255,255,255);
            line_fast(110+i*210,0,111+i*210,479,255,255,255);
            return matrix[i];
         }
      }
   }
   if ((matrix[0] == matrix[4]) && 
       (matrix[4] == matrix[8])
      )
   {
      if (matrix[0] == 'X' || matrix[0] == 'O')
      {
         line_fast(0,0,639,479,255,255,255);
         line_fast(1,0,638,479,255,255,255);
         line_fast(0,1,639,478,255,255,255);
         return matrix[0];
      }
   }
   if ((matrix[2] == matrix[4]) && 
       (matrix[4] == matrix[6])
      )
   {
      if (matrix[2] == 'X' || matrix[2] == 'O')
      {
         line_fast(639,0,0,479,255,255,255);
         line_fast(638,0,1,479,255,255,255);
         line_fast(639,1,0,478,255,255,255); 
         return matrix[2];
      }
   }
   
   count = 0;
   for (i = 0; i < 9; i++)
      if (matrix[i] == 'X' || matrix[i] == 'O')
         count++;

   if (count == 9)
      return 'D';
 
   return ' ';
}

void disp_startup(void)
{
   int x = 10, y = 10+24;

   vid_clear(0,0,0);
   bfont_draw_str(vram_s + y*640+x, 640, 0, "This is the game of Tic-Tac-Toe!");
   x = 10;
   y += 24;
   bfont_draw_str(vram_s + y*640+x, 640, 0, "You will be playing against the computer!");
   x = 10;
   y += 48;
   bfont_draw_str(vram_s + y*640+x, 640, 0, "Please press start to continue!");

   x = 10;
   y += 96;
   
   while (1)
   {
      vid_waitvbl();
	  cond = (cont_state_t *)maple_dev_status(mcont);
      if (!cond)
      {
	 #ifdef DEBUG
         printf("Error getting controller status\n");
	 #endif
         return;
      }
      if (cond->buttons & CONT_START)
      {
	 #ifdef DEBUG
         printf("Start Button Pressed! Now starting game\n");
	 #endif
         return;
      }
   }
}

void get_controller_move(int playerpos, int *x, int *y)
{
   int new_x, new_y, select;

   draw_cursor(*x, *y, playerpos);
   select = 0;

   do {
      vid_waitvbl();
	  cond = (cont_state_t *)maple_dev_status(mcont);
	  
      if (!cond)
      {
	 #ifdef DEBUG
         printf("Error getting controller status\n");
	 #endif
         return;
      }
      if (cond->buttons & CONT_DPAD_UP)
      {
         if (*y == 0)
         {
            new_y = 2;
         }
         else
         {
            new_y = *y - 1;
         }
         erase_cursor(*x, *y);
         *y = new_y;
         draw_cursor(*x, *y, playerpos);
     /*    printf("Up Pad Pressed, %d,%d\n",*x,*y);*/
         timer_spin_sleep(500);
      }
      if (cond->buttons & CONT_DPAD_DOWN)
      {
         if (*y == 2)
         {
            new_y = 0;
         }
         else
         {
            new_y = *y + 1;
         }
         erase_cursor(*x, *y);
         *y = new_y;
         draw_cursor(*x, *y, playerpos);
      /*   printf("Down Pad Pressed, %d,%d\n",*x,*y);*/
         timer_spin_sleep(500);
      }
      if (cond->buttons & CONT_DPAD_LEFT)
      {
         if (*x == 0)
         {
            new_x = 2;
         }
         else
         {
            new_x = *x - 1;
         }
         erase_cursor(*x, *y);
         *x = new_x;
         draw_cursor(*x, *y, playerpos);
      /*   printf("Left Pad Pressed, %d,%d\n",*x,*y);*/
         timer_spin_sleep(500);
      }
      if (cond->buttons & CONT_DPAD_RIGHT)
      {
         if (*x == 2)
         {
            new_x = 0;
         }
         else
         {
            new_x = *x + 1;
         }
         erase_cursor(*x, *y);
         *x = new_x;
         draw_cursor(*x, *y, playerpos);
      /*   printf("Right Pad Pressed, %d,%d\n",*x,*y);*/
         timer_spin_sleep(500);
      }
      if (
          (cond->buttons & CONT_A) || (cond->buttons & CONT_B) || (cond->buttons & CONT_X) || (cond->buttons & CONT_Y)
         )
      {
         select = 1;
         timer_spin_sleep(100);
      }
   } while (select == 0);

   erase_cursor(*x, *y);
   
}

void draw_cursor(int x, int y, int playerpos)
{
   int i, j, k, l;

   for (i = 0; i < 10; i++)
   {
      j = 20 + (y * 160);
      k = 105 + (x * 210);
      for (l = 0; l < 10; l++){
		 if (playerpos == PLAYER1)
			vram_s[(j+i)*640+k+l] = RGB565(0,0,255); //Blue
		 else
			vram_s[(j+i)*640+k+l] = RGB565(255,0,0); //Red
	  }
   } 
}

void erase_cursor(int x, int y)
{
   int i, j, k,l;

   for (i = 0; i < 10; i++)
   {
      j = 20 + (y * 160);
      k = 105 + (x * 210);
      for (l = 0; l < 10; l++)
         vram_s[(j+i)*640+k+l] = 0;
   }
}
