//Warlord - Inspired by clasic Atari game
//Updated to KOS 2.x and new PVR + MAPLE API (2013) - by Indiket

#include <kos.h>
KOS_INIT_FLAGS(INIT_DEFAULT);

#include <stdlib.h>
#include <math.h>
#include "pal.h" // 50/60Hz Selector
#include "font.h" // Can draw BIOS font on PVR
extern gpfont_t bios_font;

#define PI (3.14159265f)
#define MAXNUM_BALLS 200
#define MAXNUM_TILES 200
#define EVENT_INTERVAL 1200


float min(float a, float b) {
  return a < b ? a : b;
}

float max(float a, float b) {
  return a > b ? a : b;
}

struct ball_t {
  float x, y, dx, dy, spin;
};
struct point_t {
  float x, y;
};
typedef struct point_t point;
typedef struct ball_t ball;
struct tile_t {
  int alive;
  float w, h, x, y, ang, r, g, b;
  point p1, p2, p3, p4;
};
typedef struct tile_t tile;

float fudge = 1.0f;
int ball_sides = 8;
float ball_radius = 6.0f;

float tilebaseoffsetx = 20.0f;
float tilebaseoffsety = 20.0f;

int movement_threshold = 20;
float movement_speed = 1.0f / 5000.0f;
float tilt_speed = 1.0f / 2500.0f;
float tilt_center_speed = 1.0f / 70.0f;
float max_tilt = PI/8.0f;
float min_angle = PI/25.0f;
float slowdown_factor = 0.5f;
float speedup_factor = 1.8f;
float min_ball_speed = 0.7f;
float max_ball_speed = 6.0f;
float absolute_max_ball_speed = 10.0f;
float ball_adjust_factor = 1.01f;
int event_timer;
float max_computer_speed = 0.02f;

int screenx1 = 20;
int screenx2 = 470;
int screeny1 = 43;
int screeny2 = 443;

float static_z = 1.0f;

float base_angle[4];
float angle[4];
float tilt[4];
int player_type[4];
int players_alive;
point paddle_pos[4][4];
int hit_type[4];
int number_wins[4];

float paddle_width = 50.0f;
float paddle_height = 7.0f;
float paddle_dist = 130.0f;

int num_active_balls;

ball ball_data[MAXNUM_BALLS];
tile tile_data[MAXNUM_TILES];

int num_tiles = 0;

void translate_point(point *p, float dx, float dy)
{
  p->x += dx;
  p->y += dy;
}

void rotate_point(point *p, float theta)
{
  float xx, yy;
  float cth, sth;
  cth = cos(theta);
  sth = sin(theta);
  xx = cth * p->x + sth * p->y;
  yy = -sth * p->x + cth * p->y;
  p->x = xx;
  p->y = yy;
}

void scale_point(point *p, float scale)
{
  p->x *= scale;
  p->y *= scale;
}

/* Replace vertex_oc_t, ta_commit_vertex... for newer functions*/
void draw_square(
		 point p1, point p2, point p3, point p4,
		 float r, float g, float b, float a)
{
  pvr_vertex_t vert;
  
  static_z += 0.1f;
  
  vert.flags = PVR_CMD_VERTEX;
  vert.argb = PVR_PACK_COLOR(a,r,g,b);

  vert.x = p1.x;
  vert.y = p1.y;
  vert.z = static_z;
  pvr_prim(&vert, sizeof(vert));

  vert.x = p2.x;
  vert.y = p2.y;
  pvr_prim(&vert, sizeof(vert));

  vert.x = p3.x;
  vert.y = p3.y;
  pvr_prim(&vert, sizeof(vert));

  vert.x = p4.x;
  vert.y = p4.y;
  vert.flags = PVR_CMD_VERTEX_EOL;
  pvr_prim(&vert, sizeof(vert));
}

void calc_tile(int i)
{
  point p1, p2, p3, p4;
  p1.x = -tile_data[i].w/2.0f;
  p1.y = -tile_data[i].h/2.0f;
  p2.x = tile_data[i].w/2.0f;
  p2.y = -tile_data[i].h/2.0f;
  p3.x = -tile_data[i].w/2.0f;
  p3.y = tile_data[i].h/2.0f;
  p4.x = tile_data[i].w/2.0f;
  p4.y = tile_data[i].h/2.0f;
  rotate_point(&p1, tile_data[i].ang);
  rotate_point(&p2, tile_data[i].ang);
  rotate_point(&p3, tile_data[i].ang);
  rotate_point(&p4, tile_data[i].ang);
  translate_point(&p1, tile_data[i].x, tile_data[i].y);
  translate_point(&p2, tile_data[i].x, tile_data[i].y);
  translate_point(&p3, tile_data[i].x, tile_data[i].y);
  translate_point(&p4, tile_data[i].x, tile_data[i].y);
  tile_data[i].p1 = p1;
  tile_data[i].p2 = p2;
  tile_data[i].p3 = p3;
  tile_data[i].p4 = p4;
}

void add_tiles(float theta, float r, float w, float h, int super)
{
  int i;
  float basex, basey;
  for(i=0; i<4; i++) {
    if(i==0) {
      basex = screenx1+tilebaseoffsetx;
      basey = screeny1+tilebaseoffsety;
    }
    if(i==1) {
      basex = screenx1+tilebaseoffsetx;
      basey = screeny2-tilebaseoffsety;
    }
    if(i==2) {
      basex = screenx2-tilebaseoffsetx;
      basey = screeny2-tilebaseoffsety;
    }
    if(i==3) {
      basex = screenx2-tilebaseoffsetx;
      basey = screeny1+tilebaseoffsety;
    }
    tile_data[num_tiles].x = basex + cos(theta+base_angle[i]) * r;
    tile_data[num_tiles].y = basey - sin(theta+base_angle[i]) * r;
    //    printf("ANGLE is %f\n",theta+base_angle[i]);
    //printf("TILE placed at %f,%f\n",tile_data[num_tiles].x, tile_data[num_tiles].y);
    tile_data[num_tiles].ang = theta+base_angle[i]+0.02f;
    tile_data[num_tiles].w = w;
    tile_data[num_tiles].h = h;
    tile_data[num_tiles].alive = super + 1;
    tile_data[num_tiles].r = 0.0f;
    tile_data[num_tiles].g = 0.0f;
    tile_data[num_tiles].b = 0.0f;
    if(i==0)
      tile_data[num_tiles].r = 1.0f;
    if(i==1) {
      tile_data[num_tiles].r = 1.0f;
      tile_data[num_tiles].b = 1.0f;
    }
    if(i==2)
      tile_data[num_tiles].b = 1.0f;
    if(i==3) {
      tile_data[num_tiles].g = 1.0f;
      tile_data[num_tiles].b = 1.0f;
    }
    num_tiles++;
  }
}

void game_setup(void)
{
  int i;

  player_type[0] = 1;
  player_type[1] = 2;
  player_type[2] = 2;
  player_type[3] = 2;
  players_alive = 4;

  base_angle[0] = - PI / 2.0f;
  base_angle[1] = 0.0f;
  base_angle[2] = PI / 2.0f;
  base_angle[3] = PI;

  event_timer = 0;

  angle[0] = angle[1] = angle[2] = angle[3] = PI / 4.0f;

  tilt[0] = tilt[1] = tilt[2] = tilt[3] = 0.0f;

  hit_type[0] = hit_type[1] = hit_type[2] = hit_type[3] = 0;

  num_active_balls = 1;
  for(i=0; i<MAXNUM_BALLS; i++) {
    ball_data[i].x = 320.0f + ((rand()>>3)%400) - 200;
    ball_data[i].y = 240.0f + ((rand()>>3)%400) - 200;
    ball_data[i].dx = (((float)((rand()>>3)&0xffff) / 65536.0f) - 0.5f) * 8.0f;
    ball_data[i].dy = (((float)((rand()>>3)&0xffff) / 65536.0f) - 0.5f) * 8.0f;
    ball_data[i].spin = 0.0f;
  }

  for(i=0; i<MAXNUM_TILES; i++)
    tile_data[i].alive=0;
  num_tiles = 0;
  add_tiles(0.0f, 0.0f, 32.0f, 32.0f, 0);
  add_tiles(0.0f,    40.0f, 10.0f, 30.0f, 1);
  add_tiles(PI/4.0f, 40.0f, 10.0f, 30.0f, 1);
  add_tiles(PI/2.0f, 40.0f, 10.0f, 30.0f, 1);
  add_tiles(0.0f,    60.0f, 10.0f, 30.0f, 0);
  add_tiles(PI/6.0f, 60.0f, 10.0f, 30.0f, 0);
  add_tiles(PI/3.0f, 60.0f, 10.0f, 30.0f, 0);
  add_tiles(PI/2.0f, 60.0f, 10.0f, 30.0f, 0);
  add_tiles(0.0f*PI/8.0f, 80.0f, 10.0f, 30.0f, 0);
  //  add_tiles(1.0f*PI/8.0f, 80.0f, 10.0f, 30.0f, 0);
  add_tiles(2.0f*PI/8.0f, 80.0f, 10.0f, 30.0f, 0);
  //add_tiles(3.0f*PI/8.0f, 80.0f, 10.0f, 30.0f, 0);
  add_tiles(4.0f*PI/8.0f, 80.0f, 10.0f, 30.0f, 0);
  /*
  add_tiles(0.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  add_tiles(1.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  add_tiles(2.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  add_tiles(3.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  add_tiles(4.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  add_tiles(5.0f*PI/10.0f, 100.0f, 10.0f, 30.0f, 0);
  */
  /*
  tile_data[num_tiles].x = 320.0f;
  tile_data[num_tiles].y = 240.0f;
  tile_data[num_tiles].w = 40.0f;
  tile_data[num_tiles].h = 40.0f;
  tile_data[num_tiles].ang = PI/4.0f;
  tile_data[num_tiles].r = 0.5f;
  tile_data[num_tiles].g = 0.0f;
  tile_data[num_tiles].b = 0.5f;
  tile_data[num_tiles].alive = 100000;
  num_tiles++;
  */
  for(i=0; i<num_tiles; i++)
    calc_tile(i);
}

/* More PVR goodness :) */
void draw_ball(float x, float y)
{
  pvr_vertex_t vert;

  int i;
  static_z += 0.1f;
  vert.flags = PVR_CMD_VERTEX;
  vert.argb = PVR_PACK_COLOR(1.0f,1.0f,1.0f,1.0f);
  vert.z = static_z;

  for(i=0; i<=ball_sides; i++) {
    vert.x = x+cos((float)-i/((float)ball_sides)*2.0f*PI)*ball_radius;
    vert.y = y+sin((float)-i/((float)ball_sides)*2.0f*PI)*ball_radius;
	pvr_prim(&vert, sizeof(vert));

    vert.x = x;
    vert.y = y;
	if(i==ball_sides) vert.flags = PVR_CMD_VERTEX_EOL;
	
	pvr_prim(&vert, sizeof(vert));
  }
}

void draw_tile(int i)
{
  if(tile_data[i].alive==1)
    draw_square(tile_data[i].p1, tile_data[i].p2, tile_data[i].p3, tile_data[i].p4,
		tile_data[i].r, tile_data[i].g, tile_data[i].b, 1.0f);
  else
    draw_square(tile_data[i].p1, tile_data[i].p2, tile_data[i].p3, tile_data[i].p4,
		max(0.8f, tile_data[i].r), 
		max(0.8f, tile_data[i].g),
		max(0.8f, tile_data[i].b), 1.0f);
}

void draw_tiles()
{
  int i;
  for(i=0; i<MAXNUM_TILES; i++)
    if(tile_data[i].alive)
      draw_tile(i);
}

void calc_paddle(int i)
{
  float mx, my;
  float basex, basey;
  point p1, p2, p3, p4;

  if(i==0) {
    basex = screenx1;
    basey = screeny1;
  }
  if(i==1) {
    basex = screenx1;
    basey = screeny2;
  }
  if(i==2) {
    basex = screenx2;
    basey = screeny2;
  }
  if(i==3) {
    basex = screenx2;
    basey = screeny1;
  }

  p1.x = basex + paddle_dist + paddle_height;
  p1.y = basey - paddle_width / 2.0f;
  p2.x = p1.x;
  p2.y = basey + paddle_width / 2.0f;
  p3.x = basex + paddle_dist;
  p3.y = p1.y;
  p4.x = p3.x;
  p4.y = p2.y;

  mx = (p1.x + p2.x + p3.x + p4.x) / 4.0f;;
  my = (p1.y + p2.y + p3.y + p4.y) / 4.0f;

  translate_point(&p1, -mx, -my);
  translate_point(&p2, -mx, -my);
  translate_point(&p3, -mx, -my);
  translate_point(&p4, -mx, -my);

  rotate_point(&p1, tilt[i]);
  rotate_point(&p2, tilt[i]);
  rotate_point(&p3, tilt[i]);
  rotate_point(&p4, tilt[i]);

  translate_point(&p1, mx, my);
  translate_point(&p2, mx, my);
  translate_point(&p3, mx, my);
  translate_point(&p4, mx, my);

  translate_point(&p1, -basex, -basey);
  translate_point(&p2, -basex, -basey);
  translate_point(&p3, -basex, -basey);
  translate_point(&p4, -basex, -basey);

  rotate_point(&p1, angle[i] + base_angle[i]);
  rotate_point(&p2, angle[i] + base_angle[i]);
  rotate_point(&p3, angle[i] + base_angle[i]);
  rotate_point(&p4, angle[i] + base_angle[i]);

  translate_point(&p1, basex, basey);
  translate_point(&p2, basex, basey);
  translate_point(&p3, basex, basey);
  translate_point(&p4, basex, basey);

  paddle_pos[i][0] = p1;
  paddle_pos[i][1] = p2;
  paddle_pos[i][2] = p3;
  paddle_pos[i][3] = p4;
}

void draw_paddles(void)
{
  int i;
  for(i=0; i<4; i++)
    if(player_type[i])
      draw_square(paddle_pos[i][0], paddle_pos[i][1], paddle_pos[i][2], paddle_pos[i][3],
		  1.0f, 1.0f, 0.0f, 1.0f);
}

void draw_balls(void)
{
  int i;
  for(i=0; i<num_active_balls; i++)
    draw_ball(ball_data[i].x, ball_data[i].y);
}

void draw_frame(void)
{
  //PVR: Prepare the screen to draw objects (poly)!
  pvr_poly_hdr_t hdr;
  pvr_poly_cxt_t cxt;

  pvr_poly_cxt_col(&cxt, PVR_LIST_TR_POLY);
  pvr_poly_compile(&hdr, &cxt);

  pvr_prim(&hdr, sizeof(hdr));

  point p1, p2, p3, p4;

  static_z = 1.0f;

  p1.x = 0.0f; p1.y = 0.0f;
  p2.x = 640.0f; p2.y = 0.0f;
  p3.x = 0.0f; p3.y = 480.0f;
  p4.x = 640.0f; p4.y = 480.0f;

  draw_square(p1, p2, p3, p4, 0.3f, 0.3f, 0.3f, 1.0f);

  p1.x = screenx1; p1.y = screeny1;
  p2.x = screenx2; p2.y = screeny1;
  p3.x = screenx1; p3.y = screeny2;
  p4.x = screenx2; p4.y = screeny2;

  draw_square(p1, p2, p3, p4, 0.0f, 0.0f, 0.0f, 1.0f);

  draw_tiles();

  draw_paddles();

  draw_balls();
}

int bounce_ball_off(int bl, float x1, float y1, float x2, float y2, int type)
{
  float slope1, slope2, offset1, offset2, ix, iy;
  float olddx, olddy;
  float l,v1x, v1y, v2x, v2y, dx, dy, m1, m2, m3, m4;
  // First do some quick checks to eliminate easy cases
  if( max(x1, x2) < min(ball_data[bl].x, ball_data[bl].x+ball_data[bl].dx) )
    return 0;
  if( min(x1, x2) > max(ball_data[bl].x, ball_data[bl].x+ball_data[bl].dx) )
    return 0;
  if( max(y1, y2) < min(ball_data[bl].y, ball_data[bl].y+ball_data[bl].dy) )
    return 0;
  if( min(y1, y2) > max(ball_data[bl].y, ball_data[bl].y+ball_data[bl].dy) )
    return 0;
  olddx = ball_data[bl].dx;
  olddy = ball_data[bl].dy;
  // Calculate y=mx+b form of bouncing line
  // If line is vertical, shift slightly
  if(x2 == x1) x2 += 0.00001f;
  slope1 = (y2 - y1) / (x2 - x1);
  offset1 = y1 - slope1 * x1;
  // Calculate y=mx+b form of ball trajectory line
  // If trajectory is vertical, slift slightly
  if(ball_data[bl].dx == 0.0f) ball_data[bl].dx += 0.00001f;
  slope2 = ball_data[bl].dy / ball_data[bl].dx;
  offset2 = ball_data[bl].y - slope2 * ball_data[bl].x;
  // Now mathematically calculate the intersection point
  // y = slope1 x + offset1
  // y = slope2 x + offset2
  // slope1 x + offset1 = slope2 x + offset2
  // (slope1 - slope2) x = offset2 - offset1
  // x = (offset2 - offset1) / (slope1 - slope2)
  if(slope1 == slope2) slope2 += 0.00001f;
  ix = (offset2 - offset1) / (slope1 - slope2);
  iy = slope1 * ix + offset1;
  //Check if intersection is between ends of both lines
  if(ix >= min(x1, x2) - fudge && ix <= max(x1, x2) + fudge
     && ix >= min(ball_data[bl].x, ball_data[bl].x+ball_data[bl].dx) - fudge
     && ix <= max(ball_data[bl].x, ball_data[bl].x+ball_data[bl].dx) + fudge
     && iy >= min(y1, y2) - fudge && iy <= max(y1, y2) + fudge
     && iy >= min(ball_data[bl].y, ball_data[bl].y+ball_data[bl].dy) - fudge
     && iy <= max(ball_data[bl].y, ball_data[bl].y+ball_data[bl].dy) + fudge)
    {
      // First make a perpendicular frame representing the reflection line
      // Normalized to orthonormal vectors
      l = sqrt((x2-x1)*(x2-x1)+(y2-y1)*(y2-y1));
      v1x = (x2 - x1) / l;
      v1y = (y2 - y1) / l;
      v2x = -v1y;
      v2y = v1x;
      // Now v1 points along reflecting line, v2 is normal
      // The idea is that we start with the negation of the
      // motion vector.  We transform this vector to become
      // the resultant reflected vector by aligning with the
      // reflection line perpendicular frame, then hanging
      // the result on a perpendicular frame made up of
      // -v1 , v2.  In the end, this means that we reverse
      // the part of motion travelling tangential to the
      // reflection line.

      // Calculate reflection matrix
      m1 = -v1x*v1x+v2x*v2x;
      m2 = -v1x*v1y+v2x*v2y;
      m3 = -v1x*v1y+v2x*v2y;
      m4 = -v1y*v1y+v2y*v2y;
      dx = - ball_data[bl].dx;
      dy = - ball_data[bl].dy;
      ball_data[bl].dx = m1 * dx + m2 * dy;
      ball_data[bl].dy = m3 * dx + m4 * dy;

      if(type == 1) {
	float compv1, compv2;
	// POWER HIT
	// Calculate component of velocity in tagential direction
	compv1 = ball_data[bl].dx * v1x + ball_data[bl].dy * v1y;
	// Calculate component of velocity in normal direction
	compv2 = ball_data[bl].dx * v2x + ball_data[bl].dy * v2y;
	// Scale up the normal component
	compv2 *= speedup_factor;
	// Recombine
	ball_data[bl].dx = compv1 * v1x + compv2 * v2x;
	ball_data[bl].dy = compv1 * v1y + compv2 * v2y;
      }
      if(type == 2) {
	float compv1, compv2;
	// SLIDING HIT
	// Calculate component of velocity in tagential direction
	compv1 = ball_data[bl].dx * v1x + ball_data[bl].dy * v1y;
	// Calculate component of velocity in normal direction
	compv2 = ball_data[bl].dx * v2x + ball_data[bl].dy * v2y;
	// Weight the normal component really small
	compv2 *= slowdown_factor;
	// Recombine
	ball_data[bl].dx = compv1 * v1x + compv2 * v2x;
	ball_data[bl].dy = compv1 * v1y + compv2 * v2y;
      }
      if(type == 3) {
	// ANDREEV REFLECTION
	ball_data[bl].dx = -olddx;
	ball_data[bl].dy = -olddy;
      }

      return 1;
    }
  return 0;
}


void update_positions(void)
{
  int i, j;
  for(i=0; i<4; i++)
    calc_paddle(i);
  for(i=0; i<num_active_balls; i++) {
    if(ball_data[i].y<screeny1)
      ball_data[i].dy = fabs(ball_data[i].dy);
    if(ball_data[i].y>screeny2)
      ball_data[i].dy = -fabs(ball_data[i].dy);
    if(ball_data[i].x<screenx1)
      ball_data[i].dx = fabs(ball_data[i].dx);
    if(ball_data[i].x>screenx2)
      ball_data[i].dx = -fabs(ball_data[i].dx);
    for(j=0; j<4; j++)
      if(player_type[j])
      {
      if(bounce_ball_off(i,
			 paddle_pos[j][0].x, paddle_pos[j][0].y,
			 paddle_pos[j][1].x, paddle_pos[j][1].y,
			 hit_type[j]))
	;//	printf("Bounce player %d!\n", j);
      if(bounce_ball_off(i,
			 paddle_pos[j][2].x, paddle_pos[j][2].y,
			 paddle_pos[j][3].x, paddle_pos[j][3].y,
			 hit_type[j]))
	;//printf("Bounce player %d!\n", j);
      if(bounce_ball_off(i,
			 paddle_pos[j][0].x, paddle_pos[j][0].y,
			 paddle_pos[j][2].x, paddle_pos[j][2].y,
			 hit_type[j]))
	;//printf("Bounce player %d side!\n", j);
      if(bounce_ball_off(i,
			 paddle_pos[j][1].x, paddle_pos[j][1].y,
			 paddle_pos[j][3].x, paddle_pos[j][3].y,
			 hit_type[j]))
	;//printf("Bounce player %d side!\n", j);
    }
    for(j=0; j<MAXNUM_TILES; j++)
      if(tile_data[j].alive) {
	if(bounce_ball_off(i,
			   tile_data[j].p1.x, tile_data[j].p1.y,
			   tile_data[j].p2.x, tile_data[j].p2.y,
			   0)) {
	  //printf("TILE1\n");
	  tile_data[j].alive --;
	}
	if(bounce_ball_off(i,
			   tile_data[j].p2.x, tile_data[j].p2.y,
			   tile_data[j].p4.x, tile_data[j].p4.y,
			   0)) {
	  //printf("TILE2\n");
	  tile_data[j].alive --;
	}
	if(bounce_ball_off(i,
			   tile_data[j].p3.x, tile_data[j].p3.y,
			   tile_data[j].p4.x, tile_data[j].p4.y,
			   0)) {
	  //printf("TILE3\n");
	  tile_data[j].alive --;
	}
	if(bounce_ball_off(i,
			   tile_data[j].p1.x, tile_data[j].p1.y,
			   tile_data[j].p3.x, tile_data[j].p3.y,
			   0)) {
	  //printf("TILE4\n");
	  tile_data[j].alive --;
	}
      }
    ball_data[i].x += ball_data[i].dx;
    ball_data[i].y += ball_data[i].dy;
    if(fabs(ball_data[i].dx) > max_ball_speed)
      ball_data[i].dx /= ball_adjust_factor;
    if(fabs(ball_data[i].dy) > max_ball_speed)
      ball_data[i].dy /= ball_adjust_factor;
    if(fabs(ball_data[i].dx) + fabs(ball_data[i].dy) < min_ball_speed) {
      ball_data[i].dx *= ball_adjust_factor;
      ball_data[i].dy *= ball_adjust_factor;
    }
    while(fabs(ball_data[i].dx) + fabs(ball_data[i].dy) > absolute_max_ball_speed) {
      ball_data[i].dx *= 0.9;
      ball_data[i].dy *= 0.9;
    }  
  }  
  for(i=0; i<MAXNUM_TILES; i++)
    if(tile_data[i].alive < 0)
      tile_data[i].alive = 0;
  if(tile_data[0].alive == 0)
    player_type[0] = 0;
  if(tile_data[1].alive == 0)
    player_type[1] = 0;
  if(tile_data[2].alive == 0)
    player_type[2] = 0;
  if(tile_data[3].alive == 0)
    player_type[3] = 0;
  players_alive = 0;
  if(tile_data[0].alive) players_alive++;
  if(tile_data[1].alive) players_alive++;
  if(tile_data[2].alive) players_alive++;
  if(tile_data[3].alive) players_alive++;
  event_timer++;
  if(event_timer == EVENT_INTERVAL) {
    ball_data[num_active_balls].x = ball_data[0].x;
    ball_data[num_active_balls].y = ball_data[0].y;
    ball_data[num_active_balls].dx = ball_data[0].dx;
    ball_data[num_active_balls].dy = ball_data[0].dy;
    ball_data[num_active_balls].dx += (((float)((rand()>>3)%1024)/1024.0f) - 0.5f);
    ball_data[num_active_balls].dy += (((float)((rand()>>3)%1024)/1024.0f) - 0.5f);
    event_timer=0;
    num_active_balls ++;
  }
}

void gravity(float x, float y, float lvl)
{
  int i;
  float d;
  for(i=0; i<num_active_balls; i++) {
    d = (x-ball_data[i].x)*(x-ball_data[i].x) + (y-ball_data[i].y)*(y-ball_data[i].y);
    ball_data[i].dx += lvl * (x-ball_data[i].x) / d;
    ball_data[i].dy += lvl * (y-ball_data[i].y) / d;
  }
}

void computer_move()
{
  int i,j;
  float basex, basey;
  float d, ddx, ddy, ang, closest;
  for(i=0; i<4; i++)
    if(player_type[i] == 2) {
      if(i==0) {
	basex = screenx1;
	basey = screeny1;
      }
      if(i==1) {
	basex = screenx1;
	basey = screeny2;
      }
      if(i==2) {
	basex = screenx2;
	basey = screeny2;
      }
      if(i==3) {
	basex = screenx2;
	basey = screeny1;
      }
      closest = 10000.0f;
      ang = 0.0f;
      for(j=0; j<num_active_balls; j++) {
	ddx = ball_data[j].x - basex;
	ddy = basey - ball_data[j].y ;
	d = sqrt( ddx * ddx + ddy * ddy );
	if(d < closest) {
	  closest = d;
	  ang = atan2(ddy, ddx);
	  if( d < paddle_dist )
	    ang += PI / 2.0f;
	}
      }
      ang -= base_angle[i];
      if(ang < 0.0f) ang+=2.0f*PI;
      if(ang > 2.0f*PI ) ang-=2.0f*PI;
      if( fabs(ang - angle[i]) < max_computer_speed )
	angle[i] = ang;
      else {
	if( angle[i] < ang )
	  angle[i] += max_computer_speed;
	else
	  angle[i] -= max_computer_speed;
      }
    }
}

/* KOS update: Prepare TA region. Also, add 50/60Hz selector ;) */
int main(int argc, char **argv) {
	//Update Maple
	maple_device_t *c;
	cont_state_t *cond;
	
	int i;
	int mov;
	int mode;
	char buf[80];
	int win = 0;

	/*Init video with PM_RGB565 mode*/
	int dc_region, ct;
	  
	dc_region = flashrom_get_region();
	ct = vid_check_cable();
	 
	/* Prompt the user for whether to run in PAL50 or PAL60 if the flashrom says
		the Dreamcast is European and a VGA Box is not hooked up. */
	if(dc_region == FLASHROM_REGION_EUROPE && ct != CT_VGA) {
		if(pal_menu()) {
			vid_set_mode(DM_640x480_NTSC_IL, PM_RGB565);
		}
		else {
			vid_set_mode(DM_640x480_PAL_IL, PM_RGB565);
		}
	}
	else vid_set_mode(DM_640x480_NTSC_IL, PM_RGB565);
	  
	//Init the PVR system
	pvr_init_defaults();
	
	//Next, prepare MAPLE Controller
	c = maple_enum_type(0, MAPLE_FUNC_CONTROLLER);
		
	for(i=0; i<4; i++)
	  number_wins[i]=0;
	
	game_setup();
	mode = 0;
	
	//Init our PVR font, cause bfont_draw_str is not compatible when touching PVR
	gpfont_init();
	
	while(1) {
	
	  /* Check key status */
	  cond = (cont_state_t *)maple_dev_status(c);
	  if (!cond) {
		printf("Error reading controller\n");
	    break;
	  }

	  if (cond->buttons & CONT_START)
	    break;

	  //Mode 0 is ingame
	  if(mode == 0) {

	    //Read X-Y values
		mov = cond->joyx - 127 - (cond->joyy - 127);
		
	    if(mov < - movement_threshold)
	      angle[0] += movement_speed * (mov + movement_threshold);
	    if(mov > movement_threshold)
	      angle[0] += movement_speed * (mov - movement_threshold);
	    for(i=0; i<4; i++) {
	      if(angle[i] < min_angle)
		angle[i] = min_angle;
	      if(angle[i] > PI / 2.0f - min_angle)
		angle[i] = PI / 2.0f - min_angle;
	    }

		//Read triggers
		mov = cond->ltrig - cond->rtrig;
		
	    if(mov < - movement_threshold)
	      tilt[0] += tilt_speed * (mov + movement_threshold);
	    if(mov > movement_threshold)
	      tilt[0] += tilt_speed * (mov - movement_threshold);
	    for(i=0; i<4; i++) {
	      if(tilt[i] < - max_tilt)
		tilt[i] = - max_tilt;
	      if(tilt[i] > max_tilt)
		tilt[i] = max_tilt;
	      if(tilt[i] < 0.0f) tilt[i] += tilt_center_speed;
	      if(tilt[i] > 0.0f) tilt[i] -= tilt_center_speed;
	    }

		//Finally, type of shoot
	    hit_type[0] = 0;
	    if(cond->buttons & CONT_A)
	      hit_type[0] = 1;
	    if(cond->buttons & CONT_B)
	      hit_type[0] = 2;
	    if(cond->buttons & CONT_X)
	      hit_type[0] = 3;

	    /*
	    if(!(cond.buttons & CONT_B))
	      gravity(screenx1, screeny1, 10.0f);
	    if(!(cond.buttons & CONT_X))
	      gravity(screenx1, screeny1, -10.0f);
	    */
	    computer_move();

	    update_positions();

		/*Next, PVR render code*/
		pvr_scene_begin();
		pvr_list_begin(PVR_LIST_OP_POLY);
		pvr_list_finish(); 	/* End of opaque list */
		pvr_list_begin(PVR_LIST_TR_POLY);
		draw_frame();
		pvr_list_finish(); /* End of TR list */
		pvr_scene_finish();
		//Must wait ending PVR
		while(pvr_check_ready()!=0){

		}
		pvr_wait_ready();
		
		//Check: when only 1 player alive, ends!
	    if(players_alive <= 1)
	      mode = 1;

	  }
	  
	  /*Checking if game over*/
	  if(mode >= 1) {
	  
		//Render PVR Code
		pvr_scene_begin();
		pvr_list_begin(PVR_LIST_OP_POLY);
		pvr_list_finish(); 	/* End of opaque list */
		pvr_list_begin(PVR_LIST_TR_POLY);
				
		//Draw "GAME OVER"
	    //bfont_draw_str(vram_s+200+200*640,640,1.0,"GAME OVER");
		gpfont_printf(&bios_font, 200, 200, "GAME OVER");
		
		//Reset values
	    if(mode == 1) {
	      win = -1;
	      for(i=0; i<4; i++)
		if(player_type[i])
		  win = i;
	      if(win>=0) number_wins[win]++;
	      mode = 2;
	    }

	    if(win>=0)
	      sprintf(buf,"Winner is player %d", win+1);
	    else
	      sprintf(buf,"EVERYONE DIED!");
		
		//Draw more text
	    //bfont_draw_str(vram_s+200+250*640,640,1.0,buf);
		gpfont_printf(&bios_font, 200, 250, buf);
	    sprintf(buf,"%d",number_wins[0]);
	    //bfont_draw_str(vram_s+50+50*640,640,1.0, buf);
		gpfont_printf(&bios_font, 50, 50, buf);
	    sprintf(buf,"%d",number_wins[1]);
	    //bfont_draw_str(vram_s+50+100*640,640,1.0, buf);
		gpfont_printf(&bios_font, 50, 100, buf);
	    sprintf(buf,"%d",number_wins[2]);
	    //bfont_draw_str(vram_s+250+100*640,640,1.0, buf);
		gpfont_printf(&bios_font, 250, 100, buf);
	    sprintf(buf,"%d",number_wins[3]);
	    //bfont_draw_str(vram_s+250+50*640,640,1.0, buf);
		gpfont_printf(&bios_font, 250, 50, buf);

		//Press Y to restart
	    if(cond->buttons & CONT_Y) {
	      game_setup();
	      mode = 0;
	    }
		
		pvr_list_finish(); /* End of TR list */
		pvr_scene_finish();
		//Must wait ending PVR
		while(pvr_check_ready()!=0){

		}
		pvr_wait_ready();
	  }
	}
	
	//Shutdown PVR system
	pvr_shutdown();
  
	return 0;
}


