/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
#include "WinToolKitGeneral.h"

///////////////////////////////////////////////////////////////////
// General Methods
///////////////////////////////////////////////////////////////////

Uint32 getPixel(SDL_Surface *surface, int x, int y)
{
	int bpp = surface->format->BytesPerPixel;
	// Here p is the address to the pixel we want to retrieve
	Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;

	switch(bpp)
	{
	case 1:
		return *p;
	case 2:
		return *(Uint16 *)p;
	case 3:
		if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
			return p[0] << 16 | p[1] << 8 | p[2];
		else
			return p[0] | p[1] << 8 | p[2] << 16;
	case 4:
		return *(Uint32 *)p;
	default:
		return 0; // shouldn't happen, but avoids SV_WARNINGs
	}
}



// Set the pixel at (x, y) to the given value
// NOTE: The surface must be locked before calling this!
void putPixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
{
	// Check to make sure the pixel is in range first.
	if (x < 0 || x > surface->w)
		return;
	if (y < 0 || y > surface->h)
		return;
		
	int bpp = surface->format->BytesPerPixel;
	// Here p is the address to the pixel we want to set
	Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;

	switch(bpp)
	{
	case 1:
		*p = pixel;
		break;
	case 2:
		*(Uint16 *)p = pixel;
		break;
	case 3:
		if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
		{
			p[0] = (pixel >> 16) & 0xff;
			p[1] = (pixel >> 8) & 0xff;
			p[2] = pixel & 0xff;
		} 
		else
		{
			p[0] = pixel & 0xff;
			p[1] = (pixel >> 8) & 0xff;
			p[2] = (pixel >> 16) & 0xff;
		}
		break;
	case 4:
		*(Uint32 *)p = pixel;
		break;
	}
}

// Calculate the angles between the start and end point
// If we need it in terms of the screen then flip the y component
// I.e. 0,0 to be in the top left corner of the screen - screen mode
// or 0,0 at bottom left
double quadrantise(Point start, Point end, bool screen = false)
{
	return quadrantise(end.x - start.x, end.y - start.y, screen);
}

double quadrantise(double x, double y, bool screen = false)
{
	double angle = 0.0;

	if (screen)
		y *= -1;

	if (y != 0 && x != 0)
		angle = fabs(atan(y/x));
 
	if (x < 0 && y > 0) // #left and down
		angle = M_PI - angle;
	else if (x < 0 && y < 0) // #left and up
		angle = M_PI + angle;
	else if (x > 0 && y < 0) // #right and up
		angle = 2*M_PI - angle;
	else if (x == 0 && y > 0) // #vertical axis
		angle = M_PI/2;
	else if (x == 0 && y < 0)
		angle = 3*M_PI/2;
	else if (y == 0 && x < 0) //#horizontal axis
		angle = M_PI;
	else if (y == 0 && x > 0)
		angle = 0;

	return float(angle);
}

// Attach a filename to the end of a path
void mergeFilename(const char* path, const char* filename, char* fullFilename)
{
	strcpy(fullFilename, path);
	
	strcat(fullFilename, filename);
}


bool fileExist(const char* filename)
{
	std::ifstream the_file (filename);
	// Always check to see if file opening succeeded
	if (the_file.is_open())
	{
		the_file.close();
		return true;
	}
	return false;
}

