/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "Drawing.h"

Drawing::Drawing()
{}
Drawing::~Drawing()
{}


void Drawing::drawSurface (SDL_Surface* screen, SDL_Surface* graphic, Point position, float rotation)
{
	Drawing::drawOffsetSurface (screen, graphic, position, rotation, 0,0);
}

void Drawing::drawOffsetSurface (SDL_Surface* screen, SDL_Surface* graphic, Point position, float rotation, Point offset)
{
	position -= offset;
	Drawing::drawSurface (screen, graphic, position, rotation);
}	

void Drawing::drawOffsetSurface (SDL_Surface* screen, SDL_Surface* graphic, Point position, float rotation, int xOffsetPct, int yOffsetPct)
{
	int xOffset=0;
	int yOffset=0;
	int xOffsetC=0;
	int yOffsetC=0;
	int distCentre=0;
	
	// Re-adjust the percents if out of bounds.
	if (xOffsetPct > 100) 	xOffsetPct = 100;
	if (xOffsetPct < 0) 	xOffsetPct = 0;
	if (yOffsetPct > 100) 	yOffsetPct = 100;
	if (yOffsetPct < 0) 	yOffsetPct = 0;	
	
	SDL_Surface* rotResGraphic;
	
	//if (rotation == 0.0)
		rotResGraphic = graphic;
	//else
	//	rotResGraphic = Drawing::getRotatedGraphic(resourceID, rotation, surface);

	// Calculate the offset of the image relative to the centre of the image.
	// This is where the image is going to be rotated about. Then calculate the 
	// new position based on the rotation of the image. Then last of all relate
	// this new offset back to the 0,0 position of the new image.
	if (xOffsetPct > 0 || yOffsetPct > 0)
	{
	
		// Calculate where the offset is relative to 0,0 top left
		if (xOffsetPct > 0)
			xOffset		= graphic->w * xOffsetPct / 100;
		if (yOffsetPct > 0)
			yOffset		= graphic->h * yOffsetPct / 100;
	
		// Find them reltative to the centre of the old image.
		xOffsetC 	= xOffset - (graphic->w >> 1);
		yOffsetC	= yOffset - (graphic->h >> 1);
	
		// Adjust the Offset for the rotation.
		// Use the distance from the centre multiplied by the angle
		if (rotation != 0.0)
		{
			distCentre  = (int)round(sqrt ((xOffsetC * xOffsetC) + (yOffsetC * yOffsetC)));
			xOffsetC 	= (int)round(sin(rotation) * distCentre);
			yOffsetC 	= (int)round(cos(rotation) * distCentre);
		}
		
		// Re-align the offset relative to the centre of the new image.
		xOffset 	= xOffsetC + (rotResGraphic->w >> 1);
		yOffset 	= yOffsetC + (rotResGraphic->h >> 1);
	}
		
	SDL_Rect src, dst;
	src.x=0;
	src.y=0;
	src.h=rotResGraphic->h;
	src.w=rotResGraphic->w;
	
	dst = src;
	dst.x=(short int)round(position.x) - xOffset;
	dst.y=(short int)round(position.y) - yOffset;
	
	Log::Instance()->log("pos (%d,%d), (%f,%f)", dst.x, dst.y, position.x - xOffset, position.y - yOffset); 
	
	#ifdef DRAW_BOUNDING
		Rect size(dst.x, dst.y, dst.x + dst.w, dst.y + dst.h);
		SDL_Colour Colour = {0xff, 0,0,0};
		Drawing::drawRect(screen, Colour, size); 
	#endif
	
	SDL_BlitSurface(rotResGraphic, &src, screen, &dst);
	
}
	 
void Drawing::drawArc(SDL_Surface* surf, Uint32 colour, Point centre, Point startPos, Point endPos, Rotation rot)
{
	int radius = (int)centre.distanceFrom(startPos);
	
	Point startDiff = startPos;
	startDiff -= centre;
	double startAngle = quadrantise(startDiff.x, startDiff.y, false);
	
	Point endDiff = endPos;
	endDiff -= centre;
	double endAngle = quadrantise(endDiff.x, endDiff.y, false);
	
	Drawing::drawArc(surf, colour, centre, startAngle, endAngle, radius, rot);
	
}

void Drawing::drawArc(SDL_Surface* surf, Uint32 colour, Point centre, double startAngle, double endAngle, double radius, Rotation rot)
{
	double aStep = 0.0;            // Angle Step (rad)
    double a = 0.0;                // Current Angle (rad)
    double minAngle = 0.0;
    double maxAngle = 0.0;
    int x_last, x_next, y_last, y_next;
    int x,y;

	// Adjust the smallest angle if the arc crosses the zero boundary
	// this saves having to cross the 2PI mark
	if (rot == ROT_CLOCKWISE && startAngle < endAngle)
		startAngle += 2*M_PI;		
    else if(rot == ROT_ANTI_CLOCKWISE && endAngle < startAngle)
   		endAngle += 2*M_PI;
   	
   	// Now the angles are correct then move from the smallest angle to the largest
   	if (startAngle < endAngle)
   	{
   		minAngle = startAngle;
   		maxAngle = endAngle;
   	}
   	else
   	{
   		minAngle = endAngle;
   		maxAngle = startAngle;
   	}
   		
	aStep = M_PI / 180;
	
	// Cache the start x and y positions
	x = (int)centre.x;
	y = (int)centre.y;

    x_last = (int)round(x + cos(minAngle) * radius);
    y_last = (int)round(y - sin(minAngle) * radius);

    for(a=minAngle + aStep; a <= maxAngle; a += aStep)
    {
      x_next = x + (int)round(cos(a) * radius);
      y_next = y - (int)round(sin(a) * radius);
      Drawing::drawLine(surf, colour, x_last, y_last, x_next, y_next);
      x_last = x_next;
      y_last = y_next;
    }
}

void Drawing::drawCircle(SDL_Surface* surf, Uint32 colour, Point centre, int radius)
{
	Drawing::drawCircle(surf, colour, (int)centre.x, (int)centre.y, radius);
}

void Drawing::drawCircle(SDL_Surface* surf, Uint32 colour, int centreX, int centreY, int radius)
{
	int x = 0;
    int y = radius;
    int p = (5 - radius*4)/4;
    
    Drawing::circlePoints(surf, colour, centreX, centreY, x, y);
    while (x < y) {
        x++;
        if (p < 0) {
            p += 2*x+1;
        } else {
            y--;
            p += 2*(x-y)+1;
        }
        Drawing::circlePoints(surf, colour, centreX, centreY, x, y);
    }
}

void Drawing::circlePoints(SDL_Surface* surf, Uint32 colour, int cx, int cy, int x, int y)
{
    if (x == 0) {
        putPixel(surf, cx, cy + y, colour);
        putPixel(surf, cx, cy - y, colour);
        putPixel(surf, cx + y, cy, colour);
        putPixel(surf, cx - y, cy, colour);
    } else 
    if (x == y) {
    	putPixel(surf, cx + x, cy + y, colour);
    	putPixel(surf, cx - x, cy + y, colour);
    	putPixel(surf, cx + x, cy - y, colour);
    	putPixel(surf, cx - x, cy - y, colour);
    } else 
    if (x < y) {
    	putPixel(surf, cx + x, cy + y, colour);
    	putPixel(surf, cx - x, cy + y, colour);
    	putPixel(surf, cx + x, cy - y, colour);
    	putPixel(surf, cx - x, cy - y, colour);
    	
    	putPixel(surf, cx + y, cy + x, colour);
    	putPixel(surf, cx - y, cy + x, colour);
    	putPixel(surf, cx + y, cy - x, colour);
    	putPixel(surf, cx - y, cy - x, colour);
    }
}

void Drawing::drawDisc(SDL_Surface* surf, Uint32 colour, Point centre, int radius)
{
	Drawing::drawDisc(surf, colour, (int)centre.x, (int)centre.y, radius);
}

void Drawing::drawDisc(SDL_Surface* surf, Uint32 colour, int centreX, int centreY, int radius)
{
	int x,y;
    int x_comp, y_comp;
    double r2 = radius * radius;
    
	// Cache the start x and y positions
	x = centreX;
	y = centreY;

	// Start at the top of the disc i.e. y = radius
	// work our way down to the centre a pixel at a time and calculate the x component
	// then draw a horizonal line to at y and -y within the bounds of x, -x	
	for (y_comp = radius; y_comp >=0; y_comp--)
	{
		x_comp = (int)round(sqrt(r2 - y_comp * y_comp));
		
		Drawing::drawLine(surf, colour, x - x_comp, y + y_comp, x + x_comp, y + y_comp);
      	Drawing::drawLine(surf, colour, x - x_comp, y - y_comp, x + x_comp, y - y_comp);
	}
}

void Drawing::drawLine(SDL_Surface* surf, Uint32 colour, Point startPos, Point endPos)
{
	Drawing::drawLine(surf, colour, (int)startPos.x, (int)startPos.y, (int)endPos.x, (int)endPos.y);
}

void Drawing::drawLine(SDL_Surface* surf, Uint32 colour, int x, int y, int x2, int y2)
{
	bool yLonger=false;
	int shortLen=y2-y;
	int longLen=x2-x;
	if (abs(shortLen)>abs(longLen)) {
		int swap=shortLen;
		shortLen=longLen;
		longLen=swap;				
		yLonger=true;
	}
	int decInc;
	if (longLen==0) decInc=0;
	else decInc = (shortLen << 16) / longLen;

	if (yLonger) {
		if (longLen>0) {
			longLen+=y;
			for (int j=0x8000+(x<<16);y<=longLen;++y)
			{
				putPixel(surf,j >> 16, y, colour);	
				j+=decInc;
			}
			return;
		}
		longLen+=y;
		for (int j=0x8000+(x<<16);y>=longLen;--y)
		{
			putPixel(surf,j >> 16, y, colour);	
			j-=decInc;
		}
		return;	
	}

	if (longLen>0)
	{
		longLen+=x;
		for (int j=0x8000+(y<<16);x<=longLen;++x)
		{
			putPixel(surf, x, j >> 16, colour);
			j+=decInc;
		}
		return;
	}
	longLen+=x;
	for (int j=0x8000+(y<<16);x>=longLen;--x)
	{
		putPixel(surf, x, j >> 16, colour);
		j-=decInc;
	}
}

void Drawing::drawRect(SDL_Surface* surf, Uint32 colour, Point topLeft, Point bottomRight)
{
	Drawing::drawLine(surf, colour, (int)topLeft.x, (int)topLeft.y, (int)bottomRight.x, (int)topLeft.y);
	Drawing::drawLine(surf, colour, (int)bottomRight.x, (int)topLeft.y, (int)bottomRight.x, (int)bottomRight.y);
	Drawing::drawLine(surf, colour, (int)topLeft.x, (int)bottomRight.y, (int)bottomRight.x, (int)bottomRight.y);
	Drawing::drawLine(surf, colour, (int)topLeft.x, (int)topLeft.y, (int)topLeft.x, (int)bottomRight.y);
}

void Drawing::drawRect(SDL_Surface* surf, Uint32 colour, Rect size)
{
	Drawing::drawRect(surf, colour, size.topLeft, size.bottomRight);	
}

SDL_Surface* Drawing::newsurf_fromsurf(SDL_Surface* surf, int width, int height)
{
	SDL_Surface* newsurf;

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
    {
        Log::Instance()->log("Create surface. Unsupported Surface bit depth for transform");
		return NULL;
    }
      
	newsurf = SDL_CreateRGBSurface(surf->flags, width, height, surf->format->BitsPerPixel,
	surf->format->Rmask, surf->format->Gmask, surf->format->Bmask, surf->format->Amask);
	if(!newsurf)
    {
        Log::Instance()->log("Create surface failed %s", SDL_GetError());
		return NULL; 
    }

	/* Copy palette, colorkey, etc info */
	if(surf->format->BytesPerPixel==1 && surf->format->palette)
		SDL_SetColors(newsurf, surf->format->palette->colors, 0, surf->format->palette->ncolors);
	if(surf->flags & SDL_SRCCOLORKEY)
		SDL_SetColorKey(newsurf, (surf->flags&SDL_RLEACCEL)|SDL_SRCCOLORKEY, surf->format->colorkey);

	return newsurf;
}
