/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
#include "Button.h"

Button::Button(const char* name, Rect rect) : Control((int)rect.topLeft.x, (int)rect.topLeft.y)
{
	this->name = strdup(name);
	this->width = (int)rect.width();
	this->height = (int)rect.height();
	this->backgroundImage = NULL;
	this->activeBackgroundImage = NULL;
	this->mouseOverBackgroundImage = NULL;
	this->accessKey = SDLK_UNKNOWN;
	this->font = NULL;
	this->isMouseDown = false;
	this->isMouseOver = false;
}

Button::~Button()
{
	if (this->backgroundImage != NULL)
		SDL_FreeSurface(this->backgroundImage);
		
	if (this->activeBackgroundImage != NULL)
		SDL_FreeSurface(this->activeBackgroundImage);
	
	delete name;
	
}

void Button::loadBackgroundImage(const char* image)
{
	// Clean up the image if already loaded
	if (this->backgroundImage != NULL)
		SDL_FreeSurface(this->backgroundImage);
	
	// Load the onto the button.
	this->backgroundImage = IMG_Load(image);

	if (!this->backgroundImage)
		Log::Instance()->log("Trouble loading image %s", image);
}

void Button::loadActiveBackgroundImage(const char* image)
{
	// Clean up the image if already loaded
	if (this->activeBackgroundImage != NULL)
		SDL_FreeSurface(this->activeBackgroundImage);
	
	// Load the onto the button.
	this->activeBackgroundImage = IMG_Load(image);

	if (!this->activeBackgroundImage)
		Log::Instance()->log("Trouble loading image %s", image);
}

void Button::loadMouseOverBackgroundImage(const char* image)
{
	// Clean up the image if already loaded
	if (this->mouseOverBackgroundImage != NULL)
		SDL_FreeSurface(this->mouseOverBackgroundImage);
	
	// Load the onto the button.
	this->mouseOverBackgroundImage = IMG_Load(image);

	if (!this->mouseOverBackgroundImage)
		Log::Instance()->log("Trouble loading image %s", image);
}

void Button::setAccessKey(SDLKey key)
{
	this->accessKey = key;
}

void Button::setFont(Font* font)
{
	this->font = font;
}

void Button::setFont(char* fontFileName)
{
	this->font = FontManager::Instance()->getFont(fontFileName);
}

void Button::draw(SDL_Surface* screenDest)
{
	SDL_Rect screenRect;
	screenRect.x = this->x;
	screenRect.y = this->y;
	screenRect.w = this->width;
	screenRect.h = this->height;

	// Draw the background
	if (this->active && this->activeBackgroundImage)
		SDL_BlitSurface(this->activeBackgroundImage, NULL, screenDest, &screenRect);
	else if (this->mouseOver == true && this->mouseOverBackgroundImage)
		SDL_BlitSurface(this->mouseOverBackgroundImage, NULL, screenDest, &screenRect);
	else if (this->backgroundImage)
		SDL_BlitSurface(this->backgroundImage, NULL, screenDest, &screenRect);	
	else
	{
		// Draw a solid background colour with a beveled edge
		if (this->mouseOver == true)
			SDL_FillRect (screenDest, &screenRect, this->mouseOverColour);
		else
			SDL_FillRect (screenDest, &screenRect, this->activeColour);
			
		for (int i = 0; i < 1; i++)
		{
			// Draw the top & left
			Drawing::drawLine(screenDest, this->borderColTopLeft, this->x + i, this->y + i, this->x + i , this->y + this->height - 1 - i);
			Drawing::drawLine(screenDest, this->borderColTopLeft, this->x + 1 + i, this->y + i, this->x + this->width - 1 - 2 * i, this->y + i);
			// Draw the bottom & right
			Drawing::drawLine(screenDest, this->borderColBottomRight, this->x + i, this->y + this->height - 1 - i, this->x + this->width - 1 - 2 * i, this->y + this->height - 1 - i);
			Drawing::drawLine(screenDest, this->borderColBottomRight, this->x + this->width - 1 - i, this->y + 1 + i, this->x + this->width - 1 - i, this->y + this->height - 1 - 2 * i);
		}	
	}
	
	// Draw out the text
	if (this->font)
	{
		Rect size(screenRect.x, screenRect.y, screenRect.x + screenRect.w, screenRect.y + screenRect.h);
		this->font->drawText(screenDest, size, Middle, Centred, name);
	}
	
}

bool Button::mouseDown(int x, int y, int button)
{
	// Check if in my area
	if (!this->isPointOverMe(x,y))
		return false;
	
	// Fire a click event if one exists
	this->fireClickEvent();
	
	this->moveBy( MOUSE_DOWN_MOVE, MOUSE_DOWN_MOVE );
	this->isMouseDown = true;
	
	return true;
}

bool Button::mouseUp(int x, int y, int button)
{
	if (this->isMouseDown)
		this->moveBy( -MOUSE_DOWN_MOVE, - MOUSE_DOWN_MOVE);
	
	this->isMouseDown = false;
	
	return true;
}

bool Button::mouseMove(int x, int y)
{
	this->mouseOver = this->isPointOverMe(x, y);
	return this->mouseOver;
}

bool Button::keyPress( SDLKey key, SDLMod mod, Uint16 character )
{
	// Fire a key event if the key matches our access key
	if (this->accessKey == ANY_KEY || this->accessKey == key)
	{
		this->fireClickEvent();	
		return true;
	}
	return false;
}
