/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
 
/*
 *  This is the main engine / game which the gui talks too.
 *  It handles all of the events from the GUI like starting a new game playing a piece etc.
 */
 
#ifndef _C4GAME_H_
#define _C4GAME_H_

#include "Board.h"
#include "GamesRules.h"
#include "AIRandom.h"
#include "AIHeuristics.h"
#include "AIMinimax.h"
#include "iai.h"

// Define the states which can be used in the board
#define EMPTY		0
#define PLAYER1		1
#define PLAYER2		2

// The return state of playing a counter
enum PlayStatus 
{
	OK,				// Counter was played
	ColumnFull,		// Counter was attempted to be played in a full column
	BoardFull,		// Board is full no more moves, i.e. game drawn
	PlayerWon		// Counter won the gmae
};

// Types of players that can play the game
enum PlayerType
{
	AI,
	Human
};

// Difficulty level for the AI computer player
enum AILevel
{
	Easy,			// Random AI
	Normal,			// Heuristics AI
	Hard			// Minimax AI
};

// Define a move which has taken place for the undo list
struct Move
{
	int column;		// The column the counter was played into
	int player;		// The player which played the counter
};

class C4Game
{
private:

	// The rules for calculating connect 4
	GamesRules* rules;
	
	// Keep track of the logging
	Severity loggingLevel;
	
	// The ai player
	IAI* aiPlayer;
	
	// Level of difficulty for the ai
	AILevel aiLevel;

	// Find all of the counters which produced the connect 4 and store in winningCellsoverlay
	void findWinningCells(int player, int col, int row);
	
	// Undo list
	List<Move> undoMovesList;
	
public:

	// Boards
	Board board;					// Main playing board
	Board winningCellsoverlay;		// A board which highlights which counters were used to win the game
	
	// Game Status
	WinningStatus status;
		
	// Game stats
	int numGamesPlayerWon;
	int numGamesPlayerLost;
	int numGamesDrawn;

	// Defines whos go it is
	int playersTurn;
	int playerWentFirst;
	int playerHint;
	PlayerType playerTypes[2];
	
	// Constructor
	C4Game();
	virtual ~C4Game();
	
	// Start a new game
	void startNewGame();
	
	// Get the current Player type
	PlayerType getCurrentPlayerType();
	
	// Play a counter
	PlayStatus playCounter(int column);
	
	// Play AI counter
	PlayStatus playAICounter();
	
	// Get / Set the AI level
	AILevel getAILevel();
	void setAILevel(AILevel level);
	
	// Moves
	int getLastColumnPlayed();
	void unDoMove();
	void unDoMoves(int num);
	
	// Hint for the current player which move to make
	void hintMoveForCurrentPlayer();

	// Logging
	void toggleLogging();	

};

#endif //_C4GAME_H_
