/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef WINTOOLKITGENERAL_H
#define WINTOOLKITGENERAL_H

// System includes
#include "SDL.h"
#include "SDL_image.h"
#include "SDL_mixer.h"
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <fstream>

// Game includes
#include "../general/List.h"
#include "../general/Log.h"

 
 // Define NULL
#ifndef NULL
#define NULL		((void *)0)
#endif


// Macros
#define max(a,b)	(((a)>(b))?(a):(b))
#define min(a,b)	(((a)<(b))?(a):(b))
#define radians(a)	((a) * M_PI / 180)
#define degrees(a) 	((a) * 180 / M_PI)
#define round(a) 	(floor((a) + 0.5))
#define random(a) 	((a) * rand() / RAND_MAX)

// Roation a track section moves in.
// e.g. if an arc starts at 0 and ends at PIE then this defines if the arc is 
// at the top anti_clockwise or at the bottom clockwise
enum Rotation
{
	ROT_CLOCKWISE,
	ROT_ANTI_CLOCKWISE,
};

struct Point;

// General Methods
Uint32 getPixel(SDL_Surface *surface, int x, int y);		// Get a pixel colour from a surface
void putPixel(SDL_Surface *surface, int x, int y, Uint32 pixel);// Set a pixel colour on a surface
double quadrantise(double x, double y, bool screen);		// Calculate the angle between the two points
double quadrantise(Point start, Point end, bool screen);	
void mergeFilename(const char* path, const char* filename, char* fullFilename); // Combine a filename / path into one string

// General
bool fileExist(const char* filename);

// Basic Structures
//-----------------

// A single point (i.e. x, y position) with in 2D space
struct Point
{
	float x, y;

	inline Point()
	{
		this->x = 0;
		this->y = 0;
	};
	inline Point(float x, float y)
	{
		set(x, y);
	}

	inline void set(float x, float y)
	{
		this->x = x;
		this->y = y;
	}

	inline void set(char* value)
	{
		// set a point from a string of the format "400,400"
		char* ptemp= NULL; 
		ptemp = strchr(value,',');
		if (ptemp==NULL)
			return;
		*ptemp=0;
		this->x = atof(value);
		ptemp++;
		this->y = atof(ptemp);
	}

	inline Point& operator-= (Point& point)
	{
		this->x = this->x - point.x;
		this->y = this->y - point.y;
		return *this;
	}

	inline double distanceFrom(Point point)
	{
		float xDiff = this->x - point.x;
		float yDiff = this->y - point.y;
		float distSqrd = (yDiff * yDiff) + (xDiff * xDiff);
		if (distSqrd > 0)
			return sqrt(distSqrd);
		else
			return 0;
	}

};

// Defines a region with in 2D space
struct Rect
{
	Point topLeft, bottomRight;

	inline Rect() {};
	
	inline Rect (Point topLeft, Point bottomRight)
	{
		this->topLeft = topLeft;
		this->bottomRight = bottomRight;
	}

	inline Rect (float xTop, float yTop, float xBottom, float yBottom)
	{
		this->topLeft.set(xTop, yTop);
		this->bottomRight.set(xBottom, yBottom);
	}

	inline float width()
	{
		if (this->bottomRight.x > this->topLeft.x)
			return this->bottomRight.x - this->topLeft.x;
		else
			return this->topLeft.x - this->bottomRight.x;
	}

	inline float height()
	{
		if (this->topLeft.y > this->bottomRight.y)
			return this->topLeft.y - this->bottomRight.y;	
		else
			return this->bottomRight.y - this->topLeft.y;	
	}
	
	inline void move(int x, int y)
	{
		this->topLeft.x += x;
		this->topLeft.y += y;
		this->bottomRight.x += x;
		this->bottomRight.y += y;	
	}
};

// Define a vector for velocity defined in both carteasion(?) and angular formats
class Velocity
{
private:
	double xComponent;
	double yComponent;
	double speed, angle;

	inline void setComponents()
	{
		this->xComponent = speed * cos(angle);
		this->yComponent = speed * sin(angle);
	}
	
	inline void setAttributes()
	{
		this->angle = quadrantise(xComponent, yComponent, false);
		this->speed = sqrt(xComponent * xComponent + yComponent * yComponent);
	}

public:

	Velocity()
	{
		this->speed = this->angle = this->xComponent = this->yComponent = 0.0;	
	}
	
	Velocity(double speed, double angle)
	{
		this->speed = speed;
		this->angle = angle;
		Velocity::setComponents();
	}
	
	inline double getSpeed() { return this->speed; }
	inline void setSpeed(double speed)
	{ 
		this->speed = speed; 
		Velocity::setComponents();
	}
	
	inline double getAngle() { return this->angle; }
	inline void setAngle(double angle) 
	{ 
		this->angle = angle; 
		Velocity::setComponents();
	}
	
	inline double getXComp() { return this->xComponent; }
	inline void setXComp(double x)
	{
		this->xComponent = x;
		Velocity::setAttributes();
	}
	inline double getYComp() { return this->yComponent; }
	inline void setYComp(double y)
	{
		this->yComponent = y;
		Velocity::setAttributes();
	}
};

#endif // WINTOOLKITGENERAL_H

