/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "TabSheet.h"

TabSheet::TabSheet(int x, int y, int width, int height) : Control (x, y, width, height)
{
	for (int i = 0; i < MAX_NUM_OF_TABS; i++)
	{
		this->tabNames[i]	= NULL;
		this->tabs[i]		= NULL;
	}
	
	this->numOfTabs = 0;
	
	this->activeTab = NULL;
	this->activeTabIndex = -1;
	
	this->font = NULL;
	this->tabNameWidth = 0;
	this->tabNameHeight = 0;
}

TabSheet::~TabSheet()
{
	// Clean up all of the tab sheets
	for (int i = 0; i < this->numOfTabs; i++)
	{
		delete[] this->tabNames[i];
		delete this->tabs[i];
		this->tabNames[i]	= NULL;
		this->tabs[i]		= NULL;
	}
	
	this->activeTab = NULL;
	this->activeTabIndex = -1;
	this->font = NULL;
}


Window* TabSheet::createTab(const char* tabName)
{
	// Make sure we have enough space to create the new tab
	if (this->numOfTabs >= MAX_NUM_OF_TABS)
		return NULL;
	
	// Create the new tab
	this->tabs[this->numOfTabs] = new Window(SDL_GetVideoSurface(), this->x + TAB_WINDOW_BORDER, this->y + TAB_WINDOW_BORDER + this->tabNameHeight, this->width - TAB_WINDOW_BORDER * 2, this->height - TAB_WINDOW_BORDER * 2 - this->tabNameHeight);
	this->tabNames[this->numOfTabs] = strdup(tabName);
	
	// Check the width of the tab name
	if (this->font)
	{
		int tabNameWidth = this->font->textWidth(tabName) + TAB_NAME_HORIZ_BORDER * 2;
		if (tabNameWidth > this->tabNameWidth)
			this->tabNameWidth = tabNameWidth;
	}
	
	// Make sure we have always got an active tab
	if (!this->activeTab)
	{
		this->activeTab	= this->tabs[this->numOfTabs];
		this->activeTabIndex = this->numOfTabs;
	}
	
	// Keep track of the number of tabs
	this->numOfTabs++;
	
	return this->tabs[this->numOfTabs - 1];
}

void TabSheet::setActiveTabIndex(int index)
{
	// Make sure the index is valid
	if (index >= this->numOfTabs)
		return;
		
	// Set the active tab
	this->activeTab = this->tabs[index];
	this->activeTabIndex = index;
}

Window* TabSheet::getTab(int index)
{
	if (index >= this->numOfTabs)
		return NULL;
		
	return this->tabs[index];
}

Window* TabSheet::getActiveTab()
{
	return this->activeTab;
}

int TabSheet::getActiveTabIndex()
{
	return this->activeTabIndex;
}

int TabSheet::getNumofTabs()
{
	return this->numOfTabs;
}

int TabSheet::getTabName(int index, char* name, int bufferSize)
{
	return 0;
}

void TabSheet::setFont(Font* font)
{
	this->font = font;
	this->tabNameHeight = this->font->height() + TAB_NAME_VERTICAL_BORDER * 2;
}

void TabSheet::setFont(char* fontFileName)
{
	this->font = FontManager::Instance()->getFont(fontFileName);
	if (this->font)
		this->tabNameHeight = this->font->height() + TAB_NAME_VERTICAL_BORDER * 2;
}

int TabSheet::getTabNameHeight()
{
	return this->tabNameHeight;
}

int TabSheet::getTabNameWidth()
{
	return this->tabNameWidth;	
}

void TabSheet::animate()
{
     // Animate the current Window
	if (this->activeTab)
	{
		this->activeTab->animate();	
	}
}

void TabSheet::draw(SDL_Surface* screenDest)
{
	// Draw the tabs at the top of the screen
	int left = this->x;
	int topY = this->y + this->tabNameHeight;
	
	// Background colour rect
	SDL_Rect tabNameRect;
	tabNameRect.x = this->x;
	tabNameRect.y = this->y;
	tabNameRect.w = this->tabNameWidth;
	tabNameRect.h = this->tabNameHeight;
	
	for (int i = 0; i < this->numOfTabs; i++)
	{
		tabNameRect.x = left;
		// Draw the background colour depending on whether the active tab
		if (i == this->activeTabIndex)
			SDL_FillRect (screenDest, &tabNameRect, this->activeColour);
		else
			SDL_FillRect (screenDest, &tabNameRect, this->inActiveColour);
			
		// Draw the borders
		// Top border
		Drawing::drawLine(screenDest, this->borderColTopLeft, left, this->y, left + this->tabNameWidth, this->y);
		// Right border
		Drawing::drawLine(screenDest, this->borderColBottomRight, left + this->tabNameWidth, this->y, left + this->tabNameWidth, topY);
		
		// if not active draw the line underneath so looks like not selected
		if (i != this->activeTabIndex)
			Drawing::drawLine(screenDest, this->borderColTopLeft, left, topY, left + this->tabNameWidth, topY);
	
		// Draw the text for the tab name
		if (this->font)
		{
			Rect size(left, this->y, left + this->tabNameWidth, topY);
			this->font->drawText(screenDest, size, Middle, Centred, "%s", this->tabNames[i]);
		}
		
		// Update the left position
		left += this->tabNameWidth + 1;
	}
	
	// Draw the border around the tabsheet
	int bottomRightX = this->x + this->width;
	int bottomRightY = this->y + this->height;
	
	// Top border - fill in the remainder of what the tabs haven't drawn over
	Drawing::drawLine(screenDest, this->borderColTopLeft, left, topY, this->x + this->width, topY);
	// Left border
	Drawing::drawLine(screenDest, this->borderColTopLeft, this->x, this->y , this->x, this->y + this->height);
	// Bottom border
	Drawing::drawLine(screenDest, this->borderColBottomRight, this->x, this->y + this->height, bottomRightX, bottomRightY);
	// Right border
	Drawing::drawLine(screenDest, this->borderColBottomRight, this->x + this->width, topY, bottomRightX, bottomRightY);
	
	// Draw the current Window
	if (this->activeTab)
	{
		this->activeTab->draw(screenDest);	
	}
}

bool TabSheet::mouseDown(int xMouse, int yMouse, int button)
{	
	// Check if in my area
	if (!this->isPointOverMe(xMouse, yMouse))
	{
		this->active = false;
		return false;
	}
	
	// Must have clicked us
	this->active = true;
	
	Log::Instance()->log("TabSheet clicked");
	
	// Check if the tabs were clicked if so then change the active tab
	// Check y component is in the valid range
	if (yMouse < (int)this->y + this->tabNameHeight)
	{
		Log::Instance()->log("Tab Buttons clicked");
			
		int tabIndex = (xMouse - this->x) / this->tabNameWidth;
		
		// Check valid index
		if (tabIndex >= this->numOfTabs)
			return true;
		
		this->activeTab = this->tabs[tabIndex];
		this->activeTabIndex = tabIndex;	
		
		return true;
	}
	
	// window must have been clicked, pass control to window
	xMouse -= this->x + TAB_WINDOW_BORDER;
	yMouse -= this->y + TAB_WINDOW_BORDER;
	
	if (this->activeTab)
	{
		Log::Instance()->log("Active Tab Must have been clicked");
		this->activeTab->mouseDown(xMouse, yMouse, button);	
	}
	
	return true;
}

bool TabSheet::mouseUp(int xMouse, int yMouse, int button)
{
	// Do a primary check that the mouse was clicked on the Window first before going 
	// through all of the controls
	if 	(!this->isPointOverMe(xMouse, yMouse))
	{
		return true;
	}
	
	// If the window is hiden then don't bother to process the click event
	if (!this->visible)
		return false;

	// The control x,y are relative to the Window so the mouse click x,y should 
	// be re-aligned to match the Window.
	// window must have been clicked, pass control to window
	xMouse -= this->x + TAB_WINDOW_BORDER;
	yMouse -= this->y + TAB_WINDOW_BORDER;
	
	if (this->activeTab)
	{
		this->activeTab->mouseDown(xMouse, yMouse, button);	
	}
	
	return true;
}

bool TabSheet::mouseMove(int xMouse, int yMouse)
{
	// If the window is hiden then don't bother to process the click event
	if (!this->visible || !this->active)
		return false;

	// The control x,y are relative to the Window so the mouse click x,y should 
	// be re-aligned to match the Window.
	xMouse -= this->x + TAB_WINDOW_BORDER;
	yMouse -= this->y + TAB_WINDOW_BORDER;

	if (this->activeTab)
	{
		this->activeTab->mouseMove(xMouse, yMouse);	
	}
	
	return true;
}

bool TabSheet::keyPress( SDLKey key, SDLMod mod, Uint16 character )
{
	// If the window is hiden then don't bother to process the click event
	if (!this->visible || !this->active)
		return false;
	
	// Pass the key press to the tab
	if (this->active && this->activeTab)
	{
		return this->activeTab->keyPress(key, mod, character);
	}
	return false;
}
