/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "GamesRules.h"

GamesRules::GamesRules()
{
	
}

GamesRules::~GamesRules()
{
	
}

WinningStatus GamesRules::hasPlayerWon(Board& board, int column, int row)
{
	// Get the player at that cell, if none then don't bother checking
	int player = board.board[column][row];
	if (!player)
		return None;
		
	return GamesRules::hasPlayerWon(board, player, column, row);
}


// Check win for a partular cell
WinningStatus GamesRules::hasPlayerWon(Board& board, int player, int column, int row)
{
	// Check from the current cell in all directions if a win is possible
	if (numOfHoriCounters(board, player, column, row, false) >= 4 ||
		numOfVertCounters(board, player, column, row, false) >= 4 ||
		numOfDiagTRBLCounters(board, player, column, row, false) >= 4 || 
		numOfDiagTLBRCounters(board, player, column, row, false) >= 4)
	{
		// Someone has won return the correct status depending on which player
		if (player == 1)
			return Player1;
		else
			return Player2;	
	}
	
	return None;
}

// Winning helper functions
bool GamesRules::isCellSame(Board& board, int player, int column, int row, bool ignoreEmptyCells)
{
	return (board.board[column][row] == player || (ignoreEmptyCells && board.board[column][row] == 0));	
}

int GamesRules::numOfHoriCounters(Board& board, int player, int column, int row, bool ignoreEmptyCells)
{
	int countersInRow = 1;
	int tempCol = column + 1;
	
	// Check all of the cells to the right
	while (tempCol < BOARD_COLS && isCellSame(board, player, tempCol, row, ignoreEmptyCells))
	{
		tempCol++;
		countersInRow++;	
	}
	
	// Check all of the cells to the left
	tempCol = column - 1;
	while (tempCol >= 0 && isCellSame(board, player, tempCol, row, ignoreEmptyCells))
	{
		tempCol--;
		countersInRow++;	
	}
	
	return countersInRow;	
}

int GamesRules::numOfVertCounters(Board& board, int player, int column, int row, bool ignoreEmptyCells)
{
	// Might not be the last counter inserted so check both above and below
	// Check downwards
	int countersInRow = 1;
	int tempRow = row - 1;
	while (tempRow >= 0 && isCellSame(board, player, column, tempRow, ignoreEmptyCells))
	{
		tempRow--;
		countersInRow++;	
	}
	
	// Check upwards
	tempRow = row + 1;
	while (tempRow < BOARD_ROWS && isCellSame(board, player, column, tempRow, ignoreEmptyCells))
	{
		tempRow++;
		countersInRow++;	
	}
	return countersInRow;	
}

int GamesRules::numOfDiagTRBLCounters(Board& board, int player, int column, int row, bool ignoreEmptyCells)
{
	// top right -> bottom left
	int countersInRow = 1;
	int tempRow = 0;
	int tempCol = 0;
		
	// top right = col +ve, row +ve
	tempRow = row + 1;
	tempCol = column + 1;
	while (tempRow < BOARD_ROWS && tempCol < BOARD_COLS && isCellSame(board, player, tempCol, tempRow, ignoreEmptyCells))
	{
		tempRow++;
		tempCol++;
		countersInRow++;	
	}
	
	// bottom left = col -ve, row -ve
	tempRow = row - 1;
	tempCol = column - 1;
	while (tempRow >= 0 && tempCol >= 0 && isCellSame(board, player, tempCol, tempRow, ignoreEmptyCells))
	{
		tempRow--;
		tempCol--;
		countersInRow++;	
	}
	
	return countersInRow;
}

int GamesRules::numOfDiagTLBRCounters(Board& board, int player, int column, int row, bool ignoreEmptyCells)
{
	// top left -> bottom right
	int countersInRow = 1;
	int tempRow = 0;
	int tempCol = 0;
	
	// top left = col -ve, row +ve
	tempRow = row + 1;
	tempCol = column - 1;
	while (tempRow < BOARD_ROWS && tempCol >= 0 && isCellSame(board, player, tempCol, tempRow, ignoreEmptyCells))
	{
		tempRow++;
		tempCol--;
		countersInRow++;	
	}
	
	// bottom right = col +ve, row -ve
	tempRow = row - 1;
	tempCol = column + 1;
	while (tempRow >= 0 && tempCol < BOARD_COLS && isCellSame(board, player, tempCol, tempRow, ignoreEmptyCells))
	{
		tempRow--;
		tempCol++;
		countersInRow++;	
	}
	
	return countersInRow;	
}

// Find the number of counter / spaces in a row. i.e. the maximum number possible
int GamesRules::numOfPossibleHoriCounters(Board& board, int player, int column, int row)
{
	return GamesRules::numOfHoriCounters(board, player, column, row, true);		
}

int GamesRules::numOfPossibleVertCounters(Board& board, int player, int column, int row)
{
	return GamesRules::numOfVertCounters(board, player, column, row, true);
}

int GamesRules::numOfPossibleDiagTRBLCounters(Board& board, int player, int column, int row)
{
	return GamesRules::numOfDiagTRBLCounters(board, player, column, row, true);
}

int GamesRules::numOfPossibleDiagTLBRCounters(Board& board, int player, int column, int row)
{
	return GamesRules::numOfDiagTLBRCounters(board, player, column, row, true);
}


