/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "C4Game.h"

C4Game::C4Game()
{
	// Reset the game stats
	this->numGamesPlayerWon = 0;
	this->numGamesPlayerLost = 0;
	this->numGamesDrawn = 0;
	this->playerWentFirst = 1;
	this->rules = new GamesRules();
	C4Game::setAILevel(Hard);
	C4Game::startNewGame();
}

C4Game::~C4Game()
{
	if (this->rules)
	{
		delete this->rules;
		this->rules = NULL;	
	}
	
	if (this->aiPlayer)
	{
		delete this->aiPlayer;
		this->aiPlayer = NULL;	
	}
}

// Start a new game
void C4Game::startNewGame()
{
	// Reset the player types to both human
	this->playerTypes[0] = Human;
	this->playerTypes[1] = AI;
	this->board.reset();
	this->winningCellsoverlay.reset();
	this->playersTurn = this->playerWentFirst;
	this->undoMovesList.RemoveAll();
	this->status = None;
	this->playerHint = -1;
	
	// Toggle which player is to play first for the next game
	this->playerWentFirst = (this->playerWentFirst == 2) ? 1 : 2;
}

// Get the current Player type
PlayerType C4Game::getCurrentPlayerType()
{
	return this->playerTypes[this->playersTurn - 1];	
}

// Play a counter
PlayStatus C4Game::playCounter(int column)
{
	// Check if the game has already been won
	if (this->status != None)
		return PlayerWon;
		
	// Check to make sure we can play the counter
	if (this->board.isColumnFull(column))
		return ColumnFull;
		
	// Play the counter
	int row = this->board.nextSquareFree(column);
	this->board.board[column][row] = this->playersTurn;
	
	// Store the last move in the undo list
	Move lastMove;
	lastMove.player = this->playersTurn;
	lastMove.column = column;
	this->undoMovesList.Append(lastMove);
	
	// Reset the player hint it isn't valid any more
	this->playerHint = -1;
	
	// Check if the game was won
	this->status = rules->hasPlayerWon(this->board, this->playersTurn, column, row);
	if (this->status != None)
	{
		// Update the winning board overlay i.e. the cells which produced connect
		C4Game::findWinningCells(this->playersTurn, column, row);
		
		// Update the states for player 1
		if (this->status == Player1)
			this->numGamesPlayerWon++;
		else
			this->numGamesPlayerLost++;
		return PlayerWon;
	}
	
	// No one has won so check for a full board / i.e. a draw
	if (this->board.isFull())
	{
		this->status = Draw;
		this->numGamesDrawn++;
		return BoardFull;
	}
		
	// Move onto the next player
	this->playersTurn = (this->playersTurn == 2) ? 1 : 2;
	
	return OK;	
}

// Play AI counter
PlayStatus C4Game::playAICounter()
{
	// Make sure that it is a the ai's turn
	if (C4Game::getCurrentPlayerType() != AI)
		return OK;
	
	// Also make sure the game is in the right state
	if (this->status != None)
		return PlayerWon;

	// Make sure we don't sit in a loop trying to play a draw
	if (this->board.isFull())
		return BoardFull;		
		
	// Get the piece to play from the AI
	int counter = this->aiPlayer->playCounter(this->board, this->playersTurn);
	
	// Play as normal	
	return C4Game::playCounter(counter);
}

// Get / Set the AI level
AILevel C4Game::getAILevel()
{
	return this->aiLevel;
}

void C4Game::setAILevel(AILevel level)
{
	// Ignore if the same
	if (level == this->aiLevel && this->aiPlayer != NULL)
		return;
		
	// Cleanup the previous ai opponent
	delete this->aiPlayer;
	this->aiPlayer = NULL;
	
	// Create the new ai player
	switch (level)
	{
		case Easy:
			this->aiPlayer = new AIRandom();
			break;
		case Normal:
			this->aiPlayer = new AIHeuristics();
			break;
		case Hard:
			this->aiPlayer = new AIMinimax();
			break;
	}
	
	// Update the level of ai
	this->aiLevel = level;
}

int C4Game::getLastColumnPlayed()
{
	// Look at the undo list for the last item played
	if (this->undoMovesList.Size() > 0)
	{
		return this->undoMovesList.m_tail->m_data.column;	
	}
	return -1;
}

void C4Game::unDoMove()
{
	C4Game::unDoMoves(1);
}

void C4Game::unDoMoves(int num)
{
	// Make sure we can only undo moves if the game hasn't ended
	if (this->status != None)
		return;
	
	while (num > 0 && this->undoMovesList.Size() > 0)
	{
		// undo the move
		// Find the cell to reset
		int column = this->undoMovesList.m_tail->m_data.column;
		int row = this->board.nextSquareFree(column) - 1;
		// make sure a valid move
		if (row < 0)
			return;
		
		// Undo the move
		this->board.board[column][row] = EMPTY;
		
		// Remove the move from the undo list
		this->undoMovesList.RemoveTail();
		
		// Change back to the last player
		this->playersTurn = (this->playersTurn == 2) ? 1 : 2;
		num--;	
	}
}

// Hint for the current player which move to make
void C4Game::hintMoveForCurrentPlayer()
{
	// Only hint for human players
	if (C4Game::getCurrentPlayerType() != Human)
		return;
	
	// Also make sure the game is in the right state
	if (this->status != None)
		return;

	// Make sure we don't sit in a loop trying to play a draw
	if (this->board.isFull())
		return;
		
	// Get the column which the computer would play if it was it's turn
	this->playerHint = this->aiPlayer->playCounter(this->board, this->playersTurn);

}

// Logging
void C4Game::toggleLogging()
{
	// Toggle the logging between information and debugging
	if (this->loggingLevel == SV_INFORMATION)
	{
		this->loggingLevel = SV_DEBUG;
		Log::Instance()->log("Logging Enabled");
	}
	else
	{
		this->loggingLevel = SV_INFORMATION;
		Log::Instance()->log("Logging Disabled");
	}
	
	// Set the logging for the ai
	this->aiPlayer->loggingLevel = this->loggingLevel;
}

// Find all of the counters which produced the connect 4 and store in winningCellsoverlay
void C4Game::findWinningCells(int player, int col, int row)
{
	int colTemp = 0;
	int rowTemp = 0;
	
	this->winningCellsoverlay.reset();
	
	// Check each dimension which has connect 4 and then search along those lines for the counters
	
	// Check horizontal
	if (rules->numOfHoriCounters(this->board, player, col, row, false) >= 4)
	{
		// Check to the right
		for (colTemp = col; colTemp < BOARD_COLS && this->board.board[colTemp][row] == player; colTemp++)
			this->winningCellsoverlay.board[colTemp][row] = 1;
		
		// Check to the left
		for (colTemp = col; colTemp >= 0 && this->board.board[colTemp][row] == player; colTemp--)
			this->winningCellsoverlay.board[colTemp][row] = 1;
	}
	
	// Check for vertical
	if (rules->numOfVertCounters(this->board, player, col, row, false) >= 4)
	{
		// Check downwards
		for (rowTemp = row; rowTemp >= 0 && this->board.board[col][rowTemp] == player; rowTemp--)
			this->winningCellsoverlay.board[col][rowTemp] = 1;
	}
	
	// Check top left to bottom right diagonal
	if (rules->numOfDiagTLBRCounters(this->board, player, col, row, false) >= 4)
	{
		// Check to the top left
		for (colTemp = col, rowTemp = row; 
			colTemp >= 0 && rowTemp < BOARD_ROWS && this->board.board[colTemp][rowTemp] == player; 
			colTemp--, rowTemp++)
			this->winningCellsoverlay.board[colTemp][rowTemp] = 1;
		
		// Check to the bottom right
		for (colTemp = col, rowTemp = row; 
			colTemp < BOARD_COLS && rowTemp >= 0 && this->board.board[colTemp][rowTemp] == player; 
			colTemp++, rowTemp--)
			this->winningCellsoverlay.board[colTemp][rowTemp] = 1;
	}
	
	// Check top right to bottom left diagonal
	if (rules->numOfDiagTRBLCounters(this->board, player, col, row, false) >= 4)
	{
		// Check to the top right
		for (colTemp = col, rowTemp = row; 
			colTemp < BOARD_COLS && rowTemp < BOARD_ROWS && this->board.board[colTemp][rowTemp] == player; 
			colTemp++, rowTemp++)
			this->winningCellsoverlay.board[colTemp][rowTemp] = 1;
		
		// Check to the bottom left
		for (colTemp = col, rowTemp = row; 
			colTemp >= 0 && rowTemp >= 0 && this->board.board[colTemp][rowTemp] == player;
			 colTemp--, rowTemp--)
			this->winningCellsoverlay.board[colTemp][rowTemp] = 1;
	}
}



