/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "AIMinimax.h"

AIMinimax::AIMinimax() : AIHeuristics()
{
	this->ply = 5;		// Set the search depth
	this->rules = new GamesRules();
	this->loggingLevel = SV_INFORMATION;
}

AIMinimax::~AIMinimax()
{
	if (this->rules)
	{
		delete this->rules;	
		this->rules = NULL;
	}
}


int AIMinimax::playCounter(Board board, int player)
{
	// Use the influence map for the highest move
	AIHeuristics::resetInfluenceMap();
	
	// Figure out who our opponent is
	int opponent = (player == 2) ? 1 : 2;
	
	// Search for a column which will make a row / connect 4
	for (int col = 0; col < BOARD_COLS; col++)
	{
		// Make sure a leagal move
		if (board.isColumnFull(col))
			continue;
		
		// Play the counter for the column
		int row = board.nextSquareFree(col);
		board.board[col][row] = player;
		
		// The player to move has just moved so get the opponent to play next
		influenceMap[col] = -AIMinimax::makeMoveMinimax(&board, player, opponent, col, row, -INT_MAX, INT_MAX, ply);
		
		// Reset the counter in the board
		board.board[col][row] = 0;
	}
	
	// Search the influence map for the best column to play
	return AIHeuristics::getColumnToPlayFromInfluenceMap(influenceMap, &board);	
}

// Use this to recurse over the board evulating each move for both players
// Use alpha-beta pruning to reduce the search space.
int AIMinimax::makeMoveMinimax(Board* board, int playerMax, int currentPlayer, int col, int row, int alpha, int beta, int depth)
{
	int score = 0;
	int opponent = 0;
	
	// Check if someone has won
	if (this->rules->hasPlayerWon(*board, col, row) != None)
		// Change the value to return if someone has won based on the depth
		// Try to get the player to play for immediate wins over wins in two or more moves
		return (-INT_MAX / (this->ply + 1)) * (depth + 1);
	
	// Check if we have reached the end of the game or depth
	if (depth == 0)
		return AIMinimax::evaluateBoard(board, playerMax);
	
	// Figure out who our opponent is
	opponent = (currentPlayer == 2) ? 1 : 2;
	
	// Evaulate all possible moves
	for (int col = 0; col < BOARD_COLS; col++)
	{
		if (board->isColumnFull(col))
			continue;
			
		int row = board->nextSquareFree(col);
	
		// Set the cell as through we have played it
		board->board[col][row] = currentPlayer;
		
		// Find the minmax for the child moves
		score = -AIMinimax::makeMoveMinimax(board, playerMax, opponent, col, row, -beta, -alpha, depth - 1);
		
		// Reset the cell
		board->board[col][row] = 0;
		
		// We have found a worse move, ignore the following moves
		if (score >= beta)
			return beta;
		
		// We have found a better move keep track of it
		if (score > alpha)
        	alpha = score;
	}

	// Return the best move we have found
	return alpha;
}

int AIMinimax::evaluateBoard(Board* board, int player)
{
	// Calculate a score for the board for the current player
	// A high number means a good position for that player
	// A low score means a bad position
	
	// Base this on the number of 
	// : connect 4+
	// : 3 in a row with the potential to get connect 4
	// : 2 in a row with the potential to get connect 4
	// We don't care about 3 in a row which can't get connect 4
	
	// Parse the board vertically, horzontally, diagonally both ways
	// For both players
	
	// Figure out who our opponent is
	int opponent = (player == 2) ? 1 : 2;
	
	PlayerScores playerScores;
	PlayerScores opponentScores;
	
	AIMinimax::getPlayerScoresForBoard(board, player, &playerScores);
	AIMinimax::getPlayerScoresForBoard(board, opponent, &opponentScores);
	
	int playerScore = AIMinimax::calcScore(&playerScores, &opponentScores);
	int opponentScore = AIMinimax::calcScore(&opponentScores, &playerScores);
	
	return playerScore - opponentScore;
}

void AIMinimax::getPlayerScoresForBoard(Board* board, int player, PlayerScores* scores)
{
	// Find the score
	AIMinimax::getVerticalPlayerScore(board, player, scores);
	AIMinimax::getHorizontalPlayerScore(board, player, scores);
	AIMinimax::getDiagonalPlayerScore(board, player, scores);
}

void AIMinimax::getVerticalPlayerScore(Board* board, int player, PlayerScores* scores)
{
	// temp variables
	int col = 0;
	int row = 0;
	int found = 0;
	int foundempties = 0;
	
	// Parse the board vertically
	//---------------------------
	for (col = 0; col < BOARD_COLS; col++)
	{
		found = 0;
		foundempties = 0;
		for (row = 0; row < BOARD_ROWS; row++)
		{
			if (board->board[col][row] == player)
				found++;
			else if (board->board[col][row] == 0)
				foundempties++;
			else 
			{
				// reset the above found the opponent
				found = 0;
				foundempties = 0;	
			}
		}
		// Evaluate the column
		if (found >= 4)
			scores->connect4++;
		else if (found == 3 && foundempties >= 1)
			scores->threeInLine++;
		else if (found == 2 && foundempties >= 2)
			scores->twoInLine++;
		else if (found == 1 && foundempties >= 3)
			scores->singles++;
		
	}
}

void AIMinimax::getHorizontalPlayerScore(Board* board, int player, PlayerScores* scores)
{
	// temp variables
	int col = 0;
	int row = 0;
	int found = 0;
	int foundempties = 0;
	
	// Parse the board horizontally
	//-----------------------------
	for (row = 0; row < BOARD_ROWS; row++)
	{
		found = 0;
		foundempties = 0;
		for (col = 0; col < BOARD_COLS; col++)
		{
			if (board->board[col][row] == player)
				found++;
			else if (board->board[col][row] == 0)
				foundempties++;
			else 
			{
				// reset the above found the opponent
				found = 0;
				foundempties = 0;	
			}
		}
		// Evaluate the column
		// Found connect 4 don't bother with the rest of the evaluation
		if (found >= 4)
			scores->connect4++;
		else if (found == 3 && foundempties >= 1)
			scores->threeInLine++;
		else if (found == 2 && foundempties >= 2)
			scores->twoInLine++;
		else if (found == 1 && foundempties >= 3)
			scores->singles++;	
	}
}

void AIMinimax::getDiagonalPlayerScore(Board* board, int player, PlayerScores* scores)
{
	// temp variables
	int col = 0;
	int row = 0;
	int colStart = 0;
	int rowStart = 0;
	int found = 0;
	int foundempties = 0;
	
	// Parse the board diagonally top left to bottom right
	// We don't need to process the whole board, the top right and bottom left corners
	// can't get connect 4. So start on the 4th row up and then move up and across
	// to the 4th column across
	//-----------------------------
	colStart = 0;
	rowStart = 3;
	while (colStart + rowStart < 9)
	{
		found = 0;
		foundempties = 0;
		
		for (row = rowStart, col = colStart; row >= 0 && col < BOARD_COLS; row--, col++)
		{
			if (board->board[col][row] == player)
				found++;
			else if (board->board[col][row] == 0)
				foundempties++;
			else 
			{
				// reset the above found the opponent
				found = 0;
				foundempties = 0;	
			}
		}
		
		// Evaluate the column
		// Found connect 4 don't bother with the rest of the evaluation
		if (found >= 4)
			scores->connect4++;
		else if (found == 3 && foundempties >= 1)
			scores->threeInLine++;
		else if (found == 2 && foundempties >= 2)
			scores->twoInLine++;
		else if (found == 1 && foundempties >= 3)
			scores->singles++;	
		
		// Move onto the next line to process
		rowStart++;
		if (rowStart >= BOARD_ROWS)
		{
			rowStart = BOARD_ROWS - 1;
			colStart++;
		}
	}
	
	// Parse the board diagonally top right to bottom left
	// We don't need to process the whole board, the top left and bottom right corners
	// can't get connect 4. So start on the 2nd row up and then move down and across
	// to the 4th column across
	//-----------------------------
	colStart = 0;
	rowStart = 2;
	while (colStart + rowStart < 4)
	{
		found = 0;
		foundempties = 0;
		
		for (row = rowStart, col = colStart; row < BOARD_ROWS && col < BOARD_COLS; row++, col++)
		{
			if (board->board[col][row] == player)
				found++;
			else if (board->board[col][row] == 0)
				foundempties++;
			else 
			{
				// reset the above found the opponent
				found = 0;
				foundempties = 0;	
			}
		}
		
		// Evaluate the column
		// Found connect 4 don't bother with the rest of the evaluation
		if (found >= 4)
			scores->connect4++;
		else if (found == 3 && foundempties >= 1)
			scores->threeInLine++;
		else if (found == 2 && foundempties >= 2)
			scores->twoInLine++;
		else if (found == 1 && foundempties >= 3)
			scores->singles++;	
		
		// Move onto the next line to process
		rowStart--;
		if (rowStart < 0)
		{
			rowStart = 0;
			colStart++;
		}
	}
}

int AIMinimax::calcScore(PlayerScores* maxScore, PlayerScores* minScore)
{
	// Calculate a score for th player based on the number of potential lines
	// Player has won
	if (maxScore->connect4 > 0)
		return INT_MAX;
	
	// If the opponent has connect 4 then we shouldn't have a very high score
	if (minScore->connect4 > 0)
		return 0;	
	
	// calculate the position otherwise
	int total = 0;
	total += maxScore->threeInLine * 20;
	total += maxScore->twoInLine * 5;
	total += maxScore->singles;
	
	return total;
}
