/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "AIHeuristics.h"

AIHeuristics::AIHeuristics()
{
	attackBias = 10;
	defendBias = 10;
	
	// Reset the map
	AIHeuristics::resetInfluenceMap();
}

AIHeuristics::~AIHeuristics()
{
	
}

int AIHeuristics::playCounter(Board board, int player)
{
	// Reset the map
	AIHeuristics::resetInfluenceMap();
		
	// Find best move for me
	AIHeuristics::calcBestMoveForPlayer(board, player, attackBias);
	
	// Find best move for opponent
	int opponent = (player == 2) ? 1 : 2;
	AIHeuristics::calcBestMoveForPlayer(board, opponent, defendBias);
	
	// Search the next move that the opponent might make, 
	// Make sure we don't give them any easy connect 4s
	AIHeuristics::calcBestNextMove(board, player, opponent, -defendBias / 2);
	
	// Search the influence map for the col to play
	return AIHeuristics::getColumnToPlayFromInfluenceMap(influenceMap, &board);

}

void AIHeuristics::calcBestMoveForPlayer(Board board, int player, int bias)
{
	// Search for a column which will make a row / connect 4
	for (int col = 0; col < BOARD_COLS; col++)
	{
		if (board.isColumnFull(col))
			continue;
		int row = board.nextSquareFree(col);
		influenceMap[col] += AIHeuristics::getCellInfluence(board, player, col, row, bias);
	}
}

void AIHeuristics::calcBestNextMove(Board board, int player, int opponent, int bias)
{
	for (int col = 0; col < BOARD_COLS; col++)
	{
		if (board.isColumnFull(col))
			continue;
		int row = board.nextSquareFree(col);
	
		// Check to make sure we have at least two cells free to check	
		if (row >= BOARD_ROWS - 1)
			continue;
		
		// Set the cell as through we have played it
		board.board[col][row] = player;
		row++;
		influenceMap[col] += AIHeuristics::getCellInfluence(board, opponent, col, row, bias);
		
		// Reset the cell
		board.board[col][row] = 0;
	}
}

int AIHeuristics::getCellInfluence(Board board, int player, int col, int row, int bias)
{
	int counterTotal = 0;
	counterTotal += AIHeuristics::calcInfluence(rules.numOfHoriCounters(board, player, col, row, false), rules.numOfPossibleHoriCounters(board, player, col, row), bias);
	counterTotal += AIHeuristics::calcInfluence(rules.numOfVertCounters(board, player, col, row, false), rules.numOfPossibleVertCounters(board, player, col, row), bias);
	counterTotal += AIHeuristics::calcInfluence(rules.numOfDiagTLBRCounters(board, player, col, row, false), rules.numOfPossibleDiagTLBRCounters(board, player, col, row), bias);
	counterTotal += AIHeuristics::calcInfluence(rules.numOfDiagTRBLCounters(board, player, col, row, false), rules.numOfPossibleDiagTRBLCounters(board, player, col, row), bias);
	
	return counterTotal;	
}

int AIHeuristics::calcInfluence(int numInLine,  int numPossibleInLine, int bias)
{
	// Influence Rules
	// 1)	If possible of getting connect 4 now then produce a very big number 
	//		must take presidence over everything
	//
	// 2)	If more than one is a line and there are enough free cells / counters 
	//		in the line to get connect 4
	//
	// 3)	Lowest order
	
	// 1)
	if (numInLine >= 4)
		return numInLine * 1000 * bias;
	// 2)
	else if (numInLine > 1 && numPossibleInLine >= 4)
		return numInLine * numInLine * numPossibleInLine * numPossibleInLine * bias;
	// 3)
	else
		return numInLine * numPossibleInLine * bias;
}

void AIHeuristics::resetInfluenceMap()
{
	// Reset the map
	for (int col = 0; col < BOARD_COLS; col++)
		influenceMap[col] = 0;
}

int AIHeuristics::getColumnToPlayFromInfluenceMap(int influMap[], Board* initBoard)
{
	// Search the influence map for the col to play
	int maxInfluence = INT_MIN;
	int colMaxInfluence[BOARD_COLS];
	int numOfMaxInfluence = 0;
	int colToPlay = 0;
	for (int col = 0; col < BOARD_COLS; col++)
	{
		// Check the if the greatess influence and also if we can go in that column
		if (influMap[col] > maxInfluence && initBoard->nextSquareFree(col) < BOARD_ROWS)
		{
			maxInfluence = influMap[col];
			colMaxInfluence[0] = col;
			numOfMaxInfluence = 1;
		}
		else if (influMap[col] == maxInfluence)
		{
			// Found item with the same influence keep track of the column
			colMaxInfluence[numOfMaxInfluence] = col;
			numOfMaxInfluence++;
		}
	}
	
	// We have now searched the influence map if we have a winniner then use that
	// otherwise select a joining max at random
	if (numOfMaxInfluence == 1)
	{
		colToPlay = colMaxInfluence[0];
	}
	else
	{
		colToPlay = colMaxInfluence[rand() % (numOfMaxInfluence - 1)];
	}
	
	return colToPlay;
}

