/*
 *  Dwarf City
 *  Copyright (C) 2005  
 *  					Adam Child (adam@dwarfcity.co.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "Game.h"

Game::Game(SDL_Surface* surface) : Window (surface, 0, 0, APP_WIDTH, APP_HEIGHT)
{
	// Create a new game
	this->connect4Game = new C4Game();
	
	this->winningFlashTime = 0;
	this->winningFlashShow = true;
	
	// Define the main font to use
	mainFont = FontManager::Instance()->getFont(MAIN_FONT);
	
	// Define the area for the board
	this->counterGraphics[EMPTY] = IMG_Load(GFX_PATH "gfx/emptycell.png");
	if (!this->counterGraphics[EMPTY])
		Log::Instance()->die(1, SV_ERROR, "Trouble loading image %s", GFX_PATH "gfx/emptycell.png");
		
	this->counterGraphics[PLAYER1] = IMG_Load(GFX_PATH "gfx/player1cell.png");
	if (!this->counterGraphics[PLAYER1])
		Log::Instance()->die(1, SV_ERROR, "Trouble loading image %s", GFX_PATH "gfx/player1cell.png");
		
	this->counterGraphics[PLAYER2] = IMG_Load(GFX_PATH "gfx/player2cell.png");
	if (!this->counterGraphics[PLAYER2])
		Log::Instance()->die(1, SV_ERROR, "Trouble loading image %s", GFX_PATH "gfx/player2cell.png");
	
	this->squareSize = this->counterGraphics[EMPTY]->w;
	this->halfSquareSize = squareSize  / 2;
	
	// Define the area of the board
	this->boardRect.x = BOARD_BORDER_SIZE;
	this->boardRect.y = BOARD_BORDER_SIZE;
	this->boardRect.h = squareSize * BOARD_ROWS;
	this->boardRect.w = squareSize * BOARD_COLS;
	
	// Define the colours for the screen
	this->lastMoveColour = SDL_MapRGB(surface->format, 0x45, 0x45, 0x45);
	this->hintMoveColour = SDL_MapRGB(surface->format, 0xAA, 0x45, 0x45);
	this->buttonColour = SDL_MapRGB(surface->format, 0x73, 0x8A, 0xA5);
	this->buttonMouseOverColour = SDL_MapRGB(surface->format, 0x88, 0xAA, 0x88);
	Uint32 backgroudColour = SDL_MapRGB(surface->format, 0xF7, 0xD7, 0x8C);
	
	// Define the background colour	
	Window::setBackgroundColour(backgroudColour);
	Window::setModal(false);
	
	// Create a button to start a new game
	int border = 10;
	Rect newGameButtonRect(APP_WIDTH - 100 - border, border, APP_WIDTH - border, mainFont->height() + border + 4);
	Callback0<Game> menuButtonCallBack(*this, &Game::newGame);
	Button* newGameButton = new Button("New Game", newGameButtonRect);
	newGameButton->addClickEvent(menuButtonCallBack);
	newGameButton->setActiveColour(this->buttonColour);
	newGameButton->setMouseOverColour(this->buttonMouseOverColour);
	newGameButton->setFont(mainFont);
	Window::addControl(newGameButton);
	
	newGameButtonRect.move(0, 30);
	Callback0<Game> undoMoveButtonCallBack(*this, &Game::undoMove);
	Button* undoMoveButton = new Button("Undo Move", newGameButtonRect);
	undoMoveButton->addClickEvent(undoMoveButtonCallBack);
	undoMoveButton->setActiveColour(this->buttonColour);
	undoMoveButton->setMouseOverColour(this->buttonMouseOverColour);
	undoMoveButton->setFont(mainFont);
	Window::addControl(undoMoveButton);
	
	newGameButtonRect.move(0, 30);
	Callback0<Game> hintButtonCallBack(*this, &Game::getHint);
	Button* hintButton = new Button("Hint", newGameButtonRect);
	hintButton->addClickEvent(hintButtonCallBack);
	hintButton->setActiveColour(this->buttonColour);
	hintButton->setMouseOverColour(this->buttonMouseOverColour);
	hintButton->setFont(mainFont);
	Window::addControl(hintButton);
	
	// Add the message label
	Rect messageSize(border, APP_HEIGHT - mainFont->height() - border * 2, APP_WIDTH - border, APP_HEIGHT - border);
	this->message = new Label(messageSize);
	this->message->setFont(mainFont);
	Window::addControl(this->message);
	
	// Add the labels to display the scores
	int statsStartY = 110;
	Rect statsRect(APP_WIDTH - 160 - border, statsStartY, APP_WIDTH - border, statsStartY + mainFont->height());
	this->gamesWonLabel = new Label(statsRect);
	this->gamesWonLabel->setFont(mainFont);
	this->gamesWonLabel->setLabelText("Won: 0");
	Window::addControl(this->gamesWonLabel);
	
	statsRect.move(0, mainFont->height() + border);
	this->gamesLostLabel = new Label(statsRect);
	this->gamesLostLabel->setFont(mainFont);
	this->gamesLostLabel->setLabelText("Lost: 0");
	Window::addControl(this->gamesLostLabel);
	
	statsRect.move(0, mainFont->height() + border);
	this->gamesDrawn = new Label(statsRect);
	this->gamesDrawn->setFont(mainFont);
	this->gamesDrawn->setLabelText("Drawn: 0");
	Window::addControl(this->gamesDrawn);
	
	// Add the check boxes for the ai level you are playing against
	// User the check boxes as radio buttons just update manually
	statsRect.move(0, mainFont->height() + 3 * border);
	Label* aiEasyLabel = new Label(statsRect);
	aiEasyLabel->setFont(mainFont);
	aiEasyLabel->setLabelText("Easy AI:");
	Window::addControl(aiEasyLabel);

	Point checkBoxPos(APP_WIDTH - 50, statsRect.topLeft.y);
	Callback0<Game> aiEasyCallBack(*this,&Game::setAIEasy);
	this->aiEasyCheckBox = new CheckBox(checkBoxPos);
	this->aiEasyCheckBox->addChangeEvent(aiEasyCallBack);
	this->aiEasyCheckBox->setActiveColour(backgroudColour);
	this->aiEasyCheckBox->setInActiveColour(backgroudColour);
	this->aiEasyCheckBox->setBevelType(BEV_INNER);
	Window::addControl(this->aiEasyCheckBox);

	statsRect.move(0, mainFont->height() + border);
	Label* aiNormalLabel = new Label(statsRect);
	aiNormalLabel->setFont(mainFont);
	aiNormalLabel->setLabelText("Normal AI:");
	Window::addControl(aiNormalLabel);
	
	checkBoxPos.y = statsRect.topLeft.y;
	Callback0<Game> aiNormalCallBack(*this,&Game::setAINormal);
	this->aiNormalCheckBox = new CheckBox(checkBoxPos);
	this->aiNormalCheckBox->addChangeEvent(aiNormalCallBack);
	this->aiNormalCheckBox->setActiveColour(backgroudColour);
	this->aiNormalCheckBox->setInActiveColour(backgroudColour);
	this->aiNormalCheckBox->setBevelType(BEV_INNER);
	Window::addControl(this->aiNormalCheckBox);
	
	statsRect.move(0, mainFont->height() + border);
	Label* aiHardLabel= new Label(statsRect);
	aiHardLabel->setFont(mainFont);
	aiHardLabel->setLabelText("Hard AI:");
	Window::addControl(aiHardLabel);

	checkBoxPos.y = statsRect.topLeft.y;
	Callback0<Game> aiHardCallBack(*this,&Game::setAIHard);
	this->aiHardCheckBox = new CheckBox(checkBoxPos);
	this->aiHardCheckBox->addChangeEvent(aiHardCallBack);
	this->aiHardCheckBox->setActiveColour(backgroudColour);
	this->aiHardCheckBox->setInActiveColour(backgroudColour);
	this->aiHardCheckBox->setBevelType(BEV_INNER);
	Window::addControl(this->aiHardCheckBox);
	
	// Set the checkboxes with the level of ai which is being used
	AILevel level = this->connect4Game->getAILevel();
	Game::setAILevel(level);
		
	// Add the logo image
	// Place it in the middle of the board edge and the edge of the screen
	int logoSide = 135;
	int logoBoarder = (APP_WIDTH - this->boardRect.w - this->boardRect.x - logoSide) / 2;
	Rect logoSize(APP_WIDTH - logoBoarder - logoSide, APP_HEIGHT - logoBoarder - logoSide, APP_WIDTH - logoBoarder, APP_HEIGHT - logoBoarder);
	Image* logo = new Image(logoSize);
	logo->loadImage(GFX_PATH "gfx/logo.png");
	Window::addControl(logo);	

	// Initialise the update message
	Game::updateMessage(OK);
}

Game::~Game()
{
	delete this->connect4Game;
	this->connect4Game = NULL;
	
	if (this->counterGraphics[EMPTY] != NULL)
		SDL_FreeSurface(this->counterGraphics[EMPTY]);
	if (this->counterGraphics[PLAYER1] != NULL)
		SDL_FreeSurface(this->counterGraphics[PLAYER1]);
	if (this->counterGraphics[PLAYER2] != NULL)
		SDL_FreeSurface(this->counterGraphics[PLAYER2]);
				
}


void Game::draw(SDL_Surface* surface)
{
	// Draw the main window first
	Window::draw(surface);	
	
	// Draw the board
	SDL_Rect screenRect;
	screenRect.x = BOARD_BORDER_SIZE;
	screenRect.y = BOARD_BORDER_SIZE;
	screenRect.w = squareSize;
	screenRect.h = squareSize;
	
	int displayPlayer = 0;
	for (int row = BOARD_ROWS - 1; row >= 0; row--)
	{
		for (int col = 0; col < BOARD_COLS; col++)
		{
			// Get the counter to display for the cell
			displayPlayer = this->connect4Game->board.board[col][row];
			
			// Check if we are flashing because someone has won
			if (this->connect4Game->status != None && this->winningFlashShow == false && this->connect4Game->winningCellsoverlay.board[col][row] != 0)
				displayPlayer = 0;
			
			SDL_BlitSurface(this->counterGraphics[displayPlayer], NULL, surface, &screenRect);
			screenRect.x += squareSize;
		}
		screenRect.x = BOARD_BORDER_SIZE;
		screenRect.y += squareSize;
	}
	
	// Display the last move which was played
	int lastMove = this->connect4Game->getLastColumnPlayed();
	if (lastMove >= 0)
	{
		Drawing::drawDisc(surface, this->lastMoveColour, BOARD_BORDER_SIZE + lastMove * squareSize + halfSquareSize, screenRect.y + (squareSize >> 2), 5);
	}
	
	// Display the hint if one is available
	int hintColumn = this->connect4Game->playerHint;
	if (hintColumn >= 0)
	{
		Drawing::drawCircle(surface, this->hintMoveColour, BOARD_BORDER_SIZE + hintColumn * squareSize + halfSquareSize, screenRect.y + (squareSize >> 2), 8);
	}
	
	// Draw the latest stats
	char scoreText[512];
	sprintf(scoreText, "Won: %d", this->connect4Game->numGamesPlayerWon);
	this->gamesWonLabel->setLabelText(scoreText);
	
	sprintf(scoreText, "Lost: %d", this->connect4Game->numGamesPlayerLost);
	this->gamesLostLabel->setLabelText(scoreText);
	
	sprintf(scoreText, "Drawn: %d", this->connect4Game->numGamesDrawn);
	this->gamesDrawn->setLabelText(scoreText);
}

void Game::animate()
{
	// Check if someone has won so we need to make sure winning lines flash
	if (this->connect4Game->status != None && this->winningFlashTime + WIN_HIGHLIGHT_REFRESH < SDL_GetTicks())
	{
		// Update the times
		this->winningFlashTime = SDL_GetTicks();
		this->winningFlashShow = !this->winningFlashShow;
		
		// Redraw the screen
		WindowManager::Instance()->redraw();
	}
	
	// Don't bother updating the ai stuff if someone has won the game
	if (this->connect4Game->status != None)
		return;
	
	// Ask the game to play any ai oppenents
	if (this->connect4Game->getCurrentPlayerType() != AI)
		return;
		
	// Get the computer to play their counter
	PlayStatus returnvalue = this->connect4Game->playAICounter();
	
	Game::updateMessage(returnvalue);
	
	// Ask the windowmanager to redraw the screen for us
	WindowManager::Instance()->redraw();
}

bool Game::keyPress(SDLKey key, SDLMod mod, Uint16 character)
{
	return false;
}

bool Game::mouseDown(int xMouse, int yMouse, int button)
{
	bool handled = false;
	
	// Ask all of the child widgets if they want to handle the mouse click first
	handled = Window::mouseDown(xMouse, yMouse, button);
	
	if (handled)
		return true;
				
	// Check to make sure that the current player is a human one otherwise ignore the click
	if (this->connect4Game->getCurrentPlayerType() != Human)
		return false;
		
	// Check if the player has clicked in the board to warrent a move
	if (xMouse < this->boardRect.x || xMouse > this->boardRect.x + this->boardRect.w)
		return false;
	if (yMouse < this->boardRect.y || yMouse > this->boardRect.y + this->boardRect.h)
		return false;
		
	// Must have clicked in the board figure out where
	// Realign the mouse co-ordinates
	xMouse -= BOARD_BORDER_SIZE;
	
	// Find the column clicked
	int col = xMouse / squareSize;
	if (col > BOARD_COLS)
		return false;
		
	// Play the piece
	PlayStatus returnvalue = this->connect4Game->playCounter(col);
	
	Game::updateMessage(returnvalue);
	
	// Say we handled the mouse click event so that the screen can be redrawn
	return true;
}

void Game::newGame()
{
	// Start a new game
	this->connect4Game->startNewGame();
	
	// Initialise the update message
	Game::updateMessage(OK);
}

void Game::undoMove()
{
	// Always undo 2 moves one for the computer and one for the player
	this->connect4Game->unDoMoves(2);
}

// Messages
void Game::updateMessage(PlayStatus status)
{
	// Update the message based on the status of the game
	switch (status)
	{
		case OK:
			if (this->connect4Game->playersTurn == 1)
				this->message->setLabelText("Player 1's turn");
			else
				this->message->setLabelText("Player 2's turn");
			break;
		case ColumnFull:
				this->message->setLabelText("Column full");
			break;
		case BoardFull:
				this->message->setLabelText("Game is a draw");		
			break;
		case PlayerWon:
			if (this->connect4Game->playersTurn == 1)
				this->message->setLabelText("Player 1 has WON");
			else
				this->message->setLabelText("Player 2 has WON");	
			break;			
	}
}

// AI level change events
void Game::setAIEasy()
{
	Game::setAILevel(Easy);
}

void Game::setAINormal()
{
	Game::setAILevel(Normal);	
}

void Game::setAIHard()
{
	Game::setAILevel(Hard);	
}
	
void Game::setAILevel(AILevel level)
{
	this->connect4Game->setAILevel(level);
	
	// Update the check boxes to reflect the change
	this->aiEasyCheckBox->setChecked(false);
	this->aiNormalCheckBox->setChecked(false);
	this->aiHardCheckBox->setChecked(false);
	
	// Now set the correct one
	switch (level)
	{
		case Easy:
			this->aiEasyCheckBox->setChecked(true);
			break;
		case Normal:
			this->aiNormalCheckBox->setChecked(true);
			break;
		case Hard:
			this->aiHardCheckBox->setChecked(true);
			break;	
	}
}

// Get the computer to hint a move for the current player
void Game::getHint()
{
	this->connect4Game->hintMoveForCurrentPlayer();
}

