/** KXL library - KOS PVR
* 	Combine several functions (KXLmisc.c, KXLsound.c, KXLvisual.c) into one single file.
* 	PVR render + API KOS changes (2011) by Indiket
* 	Fixed resizing issues using PVR Render 2012 PH3NOM
**/

#include "KXL.h"
#include <malloc.h>
#include <stdio.h>
#include <string.h>

//Dreamcast KOS includes
#include <arch/spinlock.h>
#include <dc/pvr.h>
#include <dc/sound/stream.h>
#include <dc/sound/sfxmgr.h>

#define OBJ_TEXT 1
#define OBJ_SPRITE 2

float left=640.0f, top=0.0f, scalex=1.0f, scaley=1.0f;

spinlock_t obj_lock=SPINLOCK_INITIALIZER;
#define MAXOBJS 512
int nobjs=0;
struct {
	uint8 type;

	pvr_ptr_t addr;
	
	uint16 bw;  /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
	uint16 bh;
	uint16 scaled;
	
	uint16 tw;
	uint16 th;
	int16 u;
	int16 v;
	int16 w;
	int16 h;
	int16 x;
	int16 y;

	uint8 text[128];
	uint8 r;
	uint8 g;
	uint8 b;
} object[MAXOBJS];

uint8 fontr=1.0f;
uint8 fontg=1.0f;
uint8 fontb=1.0f;

#define abs(c) ((c>0) ? c : -c)

Bool KXL_TimerFlag = False;

static Sint16 sin360[] = {
    0,
    4,    8,   13,   17,   22,   26,   31,   35,   40,   44,
   48,   53,   57,   61,   66,   70,   74,   79,   83,   87,
   91,   95,  100,  104,  108,  112,  116,  120,  124,  127,
  131,  135,  139,  143,  146,  150,  154,  157,  161,  164,
  167,  171,  174,  177,  181,  184,  187,  190,  193,  196,
  198,  201,  204,  207,  209,  212,  214,  217,  219,  221,
  223,  226,  228,  230,  232,  233,  235,  237,  238,  240,
  242,  243,  244,  246,  247,  248,  249,  250,  251,  252,
  252,  253,  254,  254,  255,  255,  255,  255,  255,  256,
  255,  255,  255,  255,  255,  254,  254,  253,  252,  252,
  251,  250,  249,  248,  247,  246,  244,  243,  242,  240,
  238,  237,  235,  233,  232,  230,  228,  226,  223,  221,
  219,  217,  214,  212,  209,  207,  204,  201,  198,  196,
  193,  190,  187,  184,  181,  177,  174,  171,  167,  164,
  161,  157,  154,  150,  146,  143,  139,  135,  131,  127,
  124,  120,  116,  112,  108,  104,  100,   95,   91,   87,
   83,   79,   74,   70,   66,   61,   57,   53,   48,   44,
   40,   35,   31,   26,   22,   17,   13,    8,    4,    0,
   -4,   -8,  -13,  -17,  -22,  -26,  -31,  -35,  -40,  -44,
  -48,  -53,  -57,  -61,  -66,  -70,  -74,  -79,  -83,  -87,
  -91,  -95, -100, -104, -108, -112, -116, -120, -124, -127,
 -131, -135, -139, -143, -146, -150, -154, -157, -161, -164,
 -167, -171, -174, -177, -181, -184, -187, -190, -193, -196,
 -198, -201, -204, -207, -209, -212, -214, -217, -219, -221,
 -223, -226, -228, -230, -232, -233, -235, -237, -238, -240,
 -242, -243, -244, -246, -247, -248, -249, -250, -251, -252,
 -252, -253, -254, -254, -255, -255, -255, -255, -255, -256,
 -255, -255, -255, -255, -255, -254, -254, -253, -252, -252,
 -251, -250, -249, -248, -247, -246, -244, -243, -242, -240,
 -238, -237, -235, -233, -232, -230, -228, -226, -223, -221,
 -219, -217, -214, -212, -209, -207, -204, -201, -198, -196,
 -193, -190, -187, -184, -181, -177, -174, -171, -167, -164,
 -161, -157, -154, -150, -146, -143, -139, -135, -131, -128,
 -124, -120, -116, -112, -108, -104, -100,  -95,  -91,  -87,
  -83,  -79,  -74,  -70,  -66,  -61,  -57,  -53,  -48,  -44,
  -40,  -35,  -31,  -26,  -22,  -17,  -13,   -8,   -4,    0,
};

void	 KXL_GetDirectionAdd(Sint16 dir, Sint16 *x, Sint16 *y) {
  Sint16 dir2 = dir + 90;

  while (dir < 0)
    dir += 360;        
  while (dir > 360)
    dir -= 360;
  *x = sin360[dir];

  while (dir2 < 0)
    dir2 += 360;  
  while (dir2 > 360)
    dir2 -= 360;
  *y = sin360[dir2];
};

Uint16	 KXL_GetDirection(KXL_Rect src, KXL_Rect target) {
  Uint16 k, x, y;   
  Uint16 mx, my, yx, yy;

  mx = src.Left + src.Width / 2;
  my = src.Top + src.Height / 2;
  yx = target.Left + target.Width / 2;
  yy = target.Top + target.Height / 2;

  x = abs(yx - mx);
  y = abs(yy - my);
  if (yx == mx)
    k = (yy > my) ? 0 : 180;
  else if (yy == my)
    k = (yx > mx) ? 90 : 270;
  else if (yx > mx)
    if (yy > my)   
      k = 90 * x / (x + y);
    else
      k = 180 - (90 * x / (x + y));
  else
    if (yy > my)
      k = 360 - (90 * x / (x + y));
    else
      k = (90 * x / (x + y)) + 180;
  return k;
};

void	*KXL_Malloc(Uint32 size) {
  void *new;

  new = malloc(size);   
  if (new == NULL) {
    printf("\r\nKXL: Out Of memory!!\r\n");
    exit(1);
  }
  return new;
};

void	 KXL_Free(void *src) {
  free(src);
};

void	*KXL_Realloc(void *src, Uint32 size) {
  void *new;

  new = realloc(src, size);
  if (new == NULL) {
    printf("\r\nKXL: Out Of memory!!\r\n");
    exit(1);
  }
  return new;
};

/* PVR texture memory pointer; unlike the old "TA" system, PVR pointers
   in the new system are actually SH-4 compatible pointers and can
   be used directly in place of ta_txr_map(). */

/* Blend: 0 (mostly all) or 255 */
KXL_Image	*KXL_LoadBitmap(Uint8 *filename, Uint8 blend) {
	KXL_Image *new;
	KXL_BitmapHeader hed;

	KXL_ReadBitmapHeader(filename, &hed);
	new = (KXL_Image *)KXL_Malloc(sizeof(KXL_Image));
	new->BaseWidth = new->Width = hed.w;              /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
	new->BaseHeight = new->Height = hed.height;
	new->scaled=0;
	
	new->Width = hed.w;
	new->Height = hed.height;

//	printf("LoadBitmap: %p, %ix%i\r\n", new, new->Width, new->Height);

	//DC: Texture min. size must be 8 , less than 1024 and power of 2.
	new->TXRWidth = 8;
	new->TXRHeight = 8;
	while(new->TXRWidth < new->Width) new->TXRWidth <<= 1;
	while(new->TXRHeight < new->Height) new->TXRHeight <<= 1;
	//Sanity Check
	if (new->TXRWidth > 1024) new->TXRWidth = 1024;
	if (new->TXRHeight > 1024) new->TXRHeight = 1024;
	
	//Reserve PVR Memory with pvr_mem_malloc (equivalent of ta_txr_map)
	new->Buffer = pvr_mem_malloc(new->TXRWidth*new->TXRHeight*2);
	
//	printf("Buffer @ %p  ", new->Buffer);
	KXL_CreateBitmap8to16(hed.data, new, hed.rgb, blend);

	KXL_Free(hed.rgb);
	KXL_Free(hed.data);
	return new;
};

KXL_Image	*KXL_CopyImage(KXL_Image *src, KXL_Rect r) {
	KXL_Image *dest;

	dest = (KXL_Image *)KXL_Malloc(sizeof(KXL_Image));
	dest->Width = r.Width;
	dest->Height = r.Height;
	
	dest->TXRWidth = src->TXRWidth;
	dest->TXRHeight = src->TXRHeight;

	dest->BaseWidth = src->BaseWidth;        /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
    dest->BaseHeight = src->BaseHeight;    
    dest->scaled = src->scaled;
	
	dest->Buffer = &(src->Buffer[(r.Top*src->TXRWidth)+r.Left]);

	return dest;
};

void	 KXL_DeleteImage(KXL_Image *img) {
	//pvr_mem_free(img->Buffer);
	free(img);
};

void	 KXL_PutImage(KXL_Image *img, Sint16 x, Sint16 y) {
//	printf("PutImage: %p, +%i +%i %ix%i\r\n", img, x, y, img->Width, img->Height); 
	if(nobjs>MAXOBJS) return;
	spinlock_lock(&obj_lock);
	
	object[nobjs].type=OBJ_SPRITE;
	object[nobjs].addr=img->Buffer; //Direct access
	object[nobjs].tw=img->TXRWidth;
	object[nobjs].th=img->TXRHeight;
	
	object[nobjs].bw=img->BaseWidth;  /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
	object[nobjs].bh=img->BaseHeight;
	object[nobjs].scaled = img->scaled;
	
	object[nobjs].u=0;
	object[nobjs].v=0;
	object[nobjs].w=img->Width;
	object[nobjs].h=img->Height;
	object[nobjs].x=x-left;
	object[nobjs].y=y-top;
	nobjs++;
	spinlock_unlock(&obj_lock);
};

//First argument is the "Font Type", not used in DreamCast!
void	 KXL_Font(Uint8 *str, Uint8 r, Uint8 g, Uint8 b) {
//	printf("Font: %s Color: #%02x%02x%02x\r\n", str,r,g,b);
	fontr=r;
	fontg=g;
	fontb=b;
};

void	 KXL_PutText(Sint16 x, Sint16 y, Uint8 *str) {
//	printf("PutText: +%i +%i %s\r\n", x, y, str);
	if(nobjs>MAXOBJS) return;

	spinlock_lock(&obj_lock);

	object[nobjs].type=OBJ_TEXT;
	strcpy(object[nobjs].text, str);
	object[nobjs].x=x-left;
	object[nobjs].y=y-top;
	object[nobjs].r=fontr;
	object[nobjs].g=fontg;
	object[nobjs].b=fontb;
	nobjs++;

	spinlock_unlock(&obj_lock);
};

Uint16	 KXL_TextWidth(Uint8 *str) {
  return strlen(str)*8;
};

void	 KXL_DrawRectangle(Uint16 left, Uint16 top, Uint16 width, Uint16 height, Bool flag) {
//	printf("DrawRectangle: +%i +%i %ix%i  %i\r\n", left, top, width, height, flag);
};

typedef struct evt_s {
	Uint32 type;
	Uint16 key;
} evt_t;

evt_t EVT_STACK[64];
int events=0;
spinlock_t evt_lock=SPINLOCK_INITIALIZER;

void	 KXL_AddEvent(Uint32 evt, Uint16 key) {
	spinlock_lock(&evt_lock);

	if(events>=63) {
		memcpy(&EVT_STACK[0], &EVT_STACK[1], events*sizeof(evt_t));
		events--;
	}

	EVT_STACK[events].type=evt;
	EVT_STACK[events].key=key;
	events++;

	spinlock_unlock(&evt_lock);
}

Bool  KXL_GetEvent(Uint32 *evt,Uint16 *key) {
	Uint32 rv=0;
	spinlock_lock(&evt_lock);
	if(events) {
		evt[0]=EVT_STACK[0].type;
		key[0]=EVT_STACK[0].key;
		memcpy(&EVT_STACK[0], &EVT_STACK[1], events*sizeof(Uint32));
		events--;
		rv=1;
	}
	spinlock_unlock(&evt_lock);
	return rv;
};

Bool	 KXL_CheckEvents(void) {
	return ((events>0) ? 1 : 0);
};

//Update the Window. In Dreamcast, it's always fullscreen and 640x480.
//After that, call the TARender to update via PVR.

void	 KXL_UpDate(KXL_Rect r) {
	left=r.Left;
	top=r.Top;
	scalex=(640.0f/r.Width);
	scaley=(480.0f/r.Height);
	KXL_TARender();
};

void BackDrop() {
        pvr_poly_hdr_t poly;
		pvr_poly_cxt_t cxt;
        pvr_vertex_t vert;

		pvr_poly_cxt_col(&cxt, PVR_LIST_OP_POLY);
		pvr_poly_compile(&poly, &cxt);
		pvr_prim(&poly, sizeof(poly));

		vert.flags =  PVR_CMD_VERTEX;
		vert.argb = PVR_PACK_COLOR(1.0f, 0.0f, 0.0f, 0.0f);
        vert.x = 0.0f; vert.y = 480.0f; vert.z = 2.0f;
		pvr_prim(&vert, sizeof(vert));

        vert.y = 0.0f;
		pvr_prim(&vert, sizeof(vert));

        vert.x = 640.0f; vert.y = 480.0f;
		pvr_prim(&vert, sizeof(vert));

		vert.flags = PVR_CMD_VERTEX_EOL;
        vert.y = 0.0f;
		pvr_prim(&vert, sizeof(vert));
}

void	 KXL_DisplayName(Uint8 *name) {};

extern void draw_string(float, float, float, float, float, float, float, float, float, char *, int);
void	 DrawObjects() {
	int i;

	spinlock_lock(&obj_lock);

	for(i=0; i<nobjs; i++) {
		if(object[i].type==OBJ_TEXT) {
			draw_string(object[i].x, object[i].y, 3+i, scalex, scaley, 1.0f, object[i].r, object[i].g, object[i].b, object[i].text, strlen(object[i].text));
		}
		if(object[i].type==OBJ_SPRITE) {
			//Possible TODO: Use PVR Sprites, a simplified version of a polygon.
			pvr_poly_hdr_t poly;
			pvr_vertex_t vert;
			pvr_poly_cxt_t cxt;
			
			//Our BMP textures are NOT TWIDDLED!!
			pvr_poly_cxt_txr(&cxt, PVR_LIST_TR_POLY, PVR_TXRFMT_ARGB1555|PVR_TXRFMT_NONTWIDDLED, object[i].tw, object[i].th, object[i].addr, PVR_FILTER_NONE);
			pvr_poly_compile(&poly, &cxt);
			pvr_prim(&poly, sizeof(poly));
		
			float x1, x2, y1, y2;
			float u1, u2, v1, v2;

			u1=object[i].u/(object[i].tw*1.0f);
			v1=object[i].v/(object[i].th*1.0f);
			
			if(object[i].scaled==1) /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
			{
               u2=object[i].bw/(object[i].tw*1.0f); 
			   v2=object[i].bh/(object[i].th*1.0f);
            }
            else
            {
               u2=object[i].w/(object[i].tw*1.0f);
			   v2=object[i].h/(object[i].th*1.0f);
            }

			x1=object[i].x;
			x2=x1+object[i].w;
			y1=object[i].y;
			y2=y1+object[i].h;

			//Render to fullscreen
			x1*=scalex;
			x2*=scalex;
			y1*=scaley;
			y2*=scaley;

			vert.flags = PVR_CMD_VERTEX;
			vert.x = x1; vert.y = y2; vert.z = 3+i;
			vert.u = u1; vert.v = v2;
			vert.argb = PVR_PACK_COLOR(1.0f, 1.0f, 1.0f, 1.0f);
			vert.oargb = 0.0f;
			pvr_prim(&vert, sizeof(vert));

			vert.x = x1; vert.y = y1;
			vert.u = u1; vert.v = v1;
			pvr_prim(&vert, sizeof(vert));

			vert.x = x2; vert.y = y2;
			vert.u = u2; vert.v = v2;
			pvr_prim(&vert, sizeof(vert));

			vert.flags = PVR_CMD_VERTEX_EOL;
			vert.x = x2; vert.y = y1;
			vert.u = u2; vert.v = v1;
			pvr_prim(&vert, sizeof(vert));
		}
	}
	nobjs=0;
	spinlock_unlock(&obj_lock);
}

/*PVR: When using the PVR, one needs to tell it when beginning a new scene, and what polygon lists will be used.
The Dreamcast uses display lists when it's drawing the screen, 
and it expects all the opaque drawing instructions to be in one part of the list, followed by all the transparent parts. */
int KXL_TARender() {
	
	/* Start opaque poly list */
	pvr_scene_begin();
	
	pvr_list_begin(PVR_LIST_OP_POLY);
	BackDrop();
	pvr_list_finish(); 	/* End of opaque list */
	
	pvr_list_begin(PVR_LIST_TR_POLY);
	DrawObjects();
	KXL_AddEvent(KXL_EVENT_EXPOSE,0);
	pvr_list_finish(); /* End of TR list */
	
	/* Finish the frame */
	pvr_scene_finish();
	
	while(pvr_check_ready()!=0){

	}
	pvr_wait_ready();
	return 0;
}

void	 KXL_CreateWindow(Uint16 w, Uint16 h, Uint8 *title, Uint32 event) {
};

void	 KXL_ReSizeFrame(Uint16 w, Uint16 h) {
};

void	 KXL_ClearFrame(KXL_Rect r) {
};

void	 KXL_DeleteWindow(void) {
};

//-----------------------------------------------------
//KOS Sound API
//-----------------------------------------------------

int	*KXL_wavelist;
uint32	*KXL_musiclist;

void	 KXL_InitSound(Uint8 **fname) {
	Uint16 i, max=0;

	//Count how many sounds do we have...
	while (fname[max][0]) max ++;

	KXL_wavelist=malloc(sizeof(int)*max);

	//Init KOS Sound Stream
	snd_stream_init();	

	for (i = 0; i < max; i ++)
		KXL_wavelist[i] = snd_sfx_load(fname[i]);
};

static snd_stream_hnd_t handle = -1;
//It's very important to set a correct buffer: 8192 works fine!
static unsigned sound_buffer[8192];
Uint16 dc_song = -1;
kthread_t * thd;

void	 KXL_InitMusic(Uint8 **fname) {
	Uint16 i, max=0;

	while (fname[max][0]) max ++;

	KXL_musiclist=malloc(sizeof(uint32)*max);

	//Open WAV File
	for (i = 0; i < max; i ++){
		KXL_musiclist[i] = fs_open(fname[i], O_RDONLY);
	}
};

void *SB_callback(snd_stream_hnd_t hnd, int len, int * actual)
{
	//Reading sample...
	int bytes_read = fs_read(KXL_musiclist[dc_song],sound_buffer,len);
	*actual = len;
	
	//If we finish the file, loop again!
	if (bytes_read < len){
		fs_seek(KXL_musiclist[dc_song], 0x32, SEEK_SET);
	}
	
	return (Uint16*)(sound_buffer);
}

//BGM1 and BOSS are looped WAV's
void	 KXL_PlaySound(Uint16 no, KXL_Command action) {
	if(action==KXL_SOUND_PLAY){
		snd_sfx_play(KXL_wavelist[no], 255, 128);
	}
	else if(action==KXL_SOUND_PLAY_LOOP){
		//Check if we have to stop a song!
		KXL_EndMusic();
		
		//New song :)
		dc_song = no;
		
		//SKIP WAV header
		fs_seek(KXL_musiclist[dc_song], 0x32, SEEK_SET);
		
		//Allocate buffer?
		if (handle == -1) {
			handle = snd_stream_alloc(*SB_callback, 8192);
		}
		snd_stream_start(handle, 8000, 0); //8000 HZ, Mono
		snd_stream_volume(handle, 180);
	}
	else if(action==KXL_SOUND_PLAY_LOOP){
		KXL_EndMusic();
	}
	else if(action==KXL_SOUND_STOP_ALL){
		KXL_EndMusic();
	}
};

void	 KXL_EndSound(void) {
	snd_stream_shutdown();
	
	//Also, close files
	int i;
	for (i = 0; i < sizeof(KXL_musiclist); i ++){
		fs_close(KXL_musiclist[i]);
	}
};

//Just stops the music, but we don't free the stream
void	 KXL_EndMusic(void) {
	if (handle != -1){
		snd_stream_stop(handle);
		sq_clr(sound_buffer, 8192);
		dc_song = -1;
	}
};

void	 KXL_UpdateMusic(void) {
	if ((handle != -1) && (dc_song != -1)) {
		snd_stream_poll(handle);
	}
}

//-----------------------------------------------------
//KOS Thread API
//-----------------------------------------------------

void	 KXL_ResetTimer(void) {
	KXL_TimerFlag = False;
};

void	 *KXL_TimerThread(void *param) {
	while(1) {
		thd_sleep((int)(param));
		KXL_TimerFlag = True;
	}
};


void	 KXL_Timer(Uint16 ti) {
	int timerival = 1000/ti;
	thd_create(1,KXL_TimerThread,(int*)(timerival));  //Send calculated FRAME_RATE to the thread.
	KXL_TimerFlag = False;
};

Bool	 KXL_GetTimer(void) {
	return KXL_TimerFlag;
};

//Uint32	 time(Uint32 *t) { return jiffies/100; };

Uint32 KXL_ReadU32(FILE *fp)
{
  Uint8 c[4];
  Uint8 i;

  for (i = 0; i < 4; i ++)
    c[i] = fgetc(fp);
  return (Uint32)(c[0] + c[1] * 0x100L + c[2] * 0x10000L + c[3] * 0x1000000L);
}

Uint16 KXL_ReadU16(FILE *fp)
{
  Uint8 c[2];
  Uint8 i;
 
  for (i = 0; i < 2; i ++)
    c[i] = fgetc(fp);
  return (Uint16)(c[0] +  c[1] * 0x100);
}

void KXL_PutImageImm(KXL_Image *img, Uint16 src_l, Uint16 src_t, Uint16 src_w, Uint16 src_h, Sint16 x, Sint16 y)
{
	if(nobjs>MAXOBJS) return;

	spinlock_lock(&obj_lock);
//printf("PutImage: %p, +%i +%i %ix%i\r\n", img, x, y, src_w, src_h); 
	object[nobjs].type=OBJ_SPRITE;
	object[nobjs].addr=img->Buffer; //Direct access
	
	object[nobjs].tw=img->TXRWidth;
	object[nobjs].th=img->TXRHeight;
	object[nobjs].u=src_l;
	object[nobjs].v=src_t;
	object[nobjs].w=src_w;
	object[nobjs].h=src_h;
	object[nobjs].x=x-left;
	object[nobjs].y=y-top;
	nobjs++;
	spinlock_unlock(&obj_lock);
}

//  Copy from source image to new image
KXL_Image  *KXL_CopyImageImm(KXL_Image *src, Uint16 src_l, Uint16 src_t, Uint16 src_w, Uint16 src_h) {
	KXL_Image *dest;

	dest = (KXL_Image *)KXL_Malloc(sizeof(KXL_Image));
	dest->Width = src_w;
	dest->Height = src_h;
	dest->TXRWidth = src->TXRWidth;
	dest->TXRHeight = src->TXRHeight;

	dest->Buffer = &(src->Buffer[(src_t*src->TXRWidth)+src_l]);
	return dest;
}

//All we need to do is point to the original texture in VRAM, then let the PVR handle re-sizing during render (u,v)
KXL_Image  *KXL_StrechImage(KXL_Image *src, Uint16 width, Uint16 height)
{
	KXL_Image *dest;

	dest = (KXL_Image *)KXL_Malloc(sizeof(KXL_Image));
	dest->Width  = width;
	dest->Height = height;
	dest->TXRWidth = src->TXRWidth;
	dest->TXRHeight = src->TXRHeight; 
	
	dest->BaseWidth  = src->BaseWidth;   /* Added BaseWidth & BaseHeight (C) PH3NOM 2012 */
	dest->BaseHeight = src->BaseHeight;
	dest->scaled = 1;
    
	dest->Buffer = src->Buffer;
	return dest;
}

void KXL_PutRectImage(KXL_Image *img, KXL_Rect r, Sint16 x, Sint16 y)
{
  KXL_Image *dest = KXL_CopyImage(img, r);
  KXL_PutImage(dest, x, y);
  KXL_DeleteImage(dest);
}
