// Copyright (C) 2002 Vincent Bherer-Roy
// 
// YaSFCave is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// YaSFCave is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// main.cpp

#include <cstring>
#include <cstdlib>

#include "SDL.h"

#include "Game.h"
#include "Map.h"

#ifdef DREAMCAST
	#include <kos.h>
	#include <SDL_dreamcast.h>
	extern uint8 romdisk[];
	KOS_INIT_FLAGS(INIT_DEFAULT);
	KOS_INIT_ROMDISK(romdisk);
#endif

enum Screen
{
	kGameScreen,
	kStartScreen,
	kPauseScreen,
	kGameOverScreen,
	kAbout1Screen,
	kAbout2Screen
};

int main(int _argc, char* _argv[])
{
	#ifdef DREAMCAST
		printf("Starting...\n");
		fs_chdir("/rd");
	#endif
	// Initialize systems
	SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK);
	
	// Set our at exit function
	atexit(SDL_Quit);
	srand(SDL_GetTicks());
	
	Game game;

	game.DrawStart();
	
	/* Init joystick */
	#ifdef DREAMCAST
		SDL_JoystickOpen (0);
		SDL_DC_MapKey(0,SDL_DC_START,SDLK_r); //Reset Hiscores
		SDL_DC_MapKey(0,SDL_DC_UP,SDLK_UP);
		SDL_DC_MapKey(0,SDL_DC_DOWN,SDLK_DOWN);
		SDL_DC_MapKey(0,SDL_DC_LEFT,SDLK_LEFT); 
		SDL_DC_MapKey(0,SDL_DC_RIGHT,SDLK_RIGHT);
		SDL_DC_MapKey(0,SDL_DC_A,SDLK_SPACE); //A: Moving
		SDL_DC_MapKey(0,SDL_DC_B,SDLK_a); //B:About
		SDL_DC_MapKey(0,SDL_DC_R,SDLK_ESCAPE); //L:exits
		SDL_DC_MapKey(0,SDL_DC_L,SDLK_p); //R:Pause
		
		SDL_DC_EmulateMouse(SDL_FALSE);
		SDL_ShowCursor(0);
		
		//Dreamcast delay...
		SDL_Event dcevent;
		SDL_Delay(20);
		while(SDL_PollEvent(&dcevent))
			SDL_Delay(20);
	#endif

	// Start the game
	float fFPS		= 45.0f; // fFPS frame per second
	float fSpeedX	= 45.0f; // fSpeedX pixel per second

	Uint32 uiFrameMS = (Uint32)(1000.0f / fFPS);
	Uint32 uiLogicMS = (Uint32)(1000.0f / fSpeedX);

	bool bQuit				= false;
	bool bReleasedBefore	= false;
	bool bFirstTime			= true;
	
	Screen eScreen;
	Screen eLastScreen;
	eScreen = eLastScreen = kStartScreen;
	
	Uint32 uiLastLogicMS;
	Uint32 uiLastFrameMS;
	uiLastLogicMS = uiLastFrameMS = SDL_GetTicks();

	// Message pump
	while(!bQuit)
	{
		SDL_Event event;

		// Look for an event
		if(SDL_PollEvent(&event))
		{
			// An event was found
			switch(event.type)
			{
			case SDL_KEYUP:
				switch(event.key.keysym.sym)
				{
				case SDLK_a:
				case SDLK_F1:
					if(eScreen == kAbout1Screen)
						eScreen = kAbout2Screen;
					else
						eScreen = kAbout1Screen;
					bFirstTime = true;
					break;
				case SDLK_p:
				case SDLK_F2:
					if(eScreen == kGameScreen)
					{
						eScreen = kPauseScreen;
						bFirstTime = true;
					}
					break;
				case SDLK_r:
				case SDLK_F3:
					game.ResetHighScore();
					break;
				case SDLK_ESCAPE:
					bQuit = true;
					break;
				}
				break;

			case SDL_QUIT:
				bQuit = true;
				break;
			}
		}
		else
		{
			Uint8* pKeys = SDL_GetKeyState(NULL);

			bool bClick = (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(1)) != 0
							|| pKeys[SDLK_SPACE] != 0;
			
			if(eScreen == kGameScreen)
			{
				Uint32 uiCurMS			= SDL_GetTicks();
				Uint32 uiFrameDiffMS	= uiCurMS - uiLastFrameMS;
				Uint32 uiLogicDiffMS	= uiCurMS - uiLastLogicMS;
				
				if(uiLogicDiffMS >= uiLogicMS)
				{
					uiLastLogicMS = uiCurMS;
				
					if(game.Update(bClick))
					{
						eScreen = kGameOverScreen;
						game.DrawGame();
					}
					else
					{
						if(uiFrameDiffMS >= uiFrameMS)
						{
							uiLastFrameMS = uiCurMS;
							game.DrawGame();
						}
					}
				}
			}
			else
			{
				if(bFirstTime)
				{
					bFirstTime = false;
					bReleasedBefore = false;

					SDL_WM_GrabInput(SDL_GRAB_OFF);
					#ifndef DREAMCAST
					SDL_ShowCursor(1);
					#endif
					
					switch(eScreen)
					{
					case kStartScreen:
						game.DrawStart();
						break;
					case kPauseScreen:
						game.DrawPause();
						break;
					case kGameOverScreen:
						game.DrawGameOver();
						break;
					case kAbout1Screen:
						game.DrawAbout1();
						break;
					case kAbout2Screen:
						game.DrawAbout2();
						break;
					}

				}
				else
				{
					if(!bClick)
						bReleasedBefore = true;

					if(bClick && bReleasedBefore)
					{
						switch(eScreen)
						{
						case kStartScreen:
							game.NewGame();

							SDL_WM_GrabInput(SDL_GRAB_ON);
							SDL_ShowCursor(0);

							uiLastFrameMS = uiLastFrameMS = SDL_GetTicks();
							eScreen = eLastScreen = kGameScreen;
							bFirstTime = true;
							break;

						case kPauseScreen:
							eScreen = kGameScreen;
							bFirstTime = true;
							break;

						case kGameOverScreen:
							eScreen = eLastScreen = kStartScreen;
							bFirstTime = true;
							break;

						case kAbout1Screen:
							eScreen = kAbout2Screen;
							bFirstTime = true;
							break;
						case kAbout2Screen:
							if(eLastScreen == kGameScreen)
							{
								game.DrawGame(true);
								SDL_WM_GrabInput(SDL_GRAB_ON);
								SDL_ShowCursor(0);
							}

							eScreen = eLastScreen;
							bFirstTime = true;
							break;
						}

						bFirstTime = true;
					}
				}
			}
		}
	}
	// End of message pump

	#ifndef DREAMCAST
	SDL_ShowCursor(1);
	#endif
	SDL_WM_GrabInput(SDL_GRAB_OFF);

	game.EndGame();

	// Done
	return 0;
}

