// Copyright (C) 2002 Vincent Bherer-Roy
// 
// YaSFCave is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// YaSFCave is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// Map.cpp

#include "Map.h"

#include <cstdlib>

Map::Map(SDL_Surface* _pSurface, int _w, int _h) :
	mW(_w),
	mH(_h),
	mpSurface(NULL),
	mOffsetX(0),
	mLastColumnX(_w - 1),
	mWormX(_w / 3),
	mfWormY((float)(_h / 2)),
	mNextWormY(_h / 2),
	mLastWormY(_h / 2),
	mfSpeedY(0.0f),
	mfYAcc(0.09f),
	mTunnelHeight(_h - 40)
{
	mfTunnelBegin = (float)((mH - mTunnelHeight) / 2);
	mfIncTunnelBegin = 0.0f;
	mfNextTunnelBegin = mfTunnelBegin;

	Uint32 uiBorderHeight = (Uint32)mfTunnelBegin;

	// Init collide infos
	mpCollideInfos = new CollideInfo[2 * mW];
	for(int i = 0; i < (2 * mW); i++)
	{
		mpCollideInfos[i].tunnelBegin	= uiBorderHeight;
		mpCollideInfos[i].tunnelEnd		= uiBorderHeight + mTunnelHeight;
		mpCollideInfos[i].obstacleBegin	= mH;
		mpCollideInfos[i].obstacleEnd	= 0;
	}

	SDL_PixelFormat* pFormat = _pSurface->format;

	mWormColor		= SDL_MapRGB(pFormat, 255, 0, 0);
	mBackColor		= SDL_MapRGB(pFormat, 16, 16, 16);
	mBorderColor	= SDL_MapRGB(pFormat, 64, 255, 0);

	// Init surface

	// Try to create a hardware surface
	mpSurface = SDL_CreateRGBSurface(SDL_HWSURFACE, 2 * mW, mH, pFormat->BitsPerPixel, 
																pFormat->Rmask, 
																pFormat->Gmask,
																pFormat->Bmask,
																pFormat->Amask);
	if(mpSurface == NULL)
	{
		// if it doesn't work try software
		mpSurface = SDL_CreateRGBSurface(SDL_SWSURFACE, 2 * mW, mH, pFormat->BitsPerPixel, 
																	pFormat->Rmask, 
																	pFormat->Gmask,
																	pFormat->Bmask,
																	pFormat->Amask);
	}

	// Draw Border
	SDL_Rect dstRect;
	dstRect.x = 0;
	dstRect.y = 0;
	dstRect.w = mW;
	dstRect.h = uiBorderHeight;
	SDL_FillRect(mpSurface, &dstRect, mBorderColor);
	dstRect.y = dstRect.h;
	dstRect.h = mTunnelHeight;
	SDL_FillRect(mpSurface, &dstRect, mBackColor);
	dstRect.y = uiBorderHeight + mTunnelHeight;
	dstRect.h = uiBorderHeight;
	SDL_FillRect(mpSurface, &dstRect, mBorderColor);

	muiFrame = 0;
}

Map::~Map()
{
	SDL_FreeSurface(mpSurface);
	delete [] mpCollideInfos;
}

int Random(int _from, int _to)
{
	int randomNumber;

	if(_to > _from)
	{
		randomNumber = rand();
		randomNumber %= _to - _from + 1;
		randomNumber += _from;
	}
	else
	{
		randomNumber = _from;
	}
	
	return randomNumber;
}

void Map::AddColumn()
{
	// Check if it's time to shrink the tunnel
	if((muiFrame % 128) == 0)
		mTunnelHeight -= 2;

	// Place the tunnel at his new position
	mfTunnelBegin += mfIncTunnelBegin;

	Uint32 uiBorderHeight = (int)mfTunnelBegin;

	// Set the collide infos
	CollideInfo* pCollideInfo = &mpCollideInfos[mLastColumnX];
	pCollideInfo->tunnelBegin	= uiBorderHeight;
	pCollideInfo->tunnelEnd		= uiBorderHeight + mTunnelHeight;

	// Now that we know what is in the new column, draw it
	SDL_Rect dstRect;
	dstRect.x = mLastColumnX;
	dstRect.y = 0;
	dstRect.w = 1;
	dstRect.h = uiBorderHeight;
	SDL_FillRect(mpSurface, &dstRect, mBorderColor);
	dstRect.y = uiBorderHeight;
	dstRect.h = mTunnelHeight + 1;
	SDL_FillRect(mpSurface, &dstRect, mBackColor);
	dstRect.y = uiBorderHeight + mTunnelHeight;
	dstRect.h = mH - dstRect.y;
	SDL_FillRect(mpSurface, &dstRect, mBorderColor);

	// Check if it's time to add an obstacle (have to be done after the background draw)
	if((muiFrame % 64) == 0)
	{
		dstRect.h = mH / 8;
		dstRect.y = (Sint16)Random(uiBorderHeight, uiBorderHeight + mTunnelHeight - dstRect.h);
		SDL_FillRect(mpSurface, &dstRect, mBorderColor);

		pCollideInfo->obstacleBegin = dstRect.y;
		pCollideInfo->obstacleEnd	= dstRect.y + dstRect.h;
	}
	else
	{
		pCollideInfo->obstacleBegin = mH;
		pCollideInfo->obstacleEnd	= 0;
	}

	// Check if it's time to change the direction of the tunnel
	if((muiFrame % 32) == 0)
	{
		mfNextTunnelBegin = (float)Random(0, mH - mTunnelHeight);
		if(mfNextTunnelBegin < 1.0f)
			mfNextTunnelBegin = 1.0f;
		else if(mfNextTunnelBegin >= (float)(mH - 1))
			mfNextTunnelBegin = (float)(mH - 2);
		mfIncTunnelBegin = (mfNextTunnelBegin - mfTunnelBegin) / 32.0f;
	}
}

#define mymax(a,b)    (((a) > (b)) ? (a) : (b))
#define mymin(a,b)    (((a) < (b)) ? (a) : (b))

bool Map::MoveWorm(bool _bClick)
{
	// If the user click, accelerate up, else accelerate down
	if(_bClick)
		mfSpeedY -= mfYAcc;
	else
		mfSpeedY += mfYAcc;

	mfWormY += mfSpeedY;

	// Guards
	if(mfWormY < 0.0f)
	{
		mNextWormY	= 0;
		mfWormY		= 0.0f;
		mfSpeedY	= 0.0f;
	}
	else if(mfWormY >= (mW - 2))
	{
		mNextWormY	= mH - 2;
		mfWormY		= (float)mNextWormY;
		mfSpeedY	= 0.0f;
	}
	else
	{
		mNextWormY	= (int)mfWormY;
	}

	// Check for collision with border or obstacle
	int minPos;
	int maxPos;
	if(mNextWormY <= mLastWormY)
	{
		minPos = mNextWormY;
		maxPos = mLastWormY;
	}
	else
	{
		minPos = mLastWormY;
		maxPos = mNextWormY + 1;
	}

	// Collide
	CollideInfo* pCollideInfo = &mpCollideInfos[mWormX];

	if(minPos < pCollideInfo->tunnelBegin)
	{
		mNextWormY = pCollideInfo->tunnelBegin;
	}
	else if(maxPos >= pCollideInfo->tunnelEnd)
	{
		mNextWormY = pCollideInfo->tunnelEnd - 2;
	}
	else if(minPos >= pCollideInfo->obstacleBegin && maxPos <= pCollideInfo->obstacleEnd)
	{
		if(mNextWormY > mLastWormY)
			mNextWormY = mLastWormY;
	}
	else if(minPos <= pCollideInfo->obstacleBegin && maxPos >= pCollideInfo->obstacleBegin)
	{
		mNextWormY = mymax(minPos, pCollideInfo->obstacleBegin);
	}
	else if(minPos <= pCollideInfo->obstacleEnd && maxPos >= pCollideInfo->obstacleEnd)
	{
		mNextWormY = mymin(maxPos, pCollideInfo->obstacleEnd);
	}
	else
	{
		return false;
	}

	return true;
}

void Map::DrawWorm()
{
	SDL_Rect dstRect;

	dstRect.x = mWormX;
	dstRect.w = 1;

	if(mNextWormY < mLastWormY)
	{
		dstRect.y = mNextWormY;
		dstRect.h = mLastWormY - mNextWormY + 2;
	}
	else if(mNextWormY > mLastWormY)
	{
		dstRect.y = mLastWormY;
		dstRect.h = mNextWormY - mLastWormY + 2;
	}
	else
	{
		dstRect.y = mNextWormY;
		dstRect.h = 2;
	}
	
	SDL_FillRect(mpSurface, &dstRect, mWormColor);

	mLastWormY = mNextWormY;
}

Uint32 Map::Update(bool _bClick)
{
	mOffsetX		= muiFrame % (2 * mW);
	mLastColumnX	= (muiFrame + mW) % (2 * mW);
	mWormX			= (muiFrame + (mW / 3)) % (2 * mW);

	// Draw Border
	AddColumn();

	// Move the worm
	bool bCollide = MoveWorm(_bClick);

	// Draw the worm
	DrawWorm();

	// If collide, return the score
	if(bCollide)
		return muiFrame;

	// Increment the counters
	muiFrame++;

	return 0;
}

void Map::Blit(SDL_Surface* _pSurface, SDL_Rect* _pDstRect)
{
	SDL_Rect dstRect = *_pDstRect;
	SDL_Rect srcRect;
	srcRect.x = mOffsetX;
	srcRect.y = 0;
	srcRect.h = mH;

	// If the view doesn't overlap with the end of the buffer
	if(mOffsetX <= mW)
	{
		// simply blit
		srcRect.w = mW;
	}
	else
	{
		// else blit the first part (at the end of the buffer)
		srcRect.w = dstRect.w = (2 * mW) - mOffsetX;

		SDL_BlitSurface(mpSurface, &srcRect, _pSurface, &dstRect);

		// and the second part (at the beginning)
		srcRect.x = 0;
		dstRect.x += srcRect.w;
		srcRect.w = dstRect.w = mW - srcRect.w;
	}

	SDL_BlitSurface(mpSurface, &srcRect, _pSurface, &dstRect);
}
