// Copyright (C) 2002 Vincent Bherer-Roy
// 
// YaSFCave is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// YaSFCave is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// Game.cpp

#include "Game.h"

#include "Map.h"

#include "SDL_image.h"

#ifdef DREAMCAST
#include <kos.h>
#include <zlib/zlib.h> //VMU Compression
#include "dreamcast_icon.h"

int DC_SaveToVMU(char *src) {
    file_t file;
    int filesize = 0;
    unsigned long zipsize = 0;
    uint8 *data;
    uint8 *zipdata;
    vmu_pkg_t pkg;
    uint8 *pkg_out;
    int pkg_size;

    // Reads in the file from the CWD
    file = fs_open(src, O_RDONLY);
	if(file == 0) return -1;
    filesize = fs_total(file);
	if (filesize <= 0) {
		printf("error writing\n");
		return -1;
	}
    data = (uint8*)malloc(filesize+1);
    fs_read(file, data, filesize);
    fs_close(file); //Don't delete src file!
	
    // Allocate some memory for compression
    zipsize = filesize * 2;
    zipdata = (uint8*)malloc(zipsize);
	
    // The compressed save
    compress(zipdata, &zipsize, data, filesize);
	//zipsive has changed its size to the compress data!

    // Required VMU header
    // You will have to have a VMU icon defined
    strcpy(pkg.desc_short, "Yasfcave");
    strcpy(pkg.desc_long, "HiScores");
    strcpy(pkg.app_id, "Yasfcave");
    pkg.icon_cnt = 1;
	memcpy((void *)&pkg.icon_pal[0],(void *)&vmu_cave_pal,32);
    pkg.icon_data = (const uint8*)&vmu_cave_data;
	pkg.icon_anim_speed = 0;
    pkg.eyecatch_type = VMUPKG_EC_NONE;
    pkg.data_len = zipsize;
    pkg.data = zipdata;
	
    vmu_pkg_build(&pkg, &pkg_out, &pkg_size);

    // Save the newly created VMU save to the VMU
    fs_unlink("/vmu/a1/yasfcave");
    file = fs_open("/vmu/a1/yasfcave", O_WRONLY);
	if (!file) {
		printf("error writing\n");
		return -1;
	}
    fs_write(file, pkg_out, pkg_size);
    fs_close(file);

    // Free unused memory
    free(pkg_out);
    free(data);
    free(zipdata);

    return 0;
}

int DC_LoadFromVMU(char *dst) {

    file_t file;
    int filesize;
    unsigned long unzipsize;
    uint8* data;
    uint8* unzipdata;
    vmu_pkg_t pkg;

    // Remove VMU header
    file = fs_open("/vmu/a1/yasfcave", O_RDONLY);
    if(!file) return -1;
    filesize = fs_total(file);
    if(filesize <= 0) return -1;
    data = (uint8*)malloc(filesize);
    fs_read(file, data, filesize);
    vmu_pkg_parse(data, &pkg);
    fs_close(file);

    // Allocate memory for the uncompressed data
    unzipdata = (uint8 *)malloc(65536);
    unzipsize = 65536;

    // Uncompress the data to the CWD
    uncompress(unzipdata, &unzipsize, (uint8 *)pkg.data, pkg.data_len);
    fs_unlink(dst); //clear previous data
    file = fs_open(dst, O_WRONLY);
    fs_write(file, unzipdata, unzipsize);
    fs_close(file);

    // Free unused memory
    free(data);
    free(unzipdata);

    return 0;
}
#endif

Game::Game() :
	mpMap(NULL),
	mpSurface(NULL),
	muiHighScore(0),
	muiSpacing(5)
{
	// Read the high score
	FILE* pFile;
	#ifdef DREAMCAST
		//Load file first from VMU or SD, and save into RAM		
		if((pFile = fopen("/sd/highscore.dat","r"))==NULL) {
			//no HighScore in SD available, so try VMU -> RAM
			DC_LoadFromVMU("/ram/highscore.dat");
			pFile = fopen("/ram/highscore.dat", "rb");
		}
	#else
		pFile = fopen("highscore.dat", "rb");
	#endif
	
	if(pFile)
	{
		fread(&muiHighScore, 4, 1, pFile);
		fclose(pFile);
	}

	mWinRect.x = 0;
	mWinRect.y = 0;
	mWinRect.w = 640;
	mWinRect.h = 480;

	mViewRect.w = 256;
	mViewRect.h = 128;
	mViewRect.x = (mWinRect.w - mViewRect.w) / 2;
	mViewRect.y = (mWinRect.h - mViewRect.h) / 2;

	mpSurface = SDL_SetVideoMode(mWinRect.w, mWinRect.h, 0, SDL_ANYFORMAT);
	#ifndef DREAMCAST
		SDL_WM_SetCaption("Yet Another SFCave", NULL);
	#endif
	
	// Load the font - You don't have to use the IMGlib for this
	mFontBlue.Surface	= IMG_Load("fontblue.png");
	mFontRed.Surface	= IMG_Load("fontred.png");

	// Prepare the font for use
	InitFont2(&mFontBlue);
	InitFont2(&mFontRed);

	mBackColor		= SDL_MapRGB(mpSurface->format, 0, 0, 0);
	mBorderColor	= SDL_MapRGB(mpSurface->format, 0, 0, 128);
}

Game::~Game()
{
	SDL_FreeSurface(mFontBlue.Surface);
	SDL_FreeSurface(mFontRed.Surface);

	EndGame();
}

bool Game::Update(bool _bClick)
{
	Uint32 uiScore = mpMap->Update(_bClick);

	if(uiScore != 0)
	{
		Uint32 uiScore = mpMap->GetScore();
		if(muiHighScore < uiScore)
		{
			muiHighScore = uiScore;
			FILE* pFile;
			#ifdef DREAMCAST
				//Try to create HiScores file in SD...
				if((pFile = fopen("/sd/highscore.dat","w+b"))==NULL) {
					//Cannot create file on SD (maybe does not exist?)
					pFile = fopen("/ram/highscore.dat", "w+b");
				}
			#else
				pFile = fopen("highscore.dat", "w+b");
			#endif
			
			if(pFile)
			{
				fwrite(&muiHighScore, 4, 1, pFile);
				fclose(pFile);
			}
			#ifdef DREAMCAST
				//Check if file exists on SD
				if((pFile = fopen("/sd/highscore.dat","r"))==NULL) {
					DC_SaveToVMU("/ram/highscore.dat");
				}
				else fclose(pFile);
			#endif
		}

		return true;
	}
	else
	{
		return false;
	}
}

void Game::DrawBorder()
{
	SDL_Rect dstRect;
	dstRect.x = mViewRect.x - 1;
	dstRect.y = mViewRect.y - 1;
	dstRect.w = mViewRect.w + 2;
	dstRect.h = mViewRect.h + 2;
	SDL_FillRect(mpSurface, &mWinRect, mBackColor);
	SDL_FillRect(mpSurface, &dstRect, mBorderColor);
	SDL_FillRect(mpSurface, &mViewRect, mBackColor);
}

void Game::DrawGame(bool _bRefresh)
{
	if(!_bRefresh)
	{
		mpMap->Blit(mpSurface, &mViewRect);

		SDL_UpdateRects(mpSurface, 1, &mViewRect);
	}
	else
	{
		DrawBorder();

		mpMap->Blit(mpSurface, &mViewRect);

		SDL_UpdateRects(mpSurface, 1, &mWinRect);
	}
}

void Game::DrawStart()
{
	DrawBorder();

	int textWidth;
	char str[2048];

	int y = 20;
	sprintf(str, "Yet Another SFCave 1.0");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	#ifdef DREAMCAST
	sprintf(str, "Press A to play!");
	#else
	sprintf(str, "Click to play!");
	#endif
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				(mWinRect.h - mFontBlue.h) / 2, 
				str);

	y = mViewRect.y + mViewRect.h + 3 * muiSpacing;
	#ifdef DREAMCAST
	sprintf(str, "B (F1): About this game");
	#else
	sprintf(str, "F1 or A: About this game");
	#endif
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += muiSpacing + mFontBlue.h;
	#ifdef DREAMCAST
	sprintf(str, "R (F2) : Pause");
	#else
	sprintf(str, "F2 or P : Pause");
	#endif
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += muiSpacing + mFontBlue.h;
	#ifdef DREAMCAST
	sprintf(str, "Start (F3) : Reset high score");
	#else
	sprintf(str, "F3 or R : Reset high score");
	#endif
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += muiSpacing + mFontBlue.h;
	#ifdef DREAMCAST
	sprintf(str, "L (Esc) : Quit");
	#else
	sprintf(str, "Esc : Quit");
	#endif
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}

void Game::DrawPause()
{
	DrawBorder();

	mpMap->Blit(mpSurface, &mViewRect);

	int textWidth;
	char str[2048];

	sprintf(str, "Pause");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				(mWinRect.h - mFontBlue.h) / 2, 
				str);

	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}

void Game::DrawGameOver()
{
	DrawBorder();

	mpMap->Blit(mpSurface, &mViewRect);

	int textWidth;
	char str[2048];

	int y = 20;
	sprintf(str, "Yet Another SFCave 1.0");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	sprintf(str, "Game Over!");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				mViewRect.y - mFontBlue.h - muiSpacing, 
				str);

	y = mViewRect.y + mViewRect.h + muiSpacing;
	sprintf(str, "Score : %d", mpMap->GetScore());
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += muiSpacing + mFontBlue.h;
	sprintf(str, "High Score : %d", muiHighScore);
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y,
				str);
	
	y += muiSpacing + mFontBlue.h;
	#ifdef DREAMCAST
	sprintf(str, "Press button to continue.");
	#else
	sprintf(str, "Click to continue.");
	#endif	
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}

void Game::DrawAbout1()
{
	SDL_FillRect(mpSurface, &mWinRect, mBackColor);

	int textWidth;
	char str[2048];

	int y = 20;
	sprintf(str, "Yet Another SFCave 1.0");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += 3 * (muiSpacing + mFontBlue.h);
	sprintf(str, "Instructions :");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += muiSpacing + mFontBlue.h;
	sprintf(str, "Press and the worm goes up");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += mFontBlue.h;
	sprintf(str, "Release and the worm goes down");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += 3 * (muiSpacing + mFontBlue.h);
	sprintf(str, "Programmed by Vincent Bherer-Roy");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += 2 * (muiSpacing + mFontBlue.h);
	sprintf(str, "More infos on the Web");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += muiSpacing + mFontBlue.h;
	sprintf(str, "http://yasfcave.sourceforge.net");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += 2 * (muiSpacing + mFontBlue.h);
	sprintf(str, "This software is released under GNU GPL.");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += mFontBlue.h;
	sprintf(str, "Read COPYING file.");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}	
	
void Game::DrawAbout2()
{
	SDL_FillRect(mpSurface, &mWinRect, mBackColor);

	int textWidth;
	char str[2048];

	int y = 20;
	sprintf(str, "Yet Another SFCave 1.0");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += 3 * (muiSpacing + mFontBlue.h);
	sprintf(str, "The original Palm Pilot version will be found at");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += mFontBlue.h;
	sprintf(str, "http://hp.vector.co.jp/authors/");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	y += mFontBlue.h;
	sprintf(str, "VA003665/index-e.html");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	
	y += 3 * (muiSpacing + mFontBlue.h);
	sprintf(str, "There is also a java online version at");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);

	y += mFontBlue.h;
	sprintf(str, "http://www.liquidcode.org/worm.html");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	
	y += 3 * (muiSpacing + mFontBlue.h);
	sprintf(str, "YaSFCave uses SDL, SDL_image and SFont");
	textWidth = TextWidth2(&mFontBlue, str);
	PutString2(mpSurface, 
				&mFontBlue, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	y += mFontBlue.h;
	sprintf(str, "http://www.libsdl.org");
	textWidth = TextWidth2(&mFontRed, str);
	PutString2(mpSurface, 
				&mFontRed, 
				(mWinRect.w - textWidth) / 2, 
				y, 
				str);
	
	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}

void Game::NewGame()
{
	delete mpMap;
	mpMap = new Map(mpSurface, mViewRect.w, mViewRect.h);

	DrawBorder();

	if(mpMap)
		mpMap->Blit(mpSurface, &mViewRect);

	SDL_UpdateRects(mpSurface, 1, &mWinRect);
}

void Game::EndGame()
{
	delete mpMap;
	mpMap = NULL;
}

void Game::ResetHighScore()
{
	muiHighScore = 0;
	FILE* pFile;
	
	#ifdef DREAMCAST
		//Try to create HiScores file in SD...
		if((pFile = fopen("/sd/highscore.dat","w+b"))==NULL) {
			//Cannot create file on SD (maybe does not exist?)
			pFile = fopen("/ram/highscore.dat", "w+b");
		}
	#else
		pFile = fopen("highscore.dat", "w+b");
	#endif
	
	if(pFile)
	{
		fwrite(&muiHighScore, 4, 1, pFile);
		fclose(pFile);
	}
	
	#ifdef DREAMCAST
		//Check if file exists on SD
		if((pFile = fopen("/sd/highscore.dat","r"))==NULL) {
			DC_SaveToVMU("/ram/highscore.dat");
		}
		else fclose(pFile);
	#endif
}

