/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 * Program WebSite: http://methane.sourceforge.net/index.html              *
 * Project Email: rombust@postmaster.co.uk                                 *
 *                                                                         *
 ***************************************************************************/

//------------------------------------------------------------------------------
// The AROS main document file
//------------------------------------------------------------------------------

#include <stdlib.h>

#include "SDL.h"
#include "SDL_framerate.h"
#include "doc.h"
#include "global.h"

#ifdef METHANE_MIKMOD
#include "../mikmod/audiodrv.h"
#endif

int SCALING = 0;
int AROS_SCREEN_W = 640;
int AROS_SCREEN_H = 512;
int FS = 0;

//------------------------------------------------------------------------------
// Private Functions
//------------------------------------------------------------------------------
extern void main_code(void);

//------------------------------------------------------------------------------
// The Game Instance
//------------------------------------------------------------------------------
CMethDoc Game;
FPSmanager FpsManager;
SDL_Surface * SdlScreen = 0;


static char TheScreen[SCR_WIDTH * SCR_HEIGHT];
//static int SampleChannel = 5;	// Used by CMethDoc::PlaySample

//------------------------------------------------------------------------------
// The HighScore table filename
//------------------------------------------------------------------------------
static char HighScoreFileName[] = "Methane.HiScores";
#define HighScoreLoadBufferSize (MAX_HISCORES * 64)


//------------------------------------------------------------------------------
// The Main Function
//------------------------------------------------------------------------------

int main (int argc, char **argv)
{
        for(int i=0; i<argc; i++)
        {
		if (strcmp(argv[i],"-f")==0) FS = 1;
        	if (strcmp(argv[i],"-scale2x")==0) SCALING = 0;
        	if (strcmp(argv[i],"-scanline")==0) SCALING = 1;
		if (strcmp(argv[i],"-noscale")==0) 
		{
			SCALING = 2;
			AROS_SCREEN_H = 256;
			AROS_SCREEN_W = 320;
		}
		if (strcmp(argv[i],"-scale3x")==0) 
		{
			SCALING = 3;
			AROS_SCREEN_W = 1024;
			AROS_SCREEN_H = 768;
		}
        }
        
	// init
	if (SDL_Init(SDL_INIT_JOYSTICK|SDL_INIT_VIDEO) < 0) 
	{
		fprintf (stderr, "Can't init SDL : %s", SDL_GetError());
		return 1;
	}
	atexit (SDL_Quit);
	SDL_JoystickOpen (0);

        SDL_EnableKeyRepeat (SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);

	// run
	main_code();					
	// The main routine

	return (0) ;
}

//------------------------------------------------------------------------------
//! \brief The program main code
//------------------------------------------------------------------------------
void main_code(void)
{
	JOYSTICK *jptr1;
	JOYSTICK *jptr2;

	if(FS)
		
		SdlScreen = SDL_SetVideoMode (
				AROS_SCREEN_W, AROS_SCREEN_H, 8,  SDL_SWSURFACE|SDL_FULLSCREEN);
	else
		SdlScreen = SDL_SetVideoMode (
				AROS_SCREEN_W, AROS_SCREEN_H, 8, SDL_SWSURFACE);
				
	if (!SdlScreen)
	{
		fprintf (stderr, "Couldn't set video mode : %s\n", SDL_GetError());
		return;
	}
	SDL_ShowCursor (SDL_DISABLE);

	printf("The GNU General Public License V2 applies to this game.\n\n");
	printf("See: http://methane.sourceforge.net\n\n");
	printf("Instructions:\n\n");
	printf("Player 1 control :\n");
	printf("Joystick OR\n");
	printf("Press Ctrl to fire gas from the gun.\n");
	printf("Hold Ctrl to suck a trapped baddie into the gun.\n");
	printf("Release Ctrl to throw the trapped baddie from the gun.\n");
	printf("Direction keys to move.\n\n");
	printf("Player 2 control :\n");
	printf("Press Shift to fire gas from the gun.\n");
	printf("Hold Shift to suck a trapped baddie into the gun.\n");
	printf("Release Shift to throw the trapped baddie from the gun.\n");
	printf("D,X,C,V to move.\n\n");
	printf("Throw baddies at the wall to destroy them.\n\n");
	printf("ESC = Quit (and save high scores)\n");
	printf("TAB = Change player graphic during game\n\n");
		
	printf("Launch options:\n");
	printf("-f\t\tFULLSCREEN\n");
	printf("-scale2x\tSCALE 2X display (default)\n");
	printf("-scale3x\tSCALE 3X display\n");
	printf("-noscale\tNO SCALING display\n");
	printf("-scanline\tSCANLINE display\n");

	Game.InitSoundDriver();
	Game.InitGame ();
	Game.LoadScores();
	Game.StartGame();
	jptr1 = &Game.m_GameTarget.m_Joy1;
	jptr2 = &Game.m_GameTarget.m_Joy2;

	SDL_initFramerate (&FpsManager);
	int run = 1;
	while(run)
	{
		SDL_PumpEvents();
		Uint8 * key = SDL_GetKeyState (NULL);

		if (key[SDLK_ESCAPE]) break;
		if (key[SDLK_TAB])
		{
			Game.m_GameTarget.m_Game.TogglePuffBlow();
		}

		// event loop
		SDL_Event event;
		while(SDL_PollEvent(&event))
		{
			switch(event.type)
			{
				case SDL_KEYDOWN:
					switch(event.key.keysym.sym)
					{
					case SDLK_RIGHT:
						jptr1->right = true;
						break;
					case SDLK_LEFT:
						jptr1->left = true;
						break;
					case SDLK_UP:
						jptr1->up = true;
						break;
					case SDLK_DOWN:
						jptr1->down = true;
						break;
					case SDLK_LCTRL:
					case SDLK_RCTRL:
						jptr1->fire = true;
						break;
					case SDLK_v:
						jptr2->right = true;
						break;
					case SDLK_x:
						jptr2->left = true;
						break;
					case SDLK_d:
						jptr2->up = true;
						break;
					case SDLK_c:
						jptr2->down = true;
						break;
					case SDLK_RSHIFT:
					case SDLK_LSHIFT:
						jptr2->fire = true;
						break;
					default:
						break;
					}
					jptr1->key = event.key.keysym.sym;
					jptr2->key = event.key.keysym.sym;
					break;
				case SDL_KEYUP:
					switch(event.key.keysym.sym)
					{
					case SDLK_RIGHT:
						jptr1->right = false;
						break;
					case SDLK_LEFT:
						jptr1->left = false;
						break;
					case SDLK_UP:
						jptr1->up = false;
						break;
					case SDLK_DOWN:
						jptr1->down = false;
						break;
					case SDLK_LCTRL:
					case SDLK_RCTRL:
						jptr1->fire = false;
						break;
					case SDLK_v:
						jptr2->right = false;
						break;
					case SDLK_x:
						jptr2->left = false;
						break;
					case SDLK_d:
						jptr2->up = false;
						break;
					case SDLK_c:
						jptr2->down = false;
						break;
					case SDLK_RSHIFT:
					case SDLK_LSHIFT:
						jptr2->fire = false;
						break;
					default:
						break;
					}
					jptr1->key = NULL;
					jptr2->key = NULL;
					break;
				case SDL_JOYAXISMOTION:
					if  (event.jaxis.axis % 2)
						if (event.jaxis.value > 4096)
							jptr1->down = true;
						else if (event.jaxis.value < -4096)
							jptr1->up = true;
						else
							jptr1->up = jptr1->down = false;
					else
						if (event.jaxis.value > 4096)
							jptr1->right = true;
						else if (event.jaxis.value < -4096)
							jptr1->left = true;
						else
							jptr1->right = jptr1->left = false;
					break;
				case SDL_JOYHATMOTION:
					switch(event.jhat.value)
					{
						case SDL_HAT_CENTERED:
							jptr1->up	= false;
							jptr1->down	= false;
							jptr1->left	= false;
							jptr1->right	= false;
							break;
						case SDL_HAT_UP:
							jptr1->up	= true;
							jptr1->down	= false;
							jptr1->left	= false;
							jptr1->right	= false;
							break;
						case SDL_HAT_DOWN:
							jptr1->up	= false;
							jptr1->down	= true;
							jptr1->left	= false;
							jptr1->right	= false;
							break;
						case SDL_HAT_LEFT:
							jptr1->up	= false;
							jptr1->down	= false;
							jptr1->left	= true;
							jptr1->right	= false;
							break;
						case SDL_HAT_RIGHT:
							jptr1->up	= false;
							jptr1->down	= false;
							jptr1->left	= false;
							jptr1->right	= true;
							break;
						case SDL_HAT_RIGHTUP:
							jptr1->up	= true;
							jptr1->down	= false;
							jptr1->left	= false;
							jptr1->right	= true;
							break;
						case SDL_HAT_RIGHTDOWN:
							jptr1->up	= false;
							jptr1->down	= true;
							jptr1->left	= false;
							jptr1->right	= true;
							break;
						case SDL_HAT_LEFTUP:
							jptr1->up	= true;
							jptr1->down	= false;
							jptr1->left	= true;
							jptr1->right	= false;
							break;
						case SDL_HAT_LEFTDOWN:
							jptr1->up	= false;
							jptr1->down	= true;
							jptr1->left	= true;
							jptr1->right	= false;
							break;
					}
					break;		
				case SDL_JOYBUTTONDOWN:
					switch(event.jbutton.button)
					{
						default:
						        jptr1->key  = 13;
							jptr1->fire = true;
							break;
					}
					break;
				case SDL_JOYBUTTONUP:
					switch(event.jbutton.button)
					{
						default:
						        jptr1->key  = NULL;
							jptr1->fire = false; 			
							break;
					}
					break;
				default:
					break;
			}
		}

		// (CHEAT MODE DISABLED) --> jptr1->next_level = 0;
		Game.MainLoop (0);
		SDL_framerateDelay (&FpsManager);
	}
	Game.SaveScores ();

	SDL_FreeSurface (SdlScreen);
}

//------------------------------------------------------------------------------
//! \brief Initialise Document
//------------------------------------------------------------------------------
CMethDoc::CMethDoc()
{
#ifdef METHANE_MIKMOD
	SMB_NEW(m_pMikModDrv,CMikModDrv);
#endif
	m_GameTarget.Init(this);
}

//------------------------------------------------------------------------------
//! \brief Destroy Document
//------------------------------------------------------------------------------
CMethDoc::~CMethDoc()
{
#ifdef METHANE_MIKMOD
	if (m_pMikModDrv)
	{
		delete(m_pMikModDrv);
		m_pMikModDrv = 0;
	}
#endif
}

//------------------------------------------------------------------------------
//! \brief Initialise the game
//------------------------------------------------------------------------------
void CMethDoc::InitGame(void)
{
	m_GameTarget.InitGame (TheScreen);
	m_GameTarget.PrepareSoundDriver ();
}

//------------------------------------------------------------------------------
//! \brief Initialise the sound driver
//------------------------------------------------------------------------------
void CMethDoc::InitSoundDriver(void)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->InitDriver();
#endif
}

//------------------------------------------------------------------------------
//! \brief Remove the sound driver
//------------------------------------------------------------------------------
void CMethDoc::RemoveSoundDriver(void)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->RemoveDriver();
#endif
}

//------------------------------------------------------------------------------
//! \brief Start the game
//------------------------------------------------------------------------------
void CMethDoc::StartGame(void)
{
	m_GameTarget.StartGame();
}

//------------------------------------------------------------------------------
//! \brief Redraw the game main view
//!
//! 	\param pal_change_flag : 0 = Palette not changed
//------------------------------------------------------------------------------
void CMethDoc::RedrawMainView( int pal_change_flag )
{
	// Function not used
}

//------------------------------------------------------------------------------
//! \brief Draw the screen
//!
//! 	\param screen_ptr = UNUSED
//------------------------------------------------------------------------------
void CMethDoc::DrawScreen( void *screen_ptr )
{
	METHANE_RGB *pptr;
	int cnt;
	SDL_Color colors [PALETTE_SIZE];

	if (SDL_MUSTLOCK (SdlScreen))
	{
		SDL_LockSurface (SdlScreen);
	}

	// Set the game palette
	pptr = m_GameTarget.m_rgbPalette;
	for (cnt=0; cnt < PALETTE_SIZE; cnt++, pptr++)
	{
		colors[cnt].r = pptr->red;
		colors[cnt].g = pptr->green;
		colors[cnt].b = pptr->blue;
	}
	SDL_SetPalette (SdlScreen, SDL_LOGPAL|SDL_PHYSPAL, colors, 0, PALETTE_SIZE);

	if (SCALING == 1)
	{
		// Copy the pixels
		char * dptr = (char *) SdlScreen->pixels;
		char * sptr = TheScreen; // + 4 * AROS_SCREEN_W;
		for (int y = 0; y < SCR_HEIGHT; y++) 
		{
			for (int x = 0; x < SCR_WIDTH; x++) 
			{
				*dptr++ = *sptr;
				*dptr++ = *sptr++;
			}
			dptr += AROS_SCREEN_W;//(SdlScreen->pitch - SCR_WIDTH);
		}
	}

	if (SCALING == 0)
	{
		Uint8* srcpix = (Uint8*)TheScreen;
		Uint8* dstpix = (Uint8*)SdlScreen->pixels;
		const int srcpitch = SCR_WIDTH;
		const int dstpitch = SdlScreen->pitch;
		const int width = SCR_WIDTH;
		const int height = SCR_HEIGHT;

		Uint8 E0,E1,E2,E3,B,D,E,F,H;	

		for(int looph=0; looph < height; ++looph)
		{
			for(int loopw=0; loopw < width; ++loopw)
			{
			    	B = *(Uint8*)(srcpix + (MAX(0,looph-1)*srcpitch) + (1*loopw));
			    	D = *(Uint8*)(srcpix + (looph*srcpitch) + (1*MAX(0,loopw-1)));
			    	E = *(Uint8*)(srcpix + (looph*srcpitch) + (1*loopw));
		    		F = *(Uint8*)(srcpix + (looph*srcpitch) + (1*MIN(width-1,loopw+1)));
		    		H = *(Uint8*)(srcpix + (MIN(height-1,looph+1)*srcpitch) + (1*loopw));
			
				if (B != H && D != F)
				{
					E0 = D == B ? D : E;
					E1 = B == F ? F : E;
					E2 = D == H ? D : E;
					E3 = H == F ? F : E;
				}
				else
				{
					E0 = E;
					E1 = E;
					E2 = E;
					E3 = E;
				}			
				*(Uint8*)(dstpix + looph*2*dstpitch + loopw*2*1) = E0;
				*(Uint8*)(dstpix + looph*2*dstpitch + (loopw*2+1)) = E1;
				*(Uint8*)(dstpix + (looph*2+1)*dstpitch + loopw*2) = E2;
				*(Uint8*)(dstpix + (looph*2+1)*dstpitch + (loopw*2+1)) = E3;
			}
		}
	}
		
	if (SCALING == 2)
	{
		// Copy the pixels
		char * dptr = (char *) SdlScreen->pixels;
		char * sptr = TheScreen; // + 4 * AROS_SCREEN_W;
		for (int y = 0; y < SCR_HEIGHT; y++) 
		{
			for (int x = 0; x < SCR_WIDTH; x++) 
			{
				*dptr++ = *sptr++;
			}
			dptr += (SdlScreen->pitch - SCR_WIDTH);
		}
	}
	
	if (SCALING == 3)
	{
		Uint8* srcpix = (Uint8*)TheScreen;
		Uint8* dstpix = (Uint8*)SdlScreen->pixels;
		const int srcpitch = SCR_WIDTH;
		const int dstpitch = SdlScreen->pitch;
		const int width = SCR_WIDTH;
		const int height = SCR_HEIGHT;	

		Uint8 E0,E1,E2,E3,E4,E5,E6,E7,E8,A,B,C,D,E,F,G,H,I;

		for(int looph=0; looph < height; ++looph)
		{
			for(int loopw=0; loopw < width; ++loopw)
			{
			
				A = *(Uint8*)(srcpix + (MAX(0,looph-1)*srcpitch) + (1*MAX(0,loopw-1)));
			    	B = *(Uint8*)(srcpix + (MAX(0,looph-1)*srcpitch) + (1*loopw));
			    	C = *(Uint8*)(srcpix + (MAX(0,looph-1)*srcpitch) + (1*MIN(width-1,loopw+1)));
			    	D = *(Uint8*)(srcpix + (looph*srcpitch) + (1*MAX(0,loopw-1)));
			    	E = *(Uint8*)(srcpix + (looph*srcpitch) + (1*loopw));
			    	F = *(Uint8*)(srcpix + (looph*srcpitch) + (1*MIN(width-1,loopw+1)));
			    	G = *(Uint8*)(srcpix + (MIN(height-1,looph+1)*srcpitch) + (1*MAX(0,loopw-1)));
			    	H = *(Uint8*)(srcpix + (MIN(height-1,looph+1)*srcpitch) + (1*loopw));
			    	I = *(Uint8*)(srcpix + (MIN(height-1,looph+1)*srcpitch) + (1*MIN(width-1,loopw+1)));	

				if (B != H && D != F)
				{
					E0 = D == B ? D : E;
					E1 = (D == B && E != C) || (B == F && E != A) ? B : E;
					E2 = B == F ? F : E;
					E3 = (D == B && E != G) || (D == H && E != A) ? D : E;
					E4 = E;
					E5 = (B == F && E != I) || (H == F && E != C) ? F : E;
					E6 = D == H ? D : E;
					E7 = (D == H && E != I) || (H == F && E != G) ? H : E;
					E8 = H == F ? F : E;
				}
				else
				{
					E0 = E;
					E1 = E;
					E2 = E;
					E3 = E;
					E4 = E;
					E5 = E;
					E6 = E;
					E7 = E;
					E8 = E;
				}		    	
				*(Uint8*)(32 + dstpix + looph*3*dstpitch + loopw*3) = E0;
				*(Uint8*)(32 + dstpix + looph*3*dstpitch + (loopw*3+1)) = E1;
				*(Uint8*)(32 + dstpix + looph*3*dstpitch + (loopw*3+2)) = E2;
				*(Uint8*)(32 + dstpix + (looph*3+1)*dstpitch + loopw*3) = E3;
				*(Uint8*)(32 + dstpix + (looph*3+1)*dstpitch + (loopw*3+1)) = E4;
				*(Uint8*)(32 + dstpix + (looph*3+1)*dstpitch + (loopw*3+2)) = E5;
				*(Uint8*)(32 + dstpix + (looph*3+2)*dstpitch + loopw*3) = E6;
				*(Uint8*)(32 + dstpix + (looph*3+2)*dstpitch + (loopw*3+1)) = E7;
				*(Uint8*)(32 + dstpix + (looph*3+2)*dstpitch + (loopw*3+2)) = E8;
			}
		}
	}

	if (SDL_MUSTLOCK (SdlScreen))
	{
		SDL_UnlockSurface (SdlScreen);
	}

	// Show the new screen
	SDL_Flip (SdlScreen);
}

//------------------------------------------------------------------------------
//! \brief The Game Main Loop
//!
//! 	\param screen_ptr = UNUSED
//------------------------------------------------------------------------------
void CMethDoc::MainLoop( void *screen_ptr )
{
	m_GameTarget.MainLoop();
	DrawScreen( screen_ptr );
#ifdef METHANE_MIKMOD
	m_pMikModDrv->Update();
#endif
}

//------------------------------------------------------------------------------
//! \brief Play a sample (called from the game)
//!
//! 	\param id = SND_xxx id
//!	\param pos = Sample Position to use 0 to 255
//!	\param rate = The rate
//------------------------------------------------------------------------------
void CMethDoc::PlaySample(int id, int pos, int rate)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->PlaySample(id, pos, rate);
#endif
}

//------------------------------------------------------------------------------
//! \brief Stop the module (called from the game)
//------------------------------------------------------------------------------
void CMethDoc::StopModule(void)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->StopModule();
#endif
}

//------------------------------------------------------------------------------
//! \brief Play a module (called from the game)
//!
//! 	\param id = SMOD_xxx id
//------------------------------------------------------------------------------
void CMethDoc::PlayModule(int id)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->PlayModule(id);
#endif
}

//------------------------------------------------------------------------------
//! \brief Update the current module (ie restart the module if it has stopped) (called from the game)
//!
//! 	\param id = SMOD_xxx id (The module that should be playing)
//------------------------------------------------------------------------------
void CMethDoc::UpdateModule(int id)
{
#ifdef METHANE_MIKMOD
	m_pMikModDrv->UpdateModule(id);
#endif
}

//------------------------------------------------------------------------------
//! \brief Load the high scores
//------------------------------------------------------------------------------
void CMethDoc::LoadScores(void)
{
	FILE *fptr;
	char *mptr;
	char *tptr;
	char let;
	int cnt;

	fptr = fopen(HighScoreFileName, "r");
	if (!fptr) return;	// No scores available

	// Allocate file memory, which is cleared to zero
	mptr = (char *) calloc(1, HighScoreLoadBufferSize);
	if (!mptr)		// No memory
	{
		fclose(fptr);
		return;
	}
	fread( mptr, 1, HighScoreLoadBufferSize-2, fptr);	// Get the file

	// (Note: mptr is zero terminated)
	tptr = mptr;
	for (cnt=0; cnt<MAX_HISCORES; cnt++)	// For each highscore
	{

		if (!tptr[0]) break;

		m_GameTarget.m_Game.InsertHiScore( atoi(&tptr[4]), tptr );

		do	// Find next name
		{
			let = *(tptr++);
		}while (!( (let=='$') || (!let) ));
		if (!let) break;	// Unexpected EOF
	}

	free(mptr);

	fclose(fptr);

}

//------------------------------------------------------------------------------
//! \brief Save the high scores
//------------------------------------------------------------------------------
void CMethDoc::SaveScores(void)
{
	FILE *fptr;
	int cnt;
	HISCORES *hs;

	fptr = fopen(HighScoreFileName, "w");
	if (!fptr) return;	// Cannot write scores
	for (cnt=0, hs=m_GameTarget.m_Game.m_HiScores; cnt<MAX_HISCORES; cnt++, hs++)
	{
		fprintf(fptr, "%c%c%c%c%d$", hs->name[0], hs->name[1], hs->name[2], hs->name[3], hs->score);
	}
	fclose(fptr);
}

