/*
    This file is part of Dream Photo Manager.

    Copyright (C) 2009 Lawrence Sebald

    Dream Photo Manager is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 
    as published by the Free Software Foundation.

    Dream Photo Manager is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Dream Photo Manager; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string.h>

#include "pvrutils.h"
#include "font.h"

pvr_ptr_t create_texture(kos_img_t *kmg, int *w, int *h)    {
    pvr_ptr_t rv;
    uint16 *dst, *src;
    int i, j;

    /* Make our dimensions a power of two. */
    *w = 8;
    while(*w < kmg->w)  {
        *w <<= 1;
    }

    *h = 8;
    while(*h < kmg->h)  {
        *h <<= 1;
    }

    /* Allocate memory and set up the pointers. */
    rv = pvr_mem_malloc(*w * *h * 2);
    src = (uint16 *)kmg->data;
    dst = (uint16 *)rv;

    /* Set up the texture with proper dimensions. */
    for(i = 0; i < kmg->h; ++i) {
        for(j = 0; j < kmg->w; ++j) {
            dst[i * *w + j] = src[i * kmg->w + j];
        }
    }

    return rv;
}

void draw_fullscreen_sprite(float umax, float vmax) {
    pvr_sprite_txr_t spr;

    spr.flags = PVR_CMD_VERTEX_EOL;
    spr.ax = 0.0f;
    spr.ay = 480.0f;
    spr.az = 1.0f;
    spr.bx = 0.0f;
    spr.by = 0.0f;
    spr.bz = 1.0f;
    spr.cx = 640.0f;
    spr.cy = 0.0f;
    spr.cz = 1.0f;
    spr.dx = 640.0f;
    spr.dy = 480.0f;
    spr.dummy = 0;
    spr.auv = PVR_PACK_16BIT_UV(0.0f, vmax);
    spr.buv = PVR_PACK_16BIT_UV(0.0f, 0.0f);
    spr.cuv = PVR_PACK_16BIT_UV(umax, 0.0f);
    pvr_prim(&spr, sizeof(pvr_sprite_txr_t));
}

void draw_colored_quad(uint32 list, float x, float y, float z, float w, float h,
                       uint32 color)    {
    pvr_poly_cxt_t cxt;
    pvr_poly_hdr_t hdr;
    pvr_vertex_t vert;

    pvr_poly_cxt_col(&cxt, list);
    pvr_poly_compile(&hdr, &cxt);

    pvr_prim(&hdr, sizeof(pvr_poly_hdr_t));

    vert.flags = PVR_CMD_VERTEX;
    vert.oargb = 0;
    vert.argb = color;
    vert.u = 0.0f;
    vert.v = 0.0f;
    vert.z = z;

    vert.x = x;
    vert.y = y + h;
    pvr_prim(&vert, sizeof(pvr_vertex_t));

    vert.y = y;
    pvr_prim(&vert, sizeof(pvr_vertex_t));

    vert.x = x + w;
    vert.y = y + h;
    pvr_prim(&vert, sizeof(pvr_vertex_t));

    vert.flags = PVR_CMD_VERTEX_EOL;
    vert.y = y;
    pvr_prim(&vert, sizeof(pvr_vertex_t));
}

static void string_stats(const char *str, int *longest, int *count) {
    int i, cur, len = strlen(str);

    *longest = 0;
    *count = 0;
    cur = 0;

    for(i = 0; i < len; ++i)    {
        if(str[i] == '\n')  {
            if(*longest < cur)
                *longest = cur;
            cur = 0;
            ++(*count);
        }
        else    {
            ++cur;
        }
    }

    /* Handle the last line. */
    if(*longest < cur)
        *longest = cur;

    ++(*count);
}

void draw_dialog_centered(const char *str, float xmin, float xmax, float ymin,
                          float ymax, float z, uint32 color, pvr_ptr_t f)   {
    int longest, count;
    float w, h;
    float xs, ys;

    string_stats(str, &longest, &count);

    w = longest * FONT_CHAR_WIDTH;
    h = count * FONT_LINE_HEIGHT;

    xs = ((xmin + xmax) / 2.0f) - (w / 2.0f);
    ys = ((ymin + ymax) / 2.0f) - (h / 2.0f);

    draw_colored_quad(PVR_LIST_TR_POLY, xs, ys, z, w, h, color);

    font_draw_str_centered(str, f, (int)xmin, (int)xmax, (int)ys, (int)z + 1);
}
