/*
    This file is part of Dream Photo Manager.

    Copyright (C) 2009 Lawrence Sebald

    Dream Photo Manager is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 
    as published by the Free Software Foundation.

    Dream Photo Manager is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Dream Photo Manager; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <stdlib.h>

#include <dc/maple.h>
#include <dc/maple/dreameye.h>

#include <jpeg/jpeg.h>

#include "imgutils.h"
#include "pvrutils.h"

static int image_count = 0;
static uint8 **images = NULL;
static int *sizes = NULL;

static pvr_ptr_t txr = NULL;

int grab_all_images(maple_device_t *dev, void (*cb)(int, int))  {
    dreameye_state_t *state;
    int err, i;

    /* Grab the number of pictures on the Dreameye. */
    err = dreameye_get_image_count(dev, 1);
    if(err != MAPLE_EOK)    {
        printf("Error in fetching the image count.\n");
        return -1;
    }

    state = (dreameye_state_t *)maple_dev_status(dev);
    image_count = state->image_count;

    /* Allocate memory for the pointers and sizes of images. */
    images = (uint8 **)malloc(sizeof(uint8 *) * image_count);
    sizes = (int *)malloc(sizeof(int) * image_count);

    /* Grab all the images from the Dreameye. */
    for(i = 0; i < image_count; ++i)    {
        err = dreameye_get_image(dev, (uint8)i + 2, images + i, sizes + i);
        if(err != MAPLE_EOK)    {
            printf("Error in retrieving image %d\n", i);
        }

        cb(i, image_count);
    }

    printf("Read %d images\n", image_count);

    return image_count;
}

void release_all_images(void)   {
    int i;

    /* Free all memory. */
    if(images)  {
        for(i = 0; i < image_count; ++i)    {
            free(images[i]);
        }

        free(images);
    }

    if(sizes)   {
        free(sizes);
    }

    /* Reset the data so we know there's no stored images. */
    image_count = 0;
    images = NULL;
    sizes = NULL;
}

int load_texture(int num, float *u, float *v, pvr_poly_ic_hdr_t *dst)   {
    FILE *fp;
    kos_img_t img;
    pvr_sprite_cxt_t cxt;
    int w, h;

    if(num > image_count)
        return -1;

    /* Save the specified image in the /ram filesystem. */
    fp = fopen("/ram/image.jpg", "wb");
    fwrite(images[num], sizes[num], 1, fp);
    fclose(fp);

    if(jpeg_to_img("/ram/image.jpg", 1, &img))  {
        printf("Couldn't make a KOS image from image %d\n", num);
        return -1;
    }

    /* Free any old texture that we have sitting around. */
    if(txr != NULL) {
        pvr_mem_free(txr);
        txr = NULL;
    }

    txr = create_texture(&img, &w, &h);

    pvr_sprite_cxt_txr(&cxt, PVR_LIST_OP_POLY,
                       PVR_TXRFMT_RGB565 | PVR_TXRFMT_NONTWIDDLED, w, h, txr,
                       PVR_FILTER_NONE);
    pvr_sprite_compile(dst, &cxt);

    *u = (float)img.w / w;
    *v = (float)img.h / h;

    /* Clean up. */
    kos_img_free(&img, 0);
    unlink("/ram/image.jpg");

    return 0;
}

int erase_image(int num, maple_device_t *dev)   {
    int i;
    uint8 **tmp1;
    int *tmp2;

    dreameye_erase_image(dev, (uint8)num + 2, 1);
    --image_count;

    tmp1 = (uint8 **)malloc(sizeof(uint8 *) * image_count);
    tmp2 = (int *)malloc(sizeof(int) * image_count);

    for(i = 0; i < num; ++i)    {
        tmp1[i] = images[i];
        tmp2[i] = sizes[i];
    }

    free(images[num]);

    for(; i < image_count; ++i) {
        tmp1[i] = images[i + 1];
        tmp2[i] = sizes[i + 1];
    }

    free(images);
    free(sizes);

    images = tmp1;
    sizes = tmp2;

    return 0;
}

int get_image_count(void)   {
    return image_count;
}
