/*
    This file is part of Dream Photo Manager.

    Copyright (C) 2009 Lawrence Sebald

    Dream Photo Manager is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 
    as published by the Free Software Foundation.

    Dream Photo Manager is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Dream Photo Manager; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <string.h>

#include <arch/arch.h>
#include <arch/timer.h>
#include <dc/maple.h>
#include <dc/maple/controller.h>
#include <dc/maple/dreameye.h>
#include <dc/pvr.h>

#include "pvrutils.h"
#include "imgutils.h"
#include "font.h"

#define UNUSED __attribute__((unused))

static int done = 0;
static int img_count = 0;
static int img_shown = 0;
static float u = 0.0f;
static float v = 0.0f;
static pvr_poly_ic_hdr_t hdr;
static pvr_ptr_t font;
static int show_confirm = 0;

static void cont_callback(uint8 addr UNUSED, uint32 buttons UNUSED) {
    done = 1;
}

static void handle_input(maple_device_t *cont, maple_device_t *dreameye)    {
    static uint64 last_update = 0;
    cont_state_t *state;
    int last_img = img_shown;

    /* Check the device status. */
    state = (cont_state_t *)maple_dev_status(cont);
    
    if(state)   {
        if(state->buttons & CONT_START) {
            done = 1;
        }

        if(state->buttons & CONT_Y &&
           last_update + 200 < timer_ms_gettime64() && !show_confirm)   {
            show_confirm = 1;
            last_update = timer_ms_gettime64();
        }

        if(state->buttons & CONT_A &&
           last_update + 200 < timer_ms_gettime64() && show_confirm)    {
            printf("Erasing image %d\n", img_shown);
            erase_image(img_shown, dreameye);
            img_count = get_image_count();

            /* If we've deleted the last image, rewind one (assuming we have one
               to rewind). */
            if(img_shown == img_count && img_count) {
                --img_shown;
            }

            load_texture(img_shown, &u, &v, &hdr);
            last_update = timer_ms_gettime64();
            show_confirm = 0;
        }

        if(state->buttons & CONT_B &&
           last_update + 200 < timer_ms_gettime64() && show_confirm)    {
            last_update = timer_ms_gettime64();
            show_confirm = 0;
        }

        if(state->buttons & CONT_DPAD_LEFT &&
           last_update + 200 < timer_ms_gettime64() && !show_confirm)   {
            --img_shown;
            
            if(img_shown == -1) {
                img_shown = img_count - 1;
            }
            
            last_update = timer_ms_gettime64();
        }
        
        if(state->buttons & CONT_DPAD_RIGHT &&
           last_update + 200 < timer_ms_gettime64() && !show_confirm)   {
            ++img_shown;
            
            if(img_shown == img_count)  {
                img_shown = 0;
            }
            
            last_update = timer_ms_gettime64();
        }
    }

    /* If the user has changed images, update the texture. */
    if(img_shown != last_img)   {
        load_texture(img_shown, &u, &v, &hdr);
    }
}

static void draw_confirmation(void) {
    draw_dialog_centered("Are you sure you want to\n"
                         "delete this image from\n"
                         "the Dreameye?\n\n"
                         "THIS CANNOT BE UNDONE\n\n"
                         "Press A to delete\n"
                         "or B to cancel.", 0.0f, 640.0f, 0.0f, 480.0f, 2.0f,
                         0x800000FF, font);
}

static void __attribute__((__noreturn__)) exit_with_error(const char *err)  {
    maple_device_t *dev;
    cont_state_t *state;
    char str[strlen(err) + 30];

    strcpy(str, err);
    strcat(str, "\n\nPress any button to exit.");

    printf("%s\n", err);

    for(;;) {
        dev = maple_enum_type(0, MAPLE_FUNC_CONTROLLER);

        if(dev) {
            state = (cont_state_t *)maple_dev_status(dev);

            if(state)   {
                if(state->buttons)
                    arch_exit();
            }
        }

        pvr_wait_ready();
        pvr_scene_begin();
        pvr_list_begin(PVR_LIST_TR_POLY);
        draw_dialog_centered(str, 0.0f, 640.0f, 0.0f, 480.0f, 2.0f, 0x800000FF,
                             font);
        pvr_list_finish();
        pvr_scene_finish();
    }
}

static void progress_cb(int num, int count) {
    char str[256];

    sprintf(str, "Please wait while images are\n"
                 "downloaded from the Dreameye.\n\n"
                 "%d of %d completed", num, count);

    pvr_wait_ready();
    pvr_scene_begin();
    pvr_list_begin(PVR_LIST_TR_POLY);
    draw_dialog_centered(str, 0.0f, 640.0f, 0.0f, 480.0f, 2.0f, 0x800000FF,
                         font);
    pvr_list_finish();
    pvr_scene_finish();
}

static void paint_scene(void)   {
    pvr_wait_ready();
    pvr_scene_begin();

    if(img_count)   {
        pvr_list_begin(PVR_LIST_OP_POLY);
        pvr_prim(&hdr, sizeof(pvr_poly_ic_hdr_t));
        draw_fullscreen_sprite(u, v);
        pvr_list_finish();

        if(show_confirm)    {
            pvr_list_begin(PVR_LIST_TR_POLY);
            draw_confirmation();
            pvr_list_finish();
        }
    }
    else    {
        pvr_list_begin(PVR_LIST_TR_POLY);
        pvr_list_finish();
    }

    pvr_scene_finish();
}

int main(int argc, char *argv[])    {
    maple_device_t *cont, *dreameye;

    printf("Dream Photo Manager starting up...\n");

    pvr_init_defaults();

    cont_btn_callback(0, CONT_A | CONT_B | CONT_X | CONT_Y | CONT_START,
                      &cont_callback);

    /* Load the font in case there's any problems. */
    font = font_load_txr();

    /* Grab the first Dreameye device detected. */
    dreameye = maple_enum_type(0, MAPLE_FUNC_CAMERA);
    if(!dreameye)   {
        exit_with_error("No Dreameye devices found.");
    }

    /* Grab the first controller. Exclude the "controller" that the Dreameye has
       built in by looking for one with a D-Pad and an A and Y button. */
    cont = maple_enum_type_ex(0, MAPLE_FUNC_CONTROLLER, CONT_A | CONT_B |
                              CONT_Y | CONT_START | CONT_DPAD_LEFT |
                              CONT_DPAD_RIGHT);
    if(!cont)   {
        exit_with_error("No Controllers detected.");
    }

    /* Grab all the images off the Dreameye. */
    if((img_count = grab_all_images(dreameye, &progress_cb)) == -1) {
        exit_with_error("Could not grab images from\n"
                        "the Dreameye.");
    }

    if(!img_count)  {
        exit_with_error("No images found on the\n"
                        "connected Dreameye.");
    }

    /* Load up the first image as a texture. */
    if(load_texture(0, &u, &v, &hdr))   {
        exit_with_error("Could not load image 0 from\n"
                        "the Dreameye.");
    }

    while(!done)    {
        handle_input(cont, dreameye);
        paint_scene();
    }

    /* Clean up... */
    release_all_images();
    pvr_mem_free(font);

    return 0;
}
