/*
    This file is part of Dream Photo Manager.

    Copyright (C) 2007, 2009 Lawrence Sebald

    Dream Photo Manager is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 
    as published by the Free Software Foundation.

    Dream Photo Manager is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Dream Photo Manager; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* This file was imported from CrabEmu. */

#include <string.h>

#include <dc/biosfont.h>

#include "font.h"

#define UADV 12.0f / 128.0f
#define VADV 24.0f / 256.0f

pvr_ptr_t font_load_txr()   {
    pvr_ptr_t rv = pvr_mem_malloc(256 * 128 * 2);
    uint16 *ptr = (uint16 *)rv;

    bfont_set_encoding(BFONT_CODE_ISO8859_1);

    /* Draw the font into the texture */
    bfont_draw_str(ptr, 128, 1, " !\"#$%&'()");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "*+,-./0123");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "456789:;<=");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, ">?@ABCDEFG");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "HIJKLMNOPQ");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "RSTUVWXYZ[");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "\\]^_`abcde");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "fghijklmno");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "pqrstuvwxy");
    ptr += 128 * FONT_CHAR_HEIGHT;
    bfont_draw_str(ptr, 128, 1, "z{|}~");

    return rv;
}

void font_draw_char(char c, pvr_ptr_t font, int x, int y, int z)    {
    pvr_sprite_cxt_t cxt;
    pvr_poly_ic_hdr_t hdr;
    pvr_sprite_txr_t spr;
    int row, col;

    pvr_sprite_cxt_txr(&cxt, PVR_LIST_TR_POLY,
                       PVR_TXRFMT_ARGB1555 | PVR_TXRFMT_NONTWIDDLED, 128, 256,
                       font, PVR_FILTER_NONE);
    pvr_sprite_compile(&hdr, &cxt);

    pvr_prim(&hdr, sizeof(pvr_poly_ic_hdr_t));

    c -= 32;
    row = c / 10;
    col = c % 10;

    spr.flags = PVR_CMD_VERTEX_EOL;
    spr.ax = x;
    spr.ay = y + FONT_CHAR_HEIGHT;
    spr.az = z;
    spr.bx = x;
    spr.by = y;
    spr.bz = z;
    spr.cx = x + FONT_CHAR_WIDTH;
    spr.cy = y;
    spr.cz = z;
    spr.dx = x + FONT_CHAR_WIDTH;
    spr.dy = y + FONT_CHAR_HEIGHT;
    spr.dummy = 0;
    spr.auv = PVR_PACK_16BIT_UV(UADV * col, VADV * (row + 1));
    spr.buv = PVR_PACK_16BIT_UV(UADV * col, VADV * row);
    spr.cuv = PVR_PACK_16BIT_UV(UADV * (col + 1), VADV * row);

    pvr_prim(&spr, sizeof(pvr_sprite_txr_t));
}

void font_draw_str(const char *str, pvr_ptr_t font, int x, int y, int z)    {
    int len = strlen(str);
    int i, startx = x;

    for(i = 0; i < len; ++i)    {
        if(str[i] != '\n')  {
            font_draw_char(str[i], font, x, y, z);
            x += FONT_CHAR_WIDTH;
        }
        else    {
            x = startx;
            y += FONT_LINE_HEIGHT;
        }
    }
}

static void font_draw_one_line(const char *str, pvr_ptr_t font, int x, int y,
                               int z, int len)  {
    int i;

    for(i = 0; i < len; ++i)    {
        font_draw_char(str[i], font, x, y, z);
        x += FONT_CHAR_WIDTH;
    }
}

void font_draw_str_centered(const char *str, pvr_ptr_t font, int xmin, int xmax,
                            int y, int z)   {
    int i, count = 0;
    int center = (xmax + xmin) >> 1;
    int x = center;
    int len = strlen(str);
    const char *str2 = str;

    for(i = 0; i < len; ++i)    {
        if(str[i] != '\n' && i != len - 1)  {
            ++count;
            x -= FONT_CHAR_WIDTH / 2;
        }
        else if(str[i] != '\n' && i == len - 1) {
            x -= FONT_CHAR_WIDTH / 2;
            ++count;
            font_draw_one_line(str2, font, x, y, z, count);
        }
        else    {
            font_draw_one_line(str2, font, x, y, z, count);
            x = center;
            y += FONT_LINE_HEIGHT;
            count = 0;
            str2 = str + i + 1;
        }
    }
}
