/*
    Copyright (C) 1999, 2000, 2001, 2002, 2003  Charles MacDonald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "shared.h"

t_bitmap bitmap;
t_input input;
t_snd snd;
static int sound_tbl[262];
static int use_cpu68k_type;

#ifdef _arch_dreamcast
#include "menu.h"
#endif

#define SND_SIZE (snd.buffer_size * sizeof(int16))

static int memory_allocated = 0;

int audio_init(int rate)
{
    int i;

    /* 68000 and YM2612 clock */
    float vclk = 53693175.0 / 7;

    /* Z80 and SN76489 clock */
    float zclk = 3579545.0;

    /* Clear the sound data context */
    // Do not wipe out the buffer pointers
    snd.sample_rate = 0;
    snd.enabled = 0;
    snd.buffer_size = 0;

    /* Make sure the requested sample rate is valid */
    if(!rate || ((rate < 8000) | (rate > 44100)))
    {
        return (0);
    }

    /* Calculate the sound buffer size */
#ifdef MODE_PAL
    snd.buffer_size = (rate / 50);
#else
    snd.buffer_size = (rate / 60);
#endif
    snd.sample_rate = rate;

    /* Allocate sound buffers */
    if(!memory_allocated)
    {
#ifdef _arch_dreamcast
	snd.buffer[0] = memalign(32, SND_SIZE);
	snd.buffer[1] = memalign(32, SND_SIZE);
	snd.fm.buffer[0] = memalign(32, SND_SIZE);
	snd.fm.buffer[1] = memalign(32, SND_SIZE);
	snd.psg.buffer = memalign(32, SND_SIZE);
	snd.dac.buffer = memalign(32, SND_SIZE);
#else // !_arch_dreamcast
	snd.buffer[0] = malloc(SND_SIZE);
	snd.buffer[1] = malloc(SND_SIZE);
	snd.fm.buffer[0] = malloc(SND_SIZE);
	snd.fm.buffer[1] = malloc(SND_SIZE);
	snd.psg.buffer = malloc(SND_SIZE);
	snd.dac.buffer = malloc(SND_SIZE);
#endif // _arch_dreamcast
    }

    /* Make sure we could allocate everything */
    if(!snd.buffer[0] || !snd.buffer[1] || !snd.fm.buffer[0] || !snd.fm.buffer[1] || !snd.psg.buffer || !snd.dac.buffer)
    {
        return (0);
    }
    memory_allocated = 1;

    /* Initialize sound chip emulation */
    SN76496_sh_start(zclk, 100, rate);
    YM2612Init(1, vclk, rate, NULL, NULL);

    /* Set audio enable flag */
    snd.enabled = 1;

    /* Make sound table */
    for (i = 0; i < 262; i++)
    {
        float p = snd.buffer_size * i;
        p = p / 262;
    	sound_tbl[i] = p;
    }

    return (1);
}

void audio_shutdown(void)
{
}

void system_init(int cpu68k_type, int xvdp)
{
  use_cpu68k_type = cpu68k_type;

  gen_init(use_cpu68k_type);
  vdp_init(use_cpu68k_type, xvdp);
  render_init();
}

void system_reset(void)
{
    gen_reset();
    vdp_reset();
    render_reset();
    io_rescan();

    if(snd.enabled)
    {
        YM2612ResetChip(0);
        memset(snd.buffer[0], 0, SND_SIZE);
        memset(snd.buffer[1], 0, SND_SIZE);
        memset(snd.fm.buffer[0], 0, SND_SIZE);
        memset(snd.fm.buffer[1], 0, SND_SIZE);
        memset(snd.psg.buffer, 0, SND_SIZE);
        memset(snd.dac.buffer, 0, SND_SIZE);
    }
}

void system_shutdown(void)
{
    sram_save();
    gen_shutdown();
    vdp_shutdown();
    render_shutdown();
}

int system_frame(int xvdp)
{
    int line;

    if(gen_running == 0)
        return 0;

    /* Clear V-Blank flag */
    status &= ~0x0008;

    /* Toggle even/odd flag (IM2 only) */
    if(im2_flag)
        status ^= 0x0010;

    /* Point to start of sound buffer */
    snd.dac.lastStage = snd.dac.curStage = 0;

    if (!xvdp)
    {
      /* Parse sprites for line 0 (done on line 261) */
      parse_satb(0x80);
    }
    else
    {
      /* Set the initial colour palette */
      //	xrender_set_palette();
      //	is_color_dirty = 0;
    }

    for(line = 0; line < 262; line += 1)
    {
        /* Used by HV counter */
        v_counter = line;

        #ifdef USE_DEVICE_6BUTTON
        /* I/O Timers */
        io_scanline();
        #endif
        
        /* Run Z80 emulation (if enabled) */
        if(zreset == 1 && zbusreq == 0)
        {
#ifdef CPUZ80_USE_CZ80
			#ifdef _arch_dreamcast
				Cz80_Exec(&CZ80, dc_options.z80_clock);
			#elif defined MODE_PAL
				Cz80_Exec(&CZ80, 274);
			#else
				Cz80_Exec(&CZ80, 228);
			#endif
#else
			#ifdef _arch_dreamcast
				z80_execute(dc_options.z80_clock);
			#elif defined MODE_PAL
				z80_execute(274);
			#else
				z80_execute(228);
			#endif
#endif
            if(!gen_running) break;
        }

        /* Run 68000 emulation */
		#if defined MODE_PAL
			M68K_Exec(use_cpu68k_type, 585);
		#else
			M68K_Exec(use_cpu68k_type, 487);
		#endif
        if(!gen_running) break;

        /* If a Z80 interrupt is still pending after a scanline, cancel it */
        if(zirq == 1)
        {
            zirq = 0;
        }

        if (xvdp)
        {
          // Detect mid-frame palette changes
          //	if((is_color_dirty) && (line > 0) && (line < 224))
          //		xrender_mid_palette(line);
          //	is_color_dirty = 0;
        }
        else
        {
            if(line <  frame_end   ) render_line(line);
            if(line <  frame_end-1 ) parse_satb(0x81 + line);
        }

        /* Do H interrupt processing */
        if(line <= frame_end)
        {
            counter -= 1;
            if(counter == -1)
            {
                counter = reg[10];
                hint_pending = 1;
                if(reg[0] & 0x10)
                {
                    M68K_SetIRQ(use_cpu68k_type, 4);
                }
            }
        }
        else
        {
            counter = reg[10];
        }

        /* Do end of frame processing */
        if(line == frame_end)
        {
            status |= 0x0088;
            vint_pending = 1;

            /* Give enough time to read the interrupt pending flag before
               the interrupt actually occurs. */
            M68K_Exec(use_cpu68k_type, 16);
            if(!gen_running) break;

            if(reg[1] & 0x20)
            {
                M68K_SetIRQ(use_cpu68k_type, 6);
            }

            if(zreset == 1 && zbusreq == 0)
            {
#ifdef CPUZ80_USE_CZ80
                Cz80_Set_IRQ(&CZ80, 0);
#else
                z80_set_irq_line(0, ASSERT_LINE);
#endif                
                zirq = 1;
            }
        }
 
        fm_update_timers();

        snd.dac.curStage = sound_tbl[line];
    }

    if(snd.enabled)
    {
        audio_update();
    }

    return gen_running;
}


#define PSG_VOLUME 3
#define FM_VOLUME 2

void audio_update(void)
{
  int i;
  int16 tmp,acc;
  int16 *tempBuffer[2];

  tempBuffer[0] = snd.fm.buffer[0];
  tempBuffer[1] = snd.fm.buffer[1];

  // Do whole frame of FM and PSG
  YM2612UpdateSynth(0, (int16 **)tempBuffer, snd.buffer_size);
  SN76496Update(0, snd.psg.buffer, snd.buffer_size);

  // Finish the DAC to the end of the frame
  YM2612UpdateDacAndTimers(0, snd.dac.buffer + snd.dac.lastStage, snd.buffer_size - snd.dac.lastStage);

  for(i=0; i<snd.buffer_size; i++)
  {
    tmp = (snd.psg.buffer[i] / PSG_VOLUME) + snd.dac.buffer[i];

    acc = tmp + (snd.fm.buffer[0][i] / FM_VOLUME);
    snd.buffer[0][i] = acc;

    tmp += snd.fm.buffer[1][i] / FM_VOLUME;
    snd.buffer[1][i] = tmp;
  }
}
